#include "stdafx.h"
#include "StlMesh.h"

#include "../BaseMesh.h"

#include <C2/util/stream_util.h>

#include <set>
#include <fstream>


namespace lib_geo
{


StlMesh::StlMesh(void)
{
	ClearMessage();
}

void StlMesh::Clear(void)
{
	ClearMessage();
	m_Faces.clear();
}

void StlMesh::ClearMessage(void)
{
	for (int i = 0; i < MSG_LENGTH; ++i)
	{
		m_Message[i] = 0;
	}
}

bool StlMesh::LoadBinary(std::istream& in)
{
	Clear();

	size_t fileSize = util::GetStreamSize(in);

	in.read(m_Message, MSG_LENGTH);

	unsigned int NumFaces;
	in.read((char*)&NumFaces, sizeof(unsigned int));

	const size_t expectedBinaryFileSize = NumFaces * 50 + 84;

	if (expectedBinaryFileSize != fileSize)
		return false;

	m_Faces.resize(NumFaces);
	for (size_t i = 0; i < NumFaces; ++i)
	{
		char padding[2] = {0, 0};

		StlFace& f = m_Faces[i];
		in.read((char*)f.m_Normal.v(), sizeof(float) * 3);
		in.read((char*)f.m_Verts[0].v(), sizeof(float) * 3);
		in.read((char*)f.m_Verts[1].v(), sizeof(float) * 3);
		in.read((char*)f.m_Verts[2].v(), sizeof(float) * 3);
		in.read(padding, 2);
	}

	return true;
}

bool StlMesh::LoadAscii(std::istream& in)
{
	if (!IsExistValidAsciiHeader(in))
		return false;

	Clear();

	in.seekg(0, std::ios::beg);

	StlFace f;
	int vcount = 0;

	std::string ws;
	std::getline(in, ws);
	while (!in.eof())
	{
		std::getline(in, ws);
		if (ws.empty())
			break;

		std::istringstream is(ws);
		std::string s;
		is >> s;

		if (s == "facet")
		{
			lm::vec3f o(0.0f, 0.0f, 0.0f);
			f.m_Normal = o;
			f.m_Verts[0] = o;
			f.m_Verts[1] = o;
			f.m_Verts[2] = o;
			vcount = 0;

			lm::vec3f& n = f.m_Normal;
			is >> n.x >> n.y >> n.z;
		}
		else if (s == "endloop")
		{
			m_Faces.push_back(f);
		}
		else if (s == "vertex")
		{
			lm::vec3f& v = f.m_Verts[vcount++];
			is >> v.x >> v.y >> v.z;
		}
		else if (s == "endfacet")
		{
		}
		else if (s == "outer")
		{
		}
		else if (s == "endsolid")
		{
		}
		else
		{
			return false;
		}
	}

	return true;
}

bool StlMesh::IsExistValidAsciiHeader(std::istream& in)
{
	in.read(m_Message, MSG_LENGTH);

	m_Message[5] = '\0';
	std::string top = m_Message;
	ClearMessage();

	return top == "solid";
}

bool StlMesh::Load(std::istream& ist)
{
	if (LoadBinary(ist))
		return true;

	ist.seekg(0, std::ios::beg);

	if (LoadAscii(ist))
		return true;

	return false;
}

bool StlMesh::Load(const std::string& filename)
{
	std::ifstream ifs(filename.c_str(), std::ios::binary);
	if (!ifs.is_open())
		return false;

	return Load(ifs);
}

bool StlMesh::SaveBinary(std::ostream& ost) const
{
	unsigned int NumFaces = (unsigned int)m_Faces.size();
	ost.write(m_Message, MSG_LENGTH);
	ost.write((char*)&NumFaces, sizeof(unsigned int));

	char padding[2] = {0, 0};

	for (const StlFace& f : m_Faces)
	{
		ost.write((char*)f.m_Normal.v(), sizeof(float) * 3);
		ost.write((char*)f.m_Verts[0].v(), sizeof(float) * 3);
		ost.write((char*)f.m_Verts[1].v(), sizeof(float) * 3);
		ost.write((char*)f.m_Verts[2].v(), sizeof(float) * 3);
		ost.write(padding, 2);
	}

	return true;
}

bool StlMesh::SaveBinary(const std::string& filename) const
{
	std::ofstream ofs(filename.c_str(), std::ios::binary);
	if (!ofs.is_open())
		return false;

	return SaveBinary(ofs);
}

bool StlMesh::SaveAscii(std::ostream& ost) const
{
	ost << "solid " << m_Message << std::endl;

	for (const StlFace& f : m_Faces)
	{
		WriteAscciiVert(ost, "facet normal", f.m_Normal);
		ost << "outer loop" << std::endl;
		WriteAscciiVert(ost, "vertex", f.m_Verts[0]);
		WriteAscciiVert(ost, "vertex", f.m_Verts[1]);
		WriteAscciiVert(ost, "vertex", f.m_Verts[2]);
		ost << "endloop" << std::endl;
		ost << "endfacet" << std::endl;
	}

	ost << "endsolid" << std::endl;

	return true;
}

bool StlMesh::SaveAscii(const std::string& filename) const
{
	std::ofstream ofs(filename.c_str());
	if (!ofs.is_open())
		return false;

	return SaveAscii(ofs);
}

void StlMesh::WriteAscciiVert(std::ostream& out, const char* head, const lm::vec3f& v) const
{
	out << head
		<< " " << v.x
		<< " " << v.y
		<< " " << v.z
		<< std::endl;
}

bool StlMesh::ConvertToBaseMesh(BaseMesh& bm) const
{
	bm.Clear();

	bm.m_Verts.resize(m_Faces.size() * 3);
	bm.m_Normals.resize(m_Faces.size());
	bm.m_Faces.resize(m_Faces.size());

	for (size_t i = 0; i < m_Faces.size(); ++i)
	{
		const StlFace& f = m_Faces[i];

		bm.m_Verts[i * 3 + 0] = f.m_Verts[0];
		bm.m_Verts[i * 3 + 1] = f.m_Verts[1];
		bm.m_Verts[i * 3 + 2] = f.m_Verts[2];

		bm.m_Normals[i] = f.m_Normal;

		BaseFace& bf = bm.m_Faces[i];
		bf.m_VertIds.resize(3);
		bf.m_NormIds.resize(3);

		bf.m_VertIds[0] = (int)(i * 3 + 0);
		bf.m_VertIds[1] = (int)(i * 3 + 1);
		bf.m_VertIds[2] = (int)(i * 3 + 2);

		bf.m_NormIds[0] = (int)i;
		bf.m_NormIds[1] = (int)i;
		bf.m_NormIds[2] = (int)i;
	}

	return true;
}

//! TCY0̖@܂ޏꍇtrueԂ
bool StlMesh::ExistZeroLengthNormal(void) const
{
	for (const StlFace& f : m_Faces)
	{
		if (f.m_Normal.is_zero())
			return true;
	}

	return false;
}


}
