/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sed.cmd;

import java.io.IOException;

import net.morilib.sed.SedCommand;
import net.morilib.sed.SedLineBuffer;
import net.morilib.sed.SedSpace;


public class SedCommands {

	public static final SedCommand
	D = new SedSimpleCommand('d', "Delete") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.deleteLine();
		}

	};

	public static final SedCommand
	P = new SedSimpleCommand('p', "Print") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.print();
		}

	};

	public static final SedCommand
	L = new SedSimpleCommand('l', "Print visible") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.printVisible();
		}

	};

	public static final SedCommand
	N = new SedSimpleCommand(
			'n', "read Next line to the pattern space") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			String t;

			if((t = rd.readLine()) != null) {
				s.printImmediate();
				s.setPatternSpace(t);
			} else {
				s.deleteLine();
			}
		}

	};

	public static final SedCommand
	EQ = new SedSimpleCommand('=', "print current input line number") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.print(rd.getLineNumber());
		}

	};

	public static final SedCommand
	DL = new SedSimpleCommand(
			'D', "delete the first line in the pattern space") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.deleteFirstLine();
		}

	};

	public static final SedCommand
	NL = new SedSimpleCommand(
			'N', "append Next line to the pattern space") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			String t;

			if((t = rd.readLine()) != null) {
				s.appendPatternSpace(t);
			}
		}

	};

	public static final SedCommand
	PL = new SedSimpleCommand(
			'P', "Print the pattern space up to the first newline") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.printFirstLine();
		}

	};

	public static final SedCommand
	H = new SedSimpleCommand(
			'h', "replace the Hold space with the pattern space") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.replaceHoldSpace();
		}

	};

	public static final SedCommand
	HL = new SedSimpleCommand(
			'H', "append the Hold space to the pattern space") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.appendHoldSpace();
		}

	};

	public static final SedCommand
	G = new SedSimpleCommand(
			'g', "replace the pattern space with the hold space") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.replacePatternSpace();
		}

	};

	public static final SedCommand
	GL = new SedSimpleCommand(
			'G', "append the pattern space to the hold space") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.appendPatternSpace();
		}

	};

	public static final SedCommand
	X = new SedSimpleCommand(
			'x', "eXchange the hold and pattern spaces") {

		public void process(SedSpace s,
				SedLineBuffer rd) throws IOException {
			s.exchange();
		}

	};

}
