/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.unix.regex;

import junit.framework.TestCase;

public class BasicPatternTest extends TestCase {

	static void mt(String pt, String s) {
		assertTrue(BasicPattern.matches(pt, s));
	}

	static void mf(String pt, String s) {
		assertFalse(BasicPattern.matches(pt, s));
	}

	static void er(String pt) {
		try {
			BasicPattern.compile(pt);  fail();
		} catch(BasicPatternSyntaxException e) {}
	}

	static void at(String pt, String s) {
		assertTrue(BasicPattern.matchesAll(pt, s));
	}

	static void af(String pt, String s) {
		assertFalse(BasicPattern.matchesAll(pt, s));
	}

	static void rf(String pt, String s, String t, String v) {
		assertEquals(v, BasicPattern.compile(pt).matcher(s).replaceFirst(t));
	}

	static void ra(String pt, String s, String t, String v) {
		assertEquals(v, BasicPattern.compile(pt).matcher(s).replaceAll(t));
	}

	static void rn(String pt, String s, String t, String v, int n) {
		assertEquals(v, BasicPattern.compile(pt).matcher(s).replace(t, n));
	}

	public void testExtend11() {
		at("a\\+", "a");
		at("a\\+", "aa");
		af("a\\+", "");

		at("a\\?", "a");
		at("a\\?", "");
		af("a\\?", "aa");
	}

	public void testSeq1() {
		at("a\\{3\\}", "aaa");
		af("a\\{3\\}", "aaaa");
		af("a\\{3\\}", "aa");

		at("a\\{0\\}", "");
		af("a\\{0\\}", "a");
	}

	public void testSeq2() {
		at("a\\{3,5\\}", "aaa");
		at("a\\{3,5\\}", "aaaa");
		at("a\\{3,5\\}", "aaaaa");
		af("a\\{3,5\\}", "aaaaaa");
		af("a\\{3,5\\}", "aa");

		at("a\\{0,3\\}a", "a");
		at("a\\{0,3\\}a", "aa");
		at("a\\{0,3\\}a", "aaa");
		at("a\\{0,3\\}a", "aaaa");

		at("a\\{0,3\\}b", "b");
		at("a\\{0,3\\}b", "ab");
		at("a\\{0,3\\}b", "aab");
		at("a\\{0,3\\}b", "aaab");
	}

	public void testSeq3() {
		at("a\\{3,\\}", "aaa");
		at("a\\{3,\\}", "aaaa");
		at("a\\{3,\\}", "aaaaa");
		at("a\\{3,\\}", "aaaaaa");
		af("a\\{3,\\}", "aa");
	}

	public void testMatchAll4() {
		at("\\(.\\)\\1", "aa");
		af("\\(.\\)\\1", "ab");

		at("\\(..\\)\\1", "abab");
		at("\\(..\\)\\1", "aaaa");
		af("\\(..\\)\\1", "abbb");
		af("\\(..\\)\\1", "abac");
		af("\\(..\\)\\1", "abba");
		af("\\(..\\)\\1", "aaa");
		af("\\(..\\)\\1", "aa");
		af("\\(..\\)\\1", "a");

		at("\\(..\\)\\1\\1", "ababab");
		at("\\(..\\)\\1\\1", "aaaaaa");
		af("\\(..\\)\\1\\1", "aaaaab");

		at("\\(.*\\)\\1", "aa");
		at("\\(.*\\)\\1", "aaaa");
		at("\\(.*\\)\\1", "abab");
		at("\\(.*\\)\\1", "abaaba");
		at("\\(.*\\)\\1", "");
		af("\\(.*\\)\\1", "a");
		af("\\(.*\\)\\1", "aaa");

		at("\\(.*\\)\\1\\1", "aaa");
		at("\\(.*\\)\\1\\1", "aaaaaa");
		at("\\(.*\\)\\1\\1", "ababab");
		at("\\(.*\\)\\1\\1", "");
		af("\\(.*\\)\\1\\1", "aaaaa");
		af("\\(.*\\)\\1\\1", "aaaa");
		af("\\(.*\\)\\1\\1", "aa");

		at("c\\(.*\\)c\\1c", "cacac");
		at("c\\(.*\\)c\\1c", "ccc");
		at("c\\(.*\\)c\\1c", "cabcabc");
		at("c\\(.*\\)c\\1c", "ccccc");
		af("c\\(.*\\)c\\1c", "cccccc");
		af("c\\(.*\\)c\\1c", "");

		at("\\(.\\(.\\)\\)\\1\\2", "ababb");
		at("\\(.\\(.\\)\\)\\1\\2", "aaaaa");
		af("\\(.\\(.\\)\\)\\1\\2", "ababa");
		af("\\(.\\(.\\)\\)\\1\\2", "");

		at("\\(.*\\)\\(.\\)\\2\\1", "abba");
		at("\\(.*\\)\\(.\\)\\2\\1", "aaaabbaaaa");
		at("\\(.*\\)\\(.\\)\\2\\1", "bb");
		at("\\(.*\\)\\(.\\)\\2\\1", "bbbb");
		af("\\(.*\\)\\(.\\)\\2\\1", "");
		af("\\(.*\\)\\(.\\)\\2\\1", "b");
		af("\\(.*\\)\\(.\\)\\2\\1", "bbb");
		af("\\(.*\\)\\(.\\)\\2\\1", "abccba");
		af("\\(.*\\)\\(.\\)\\2\\1", "abccaa");
		af("\\(.*\\)\\(.\\)\\2\\1", "abcaab");
		af("\\(.*\\)\\(.\\)\\2\\1", "abcabc");
	}

	public void testMatchAll1() {
		at("a", "a");
		af("a", "b");
		af("a", "aa");
		af("a", "ab");
		af("a", "ba");
		af("a", "baa");
		at("abc", "abc");
		af("abc", "abb");
		af("abc", "aabc");
		af("abc", "abca");
		af("abc", "abcab");
		af("abc", "abcabc");
		af("abc", "babcb");
	}

	public void testMatchAll2() {
		af("[bcdf]", "a");
		at("[bcdf]", "b");
		at("[bcdf]", "c");
		at("[bcdf]", "d");
		af("[bcdf]", "e");
		at("[bcdf]", "f");

		at("[b-y]", "b");
		at("[b-y]", "h");
		at("[b-y]", "y");
		af("[b-y]", "a");
		af("[b-y]", "z");
		af("[b-y]", "[");
		af("[b-y]", "-");
		af("[b-y]", "]");

		at("[b-yB-Y]", "b");
		at("[b-yB-Y]", "h");
		at("[b-yB-Y]", "y");
		af("[b-yB-Y]", "a");
		af("[b-yB-Y]", "z");
		at("[b-yB-Y]", "B");
		at("[b-yB-Y]", "I");
		at("[b-yB-Y]", "Y");
		af("[b-yB-Y]", "A");
		af("[b-yB-Y]", "Z");
		af("[b-yB-Y]", "-");
		af("[b-yB-Y]", "]");

		at("[-a\\]", "a");
		at("[-a\\]", "-");
		af("[-a\\]", "[");
		af("[-a\\]", "\\");

		at("[a-z-1]", "a");
		at("[a-f-h]", "-");
		at("[a-f-h]", "h");
		af("[a-f-h]", "g");
		af("[a-f-h]", "i");

		at("[][]", "]");
		at("[][]", "[");

		af("[bcdf]", "bd");
		af("[bcdf]", "ac");
		af("[bcdf]", "da");
		af("[bcdf]", "aea");

		at(".", "a");
		at(".", "\n");
	}

	public void testMatchAll3() {
		at("a*", "a");
		at("a*", "");
		at("a*", "aaaaa");
		af("a*", "b");
		af("a*", "ab");
		af("a*", "ba");
		af("a*", "bab");

		at("a*b", "ab");
		at("a*b", "b");
		at("a*b", "aaaaab");
		af("a*b", "aaaaabb");

		at("a*ba*", "aba");
		at("a*ba*", "aaabaa");
		at("a*ba*", "b");
		af("a*ba*", "aaaaabb");

		at("[ab]*a[ab]*", "a");
		at("[ab]*a[ab]*", "aaaaaa");
		at("[ab]*a[ab]*", "bbabb");
		af("[ab]*a[ab]*", "");

		at("[ab]*bb", "abb");
		at("[ab]*bb", "bb");
		at("[ab]*bb", "aaaaabb");
		at("[ab]*bb", "aaaaabbb");
		af("[ab]*bb", "aaaaab");
		af("[ab]*bb", "b");
		af("[ab]*bb", "");

		at("[ab]*a[ab]*a[ab]*", "aa");
		at("[ab]*a[ab]*a[ab]*", "aaaaaa");
		at("[ab]*a[ab]*a[ab]*", "bbabbabb");
		af("[ab]*a[ab]*a[ab]*", "bab");
		af("[ab]*a[ab]*a[ab]*", "bbabb");

		at("[ab]*abab[ab]*", "aababa");
		at("[ab]*abab[ab]*", "abab");
		at("[ab]*abab[ab]*", "ababa");
		at("[ab]*abab[ab]*", "aabab");
		at("[ab]*abab[ab]*", "aaababaaa");
		at("[ab]*abab[ab]*", "bbababbb");
		at("[ab]*abab[ab]*", "abababab");
		af("[ab]*abab[ab]*", "aba");
		af("[ab]*abab[ab]*", "ab");
		af("[ab]*abab[ab]*", "a");
		af("[ab]*abab[ab]*", "");

		at("aa[ab]*bb", "aabb");
		at("aa[ab]*bb", "aaababbb");
		af("aa[ab]*bb", "bbaabbaa");
		af("aa[ab]*bb", "aab");
		af("aa[ab]*bb", "abb");
		af("aa[ab]*bb", "ab");
		af("aa[ab]*bb", "a");
		af("aa[ab]*bb", "b");
		af("aa[ab]*bb", "");
	}

	public void testMatch1() {
		mt("a", "a");
		mf("a", "b");
		mt("a", "aa");
		mt("a", "ab");
		mt("a", "ba");
		mt("a", "baa");
		mt("abc", "abc");
		mf("abc", "abb");
		mt("abc", "aabc");
		mt("abc", "abca");
		mt("abc", "abcab");
		mt("abc", "abcabc");
		mt("abc", "babcb");
	}

	public void testMatch2() {
		mt("[bcdf]", "bd");
		mt("[bcdf]", "ac");
		mt("[bcdf]", "da");
		mt("[bcdf]", "afa");
	}

	public void testMatch3() {
		mt("a*", "a");
		mt("a*", "");
		mt("a*", "aaaaa");
		mt("a*", "b");
		mt("a*", "ab");
		mt("a*", "ba");
		mt("a*", "bab");

		mt("a*b", "ab");
		mt("a*b", "b");
		mt("a*b", "aaaaab");
		mt("a*b", "aaaaabb");

		mt("a*ba*", "aba");
		mt("a*ba*", "aaabaa");
		mt("a*ba*", "b");
		mt("a*ba*", "aaaaabb");

		mt("[ab]*a[ab]*", "a");
		mt("[ab]*a[ab]*", "aaaaaa");
		mt("[ab]*a[ab]*", "bbabb");
		mf("[ab]*a[ab]*", "");

		mt("[ab]*bb", "abb");
		mt("[ab]*bb", "bb");
		mt("[ab]*bb", "aaaaabb");
		mt("[ab]*bb", "aaaaabbb");
		mf("[ab]*bb", "aaaaab");
		mf("[ab]*bb", "b");
		mf("[ab]*bb", "");

		mt("[ab]*a[ab]*a[ab]*", "aa");
		mt("[ab]*a[ab]*a[ab]*", "aaaaaa");
		mt("[ab]*a[ab]*a[ab]*", "bbabbabb");
		mf("[ab]*a[ab]*a[ab]*", "bab");
		mf("[ab]*a[ab]*a[ab]*", "bbabb");

		mt("[ab]*abab[ab]*", "aababa");
		mt("[ab]*abab[ab]*", "abab");
		mt("[ab]*abab[ab]*", "ababa");
		mt("[ab]*abab[ab]*", "aabab");
		mt("[ab]*abab[ab]*", "aaababaaa");
		mt("[ab]*abab[ab]*", "bbababbb");
		mt("[ab]*abab[ab]*", "abababab");
		mf("[ab]*abab[ab]*", "aba");
		mf("[ab]*abab[ab]*", "ab");
		mf("[ab]*abab[ab]*", "a");
		mf("[ab]*abab[ab]*", "");

		mt("aa[ab]*bb", "aabb");
		mt("aa[ab]*bb", "aaababbb");
		mt("aa[ab]*bb", "bbaabbaa");
		mf("aa[ab]*bb", "aab");
		mf("aa[ab]*bb", "abb");
		mf("aa[ab]*bb", "ab");
		mf("aa[ab]*bb", "a");
		mf("aa[ab]*bb", "b");
		mf("aa[ab]*bb", "");
	}

	public void testReplace1() {
		rf("a", "caaac", "b", "cbaac");
		ra("a", "caaac", "b", "cbbbc");
		rn("a", "caaac", "b", "cabac", 2);

		rf("ab", "cabababc", "b", "cbababc");
		ra("ab", "cabababc", "b", "cbbbc");
		rn("ab", "cabababc", "b", "cabbabc", 2);

		rf("a", "caaac", "df", "cdfaac");
		ra("a", "caaac", "df", "cdfdfdfc");
		rn("a", "caaac", "df", "cadfac", 2);

		rf("aa*", "caaac", "b", "cbc");
		rf("aa*", "caaacaaacaaac", "b", "cbcaaacaaac");
		ra("aa*", "caaacaaacaaac", "b", "cbcbcbc");
		rn("aa*", "caaacaaacaaac", "b", "caaacbcaaac", 2);
		ra("aa*", "aaaaa", "b", "b");

		rf("a*", "caaac", "b", "bcaaac");
		ra("a*", "caaac", "b", "bcbcb");

		ra("\\([a-z][a-z]*\\)", "abcd", "\\1", "abcd");
		ra("\\([a-z][a-z]*\\)", "", "\\1", "");

		ra("\\([a-z]*\\)", "AaaaA", "\\1\\1", "AaaaaaaA");
		rf("\\([a-z]*\\)", "AaaaA", "\\1\\1", "AaaaA");
		rn("\\([a-z]*\\)", "AaaaA", "\\1\\1", "AaaaaaaA", 2);

		ra("\\([a-z][a-z]*\\)\\([a-z][a-z]*\\)", "abcd", "\\2\\1", "dabc");
	}

	public void testAnchor1() {
		mt("abc", "abc");
		mt("abc", "aabc");
		mt("abc", "abca");
		mt("abc", "aabca");

		mt("^abc", "abc");
		mf("^abc", "aabc");
		mt("^abc", "abca");
		mf("^abc", "aabca");

		mt("abc$", "abc");
		mt("abc$", "aabc");
		mf("abc$", "abca");
		mf("abc$", "aabca");

		mt("^abc$", "abc");
		mf("^abc$", "aabc");
		mf("^abc$", "abca");
		mf("^abc$", "aabca");

		mt("", "");
		mt("", "a");
		mt("^", "");
		mt("^", "a");
		mt("$", "");
		mt("$", "a");
		mt("^$", "");
		mf("^$", "a");

		mt("^^$$", "^$");
		mt("\\^", "^");
		mt("\\$", "$");
	}

}
