#include <stdio.h>

//色定数
#define COL8_BLACK			0
#define COL8_RED			1
#define COL8_GREEN			2
#define	COL8_YELLOW			3
#define COL8_BLUE			4
#define COL8_PURPLE			5
#define COL8_PALEBLUE		6
#define COL8_WHITE			7
#define	COL8_LIGHT_GRAY		8
#define	COL8_DARK_RED		9
#define COL8_DARK_GREEN 	10
#define	COL8_DARK_YELLOW	11
#define COL8_DARK_BLUE		12
#define COL8_DARK_PURPLE	13
#define COL8_DARK_PALEBLUE	14
#define	COL8_DARK_GRAY		15

//CPUを一時停止する。
void io_hlt(void);
void io_cli(void);
void io_out8(int port, int data);
int io_load_eflags(void);
void io_store_eflags(int eflags);
void load_gdtr(int limit, int addr);
void load_idtr(int limit, int addr);


void init_palette(void);
void set_palette(int start, int end, unsigned char *rgb);

void boxfill8(unsigned char *vram, int xsize, unsigned char c, int x0, int y0, int x1, int y1);
void init_screen(char *vram, int xsize, int ysize);
void putfont8(char *vram, int xsize, int x, int y, char color, char *font);
void putfonts8_asc(char *vram, int xsize, int x, int y, char color, unsigned char *str);
void init_mouse_cursor8(char *mouse, char bc);
void putblock8_8(char *vram, int vxsize, int pxsize, int pysize, int px0, int py0, char *buf, int bxsize);
void init_gdtidt(void);


struct BOOTINFO {
	char cyls, leds, vmode, reserve;
	short scrnx, scrny;
	char *vram;
};

struct SEGMENT_DESCRIPTOR {
	short limit_low, base_low;
	char base_mid, access_right;
	char limit_high, base_high;
};

struct GATE_DESCRIPTOR {
	short offset_low, selector;
	char dw_count, access_right;
	short offset_high;
};

void set_gatedesc(struct GATE_DESCRIPTOR *gd, int offset, int selector, int ar);
void set_segmdesc(struct SEGMENT_DESCRIPTOR *sd, unsigned int limit, int base, int ar);

void HariMain(void)
{
	char s[100];
	struct BOOTINFO *binfo = (struct BOOTINFO *) 0x0ff0;
    char mcursor[16 * 16];
	init_gdtidt();
	int mx = 152, my = 78;
	init_palette();
	init_screen(binfo->vram, binfo->scrnx, binfo->scrny);
	putfonts8_asc(binfo->vram, binfo->scrnx, 8, 8, COL8_RED, "ABC 123");
	putfonts8_asc(binfo->vram, binfo->scrnx, 31,31, COL8_BLACK, "Haribote OS.");
	putfonts8_asc(binfo->vram, binfo->scrnx, 30,30, COL8_WHITE, "Haribote OS.");
	sprintf(s, "(%d,%d)", mx, my);
	putfonts8_asc(binfo->vram, binfo->scrnx, 16, 64, COL8_BLACK, s);
	
	//マウスの設定
	init_mouse_cursor8(mcursor, COL8_DARK_PALEBLUE);
	putblock8_8(binfo->vram, binfo->scrnx, 16, 16, mx, my, mcursor, 16);
	while(1){
		io_hlt();
	}
}
void init_gdtidt(void)
{
	struct SEGMENT_DESCRIPTOR *gdt = (struct SEGMENT_DESCRIPTOR *) 0x00270000;
	struct GATE_DESCRIPTOR *idt = (struct GATE_DESCRIPTOR *) 0x0025f800;
	int i;
	
	/*GDTの初期化*/
	for (i = 0; i < 8192; i++){
		set_segmdesc(gdt + i, 0, 0, 0);
	}
	set_segmdesc(gdt + 1, 0xffffffff, 0x00000000, 0x4092);
	set_segmdesc(gdt + 2, 0x0007ffff, 0x00280000, 0x409a);
	load_gdtr(0xffff, 0x00270000);
	
	/*IDTの初期化*/
	for(i = 0; i < 256; i++){
		set_gatedesc(idt + i, 0, 0, 0);
	}
	load_idtr(0x7ff, 0x0026f800);
	
	return;
}

void set_segmdesc(struct SEGMENT_DESCRIPTOR *sd, unsigned int limit, int base, int ar)
{
	if (limit> 0xfffff){
		ar |= 0x8000; // G_bit = 1;
		limit /= 0x1000;
	}
	sd->limit_low = limit & 0xffff;
	sd->base_low = base & 0xffff;
	sd->base_mid = (base >> 16) & 0xff;
	sd->access_right = ar & 0xff;
	sd->limit_high = ((limit >> 16) & 0x0f) | ((ar >> 8) & 0xf0);
	sd->base_high = (base >> 24) & 0xff;
	return;
}

void set_gatedesc(struct GATE_DESCRIPTOR *gd, int offset, int selector, int ar)
{
	gd->offset_low = offset & 0xffff;
	gd->selector = selector;
	gd->dw_count = (ar >> 8) & 0xff;
	gd->access_right = ar & 0xff;
	gd->offset_high = (offset >> 16) & 0xffff;
	return;
}
void init_mouse_cursor8(char *mouse, char backcolor)
/*マウスカーソルを準備(16x16*/
{
	static char cursor[16][16] = {
		"**************..",
		"*ooooooooooo*...",
		"*oooooooooo*....",
		"*ooooooooo*.....",
		"*oooooooo*......",
		"*ooooooo*.......",
		"*ooooooo*.......",
		"*oooooooo*......",
		"*oooo**ooo*.....",
		"*ooo*..*ooo*....",
		"*oo*....*ooo*...",
		"*o*......*ooo*..",
		"**........*ooo*.",
		"*..........*ooo*",
		"............*oo*",
		"............,***"
	};
	int x, y;
	
	for (y = 0; y < 16; y++){
		for (x = 0; x < 16; x++){
			if (cursor[y][x] == '*'){
				mouse[y * 16 + x] = COL8_BLACK;
			}
			if (cursor[y][x] == 'o'){
				mouse[y * 16 + x] = COL8_WHITE;
			}
			if (cursor[y][x] == '.'){
				mouse[y * 16 + x] = backcolor;
			}
		}
	}
	return;
}
void putblock8_8(char *vram, int vxsize, int pxsize, int pysize, int px0, int py0, char *buf, int bxsize)
{
	int x, y;
	for (y = 0; y < pysize; y++){
		for (x = 0; x < pxsize; x++){
			vram[(py0 + y) * vxsize + (px0 + x)] = buf[y * bxsize + x];
		}
	}
	return;
}
void putfonts8_asc(char *vram, int xsize, int x, int y, char color, unsigned char *str)
{
	extern char hankaku[4096];
	for(;*str != '\0'; str++){
		putfont8(vram, xsize, x, y, color, hankaku + *str * 16);
		x += 8;
	}
	return;
}
void putfont8(char *vram, int xsize, int x, int y, char color, char *font)
{
	int i;
	char d; /*data*/
	char *p;
	for (i = 0; i < 16; i++){
		d = font[i];
		p = vram + (y + i) * xsize + x;
		if ((d & 0x80) != 0) { p[0] = color;}
		if ((d & 0x40) != 0) { p[1] = color;}
		if ((d & 0x20) != 0) { p[2] = color;}
		if ((d & 0x10) != 0) { p[3] = color;}
		if ((d & 0x08) != 0) { p[4] = color;}
		if ((d & 0x04) != 0) { p[5] = color;}
		if ((d & 0x02) != 0) { p[6] = color;}
		if ((d & 0x01) != 0) { p[7] = color;}
	}
	return;
}
void init_screen(char *vram, int xsize, int ysize)
{
	boxfill8(vram, xsize, COL8_DARK_PALEBLUE, 0, 0, xsize - 1, ysize - 29);
	boxfill8(vram, xsize, COL8_LIGHT_GRAY, 0, ysize - 28, xsize - 1, ysize - 28);
	boxfill8(vram, xsize, COL8_WHITE, 0, ysize - 27, xsize - 1, ysize - 27);
	boxfill8(vram, xsize, COL8_LIGHT_GRAY, 0, ysize - 26, xsize - 1, ysize - 1);
	
	boxfill8(vram, xsize, COL8_WHITE, 3, ysize - 24, 59, ysize -24);
	boxfill8(vram, xsize, COL8_WHITE, 2, ysize - 24, 2, ysize - 4);
	boxfill8(vram, xsize, COL8_DARK_GRAY, 3, ysize - 4, 59, ysize -4);
	boxfill8(vram, xsize, COL8_DARK_GRAY, 59, ysize - 23, 59, ysize - 5);
	boxfill8(vram, xsize, COL8_BLACK, 2, ysize - 3, 59, ysize - 3);
	boxfill8(vram, xsize, COL8_BLACK, 60, ysize - 24, 60, ysize - 3);
	
	boxfill8(vram, xsize, COL8_DARK_GRAY, xsize - 47, ysize - 24, xsize - 4, ysize - 24);
	boxfill8(vram, xsize, COL8_DARK_GRAY, xsize - 47, ysize - 23, xsize - 47, ysize - 4);
	boxfill8(vram, xsize, COL8_WHITE, xsize - 47, ysize - 3, xsize - 4, ysize - 3);
	boxfill8(vram, xsize, COL8_WHITE, xsize - 3, ysize - 24, xsize - 3, ysize - 3);
}

void boxfill8(unsigned char *vram, int xsize, unsigned char c, int x0, int y0, int x1, int y1)
{
	int x, y;
	for (y = y0; y <= y1; y++){
		for (x = x0; x <= x1; x++){
			vram[y * xsize + x] = c;
		}
	}
	return;
}

void init_palette(void)
{
	static unsigned char table_rgb[16 * 3] = {
		0x00, 0x00, 0x00, /*0:黒*/
		0xff, 0x00, 0x00, /*1:明るい赤 */
		0x00, 0xff, 0x00, /*2:明るい緑*/
		0xff, 0xff, 0x00, /*3:明るい黄色*/
		0x00, 0x00, 0xff, /*4:明るい青*/
		0xff, 0x00, 0xff, /*5:明るい紫*/
		0x00, 0xff, 0xff, /*6:明るい水色*/
		0xff, 0xff, 0xff, /*7:白*/
		0xc6, 0xc6, 0xc6, /*8:明るい灰色*/
		0x84, 0x00, 0x00, /*9:暗い赤*/
		0x00, 0x84, 0x00, /*10:暗い緑*/
		0x84, 0x84, 0x00, /*11:暗い黄色*/
		0x00, 0x00, 0x84, /*12:暗い青*/
		0x84, 0x00, 0x84, /*13:暗い紫*/
		0x00, 0x84, 0x84, /*14:暗い水色*/
		0x84, 0x84, 0x84, /*15:暗い灰色*/
	};
	set_palette(0, 15, table_rgb);
	return;
}

void set_palette(int start, int end, unsigned char *rgb)
{
	int i, eflags;
	eflags = io_load_eflags(); //割り込み許可フラグの値を記録する。
	io_cli(); //許可フラグを0にして割り込み禁止にする
	io_out8(0x3c8, start);
	for (i = start; i <= end; i++){
		io_out8(0x3c9, rgb[0] / 4);
		io_out8(0x3c9, rgb[1] / 4);
		io_out8(0x3c9, rgb[2] / 4);
		rgb += 3;
	}
	io_store_eflags(eflags); //割り込み許可フラグを元に戻す。
	return;
}