#include <stdio.h>
#include "bootpack.h"
#include "fifo.h"

extern struct fifo keyfifo;
extern struct fifo mousefifo;

#define PORT_KEYDAT 0x0060
#define PORT_KEYSTA 0x0064
#define PORT_KEYCMD 0x0064
#define KEYSTA_SEND_NOTREADY 0x02
#define KEYCMD_WRITE_MODE 0x60
#define KBC_MODE 0x47

/**
* @fn キーボードコントローラがデータ送信可能になるのを待つ
**/
void wait_KBC_sendready(void);

/**
* @fn キーボードコントローラの初期化
**/
void init_keyboard(void);

#define KEYCMD_SENDTO_MOUSE	0xd4
#define MOUSECMD_ENABLE	0xf4

/**
* @fn マウスを3バイトずつ読むためのデータ構造
**/
struct MOUSE_DEC {
	unsigned char buf[3], phase;
	int x, y, btn;
};

#define MOUSE_LEFT_BUTTON 0x01
#define MOUSE_RIGHT_BUTTON 0x02
#define MOUSE_CENTRAL_BUTTON 0x04

/**
* @fn マウス有効化
**/
void enable_mouse(struct MOUSE_DEC *mdec);

/**
* @fn マウスの解析
**/
int mouse_decode(struct MOUSE_DEC *mdec, unsigned char dat);


void HariMain(void)
{
	char s[100];
	struct BOOTINFO *binfo = (struct BOOTINFO *) ADR_BOOTINFO;
    char mcursor[16 * 16];
    char keybuf[32], mousebuf[128];
    struct MOUSE_DEC mdec;

	int mx, my, i;
	
	init_gdtidt();
	init_pic();
	io_sti(); /* IDT/PICの初期化が終わったのでCPUの割り込み禁止を解除 */
	

	init_palette();
	init_screen(binfo->vram, binfo->scrnx, binfo->scrny);
	

	io_out8(PIC0_IMR, 0xf9); /* PIC1とキーボードを許可(11111001) */
	io_out8(PIC1_IMR, 0xef); /* マウスを許可(11101111) */
	
	init_keyboard();	

	
	mx = (binfo->scrnx - 16) / 2; /* 画面中央になるように座標計算 */
	my = (binfo->scrny - 28 - 16) / 2;

	
	//マウスの設定
	init_mouse_cursor8(mcursor, COL8_DARK_PALEBLUE);
	putblock8_8(binfo->vram, binfo->scrnx, 16, 16, mx, my, mcursor, 16);
	
	sprintf(s, "(%d, %d)", mx, my);
	putfonts8_asc(binfo->vram, binfo->scrnx, 0, 0, COL8_WHITE, s);
	
	enable_mouse(&mdec);
		
	fifo8_init(&keyfifo, LENOF(keybuf), keybuf);
	fifo8_init(&mousefifo, LENOF(mousebuf), mousebuf);

	while(1){
		io_cli();
		if (fifo8_status(&keyfifo) + fifo8_status(&mousefifo) == 0){
			io_stihlt();
		} else {
			if(fifo8_status(&keyfifo) != 0) {
				i = fifo8_get(&keyfifo);
				io_sti();
				sprintf(s, "%02X", i);
				boxfill8(binfo->vram, binfo->scrnx, COL8_DARK_PURPLE, 0, 16, 15, 31);
				putfonts8_asc(binfo->vram, binfo->scrnx, 0, 16, COL8_WHITE, s);
			} else if (fifo8_status(&mousefifo) != 0){
				i = fifo8_get(&mousefifo);
				io_sti();
				if (mouse_decode(&mdec, i) != 0) {
					/* データが３バイト揃ったので表示 */
					sprintf(s, "[lcr %4d %4d]", mdec.x, mdec.y);
					if ((mdec.btn & MOUSE_LEFT_BUTTON) != 0) {
						s[1] = 'L';
					}
					if ((mdec.btn & MOUSE_RIGHT_BUTTON) != 0) {
						s[3] = 'R';
					}
					if ((mdec.btn & MOUSE_CENTRAL_BUTTON) != 0) {
						s[2] = 'C';
					}
					boxfill8(binfo->vram, binfo->scrnx, COL8_DARK_PALEBLUE, 32, 16, 32 + 15 * 8 - 1, 31);
					putfonts8_asc(binfo->vram, binfo->scrnx, 32, 16, COL8_WHITE, s);
					/* マウスカーソルの移動 */
					boxfill8(binfo->vram, binfo->scrnx, COL8_DARK_PALEBLUE, mx, my, mx + 15, my + 15); /*マウスを消す*/
					mx += mdec.x;
					my += mdec.y;
					if (mx < 0) {
						mx = 0;
					}
					if (mx > binfo->scrnx - 16) {
						mx = binfo-> scrnx - 16;
					}
					if (my < 0) {
						my = 0;
					}
					if (my > binfo->scrny - 16) {
						my = binfo-> scrny - 16;
					}
					sprintf(s, "(%3d, %3d)", mx, my);
					boxfill8(binfo->vram, binfo->scrnx, COL8_DARK_PALEBLUE, 0, 0, 79, 15);/*座標を消す*/
					putfonts8_asc(binfo->vram, binfo->scrnx, 0, 0, COL8_WHITE, s);/*座標を書く*/
					putblock8_8(binfo->vram, binfo->scrnx, 16, 16, mx, my, mcursor, 16); /*マウスを描く*/
				}
			}
		}
	}
}

void enable_mouse(struct MOUSE_DEC *mdec)
{
	/* マウス有効 */
	wait_KBC_sendready();
	io_out8(PORT_KEYCMD, KEYCMD_SENDTO_MOUSE);
	wait_KBC_sendready();
	io_out8(PORT_KEYDAT, MOUSECMD_ENABLE);
	/* うまくいくとACK(0xfa)が送信されてくる。*/
	mdec->phase = 0; /* マウスのACKを待っている段階 */
	return;
}

int mouse_decode(struct MOUSE_DEC *mdec, unsigned char dat)
{
	int x, y;
	if (mdec->phase == 0) {
		/* マウスの0xfaを待っている段階 */
		if (dat == 0xfa) {
			mdec->phase = 1;
		}
		return 0;
	} else if (mdec->phase == 1) {
		/* マウスの1バイト目を待っている段階 */	
		if((dat & 0xc8) == 0x08) {
			/*正しい1バイト目だった*/
			mdec->buf[0] = dat;
			mdec->phase = 2;
		}
		return 0;
	} else if (mdec->phase == 2) {
		/* マウスの2バイト目を待っている段階 */
		mdec->buf[1] = dat;
		mdec->phase = 3;
		return 0;
	} else if (mdec->phase == 3) {
		/* マウスの3バイト目を待っている段階 */
		mdec->buf[2] = dat;
		mdec->phase = 1;
		mdec->btn = mdec->buf[0] & 0x07;
		x = mdec->buf[1];
		y = mdec->buf[2];
		if ((mdec->buf[0] & 0x10) != 0) {
			x |= 0xffffff00;
		}
		if ((mdec->buf[0] & 0x20) != 0) {
			y |= 0xffffff00;
		}
		mdec->x =  x;
		mdec->y = -y;/*マウスではy方向の符号が画面と反対 */
		return 1;
	}
	return -1; /*ここに来るはずはない。*/
}
void wait_KBC_sendready(void)
{
	for(;;) {
		if ((io_in8(PORT_KEYSTA) & KEYSTA_SEND_NOTREADY) == 0){
			break;
		}
	}
	return;
}

void init_keyboard(void)
{
	wait_KBC_sendready();
	io_out8(PORT_KEYCMD, KEYCMD_WRITE_MODE);
	wait_KBC_sendready();
	io_out8(PORT_KEYDAT, KBC_MODE);
	return;
}