using System;
using nft.core.game;
using nft.core.view;
using System.Diagnostics;

namespace nft.core.schedule
{
	/// <summary>
	/// Span of time
	/// </summary>
	[Serializable]
	public class TimeLength : Time
	{
        internal TimeLength(long _ticks) : base(_ticks){
        }
		
		// create new time span objects that correspond to the specified period of the time.
        public static TimeLength FromMinutes(long min) {
            return FromMinutes(min, Calendar.ActiveCalendar);
        }
        public static TimeLength FromHours(ITimeTickConverter converter, long hours) {
            return FromHours(hours, Calendar.ActiveCalendar);
        }
        public static TimeLength FromDays(ITimeTickConverter converter, long days) {
            return FromDays(days, Calendar.ActiveCalendar);
        }
        public static TimeLength FromMinutes(long min, Calendar c) {
            return new TimeLength(c.TimeTickConverter.MinutelyTicks * min);
        }
        public static TimeLength FromHours(long hours, Calendar c) {
            return new TimeLength(c.TimeTickConverter.HourlyTicks * hours);
        }
        public static TimeLength FromDays(long days, Calendar c) {
            return new TimeLength(c.TimeTickConverter.DailyTicks * days);
        }
		
        public override string ToString() {
            long days;
            long t2 = Math.DivRem(ticks, Converter.DailyTicks, out days);
            if (days > 30) {
                return string.Format("TimeLength:[about {0} Days. (ticks={1})]", days, Ticks);
            } else {
                TimeOfDay td = Converter.ToTimeOfDay(t2);
                if (days == 0) {
                    return string.Format("TimeLength:[about {0} Hours & {1} Mins. (ticks={2})]",
                        days, td.Hour, td.Minute, Ticks);
                } else {
                    return string.Format("TimeLength:[about {0} Days & {1} Hours. (ticks={2})]",
                        days, td.Hour, Ticks);
                }
            }
        }
		
		public static TimeLength operator + ( TimeLength a, TimeLength b ) 
		{
            Debug.Assert(a.Converter == b.Converter);
            return new TimeLength(a.Ticks + b.Ticks );
		}
		public static TimeLength operator - ( TimeLength a, TimeLength b ) 
		{
            Debug.Assert(a.Converter == b.Converter);
            return new TimeLength(a.Ticks - b.Ticks);
		}
	}
}
