﻿using nft.framework;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace nft.framework.repository
{
    /// <summary>
    /// データリポジトリのコアインターフェース
    /// 標準実装はRepositoryBase
    /// </summary>
    /// <typeparam name="S"></typeparam>
    /// <typeparam name="T"></typeparam>
    public interface IRepository<S,T> where T: IHasNameAndID where S: IIdentifier
    {
        /// <summary>
        /// モデルを追加する。既にモデル
        /// </summary>
        /// <param name="model"></param>
        void Add(T model);

        /// <summary>
        /// IDで指定したモデルを削除し、削除したモデルを返す。該当モデルが存在しない場合エラーにはならず、nullを返す。
        /// </summary>
        /// <param name="id"></param>
        /// <returns>removed model or null if not exist</returns>
        T RemoveByID(S id);

        /// <summary>
        /// IDを指定してモデルを取得する。
        /// setがないのは、間違ったIDでmodel登録されるのを防ぐため。Addを使用してください。
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        T this[S id] { get; }

        /// <summary>
        /// リポジトリに含まれるすべてのモデルのリストを取得します
        /// </summary>
        IEnumerable<T> List { get; }

        /// <summary>
        /// リポジトリに含まれるモデルの数を取得します
        /// </summary>
        int Count { get; }
    }

    /// <summary>
    /// IRepositoryの拡張メソッド
    /// </summary>
    public static class RepositoryExtesion
    {
        /// <summary>
        /// 指定したモデルを削除する。
        /// </summary>
        /// <typeparam name="S"></typeparam>
        /// <typeparam name="T"></typeparam>
        /// <param name="repository"></param>
        /// <param name="model">The model to be removed</param>
        public static void Remove<S,T>(this IRepository<S,T> repository, T model) where T: IHasNameAndID where S : IIdentifier
        {
            repository.RemoveByID((S) model.ID);
        }

        /// <summary>
        /// リポジトリに指定したIDのモデルが存在するかどうか調べる。存在すればtrue
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="repository"></param>
        /// <param name="model"></param>
        /// <returns>true if the model correspond to the id is exist.</returns>
        public static bool HasID<S,T>(this IRepository<S,T> repository, T model) where T : class, IHasNameAndID where S : IIdentifier
        {
            return repository[(S) model.ID] != null;
        }

        public static T findByID<S,T>(this IRepository<S,T> repository, S id, T _default = null) where T : class, IHasNameAndID where S : IIdentifier
        {
            return repository[id] ?? _default;
        }
    }
}
