package project.common.db;

import java.util.Objects;

import core.util.NumberUtil;

/**
 * メタ情報
 * @author Tadashi Nakayama
 * @version 1.0.0
 */
public final class DBColumnInfo {
	/** タイプ */
	private final int type;
	/** サイズ */
	private final int size;
	/** null不可 */
	private final boolean notnull;
	/** コメント */
	private final String comment;
	/** スケール */
	private final int scale;
	/** デフォルト値 */
	private final Object def;

	/**
	 * コンストラクタ
	 * @param arg0 タイプ
	 * @param arg1 サイズ
	 * @param arg2 null不可
	 * @param arg3 コメント
	 * @param arg4 スケール
	 * @param arg5 デフォルト値
	 */
	public DBColumnInfo(final int arg0, final int arg1, final boolean arg2,
			final String arg3, final int arg4, final String arg5) {
		this.type = arg0;
		this.size = arg1;
		this.notnull = arg2;
		this.comment = arg3;
		this.scale = arg4;
		this.def = toValue(arg5);
	}

	/**
	 * タイプ取得
	 * @return タイプ
	 */
	public int getType() {
		return this.type;
	}

	/**
	 * サイズ取得
	 * @return サイズ
	 */
	public int getSize() {
		return this.size;
	}

	/**
	 * null不可
	 * @return null不可の場合 true を返す。
	 */
	public boolean isNotNull() {
		return this.notnull;
	}

	/**
	 * コメント取得
	 * @return コメント
	 */
	public String getComment() {
		return this.comment;
	}

	/**
	 * スケール取得
	 * @return スケール
	 */
	public int getScale() {
		return this.scale;
	}

	/**
	 * デフォルト値取得
	 * @return デフォルト値
	 */
	public Object getDefault() {
		return this.def;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "[type:" + this.type + " size:" + this.size
				+ " not null:" + this.notnull + " comment:" + this.comment
				+ " scale:" + this.scale + "]";
	}

	/**
	 * 値変換
	 * 関数未対応
	 * @param val デフォルト値
	 * @return 値
	 */
	private Object toValue(final String val) {
		if (!Objects.toString(val, "").isEmpty()) {
			if (val.startsWith("'")) {
				final int loc = val.lastIndexOf('\'');
				return val.substring("'".length(), loc);
			}
			return NumberUtil.toBigDecimal(val);
		}
		return null;
	}
}
