/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.impl;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;

import org.apache.commons.chain2.Catalog;
import org.apache.commons.chain2.CatalogFactory;
import org.apache.commons.chain2.ChainUtil;
import org.apache.commons.chain2.Command;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * <p>A simple implementation of {@link CatalogFactory}.</p>
 *
 * @version $Id$
 * @param <K> Context key type
 * @param <V> Context value type
 * @param <C> Type of the context associated with this command
 */
public class CatalogFactoryBase<K, V, C extends Map<K, V>> implements CatalogFactory<K, V, C> {

    // ------------------------------------------------------- Static Variables

    /**
     * <p>The set of registered {@link CatalogFactoryBase} instances,
     * keyed by the relevant class loader.</p>
     */
    private static final Map<ClassLoader, CatalogFactoryBase<?, ?, ? extends Map<?, ?>>> FACTORIES =
            new HashMap<>();

    // ----------------------------------------------------- Instance Variables

    /**
     * <p>The default {@link Catalog} for this {@link CatalogFactoryBase}.</p>
     */
    private Catalog<K, V, C> catalog = null;

    /**
     * <p>Map of named {@link Catalog}s, keyed by catalog name.</p>
     */
    private final Map<String, Catalog<K, V, C>> catalogs = new ConcurrentHashMap<>();

    // ----------------------------------------------------------- Constructors

    /**
     * <p>Construct an empty instance of {@link CatalogFactoryBase}.  This
     * constructor is intended solely for use by {@link CatalogFactory}.</p>
     */
    public CatalogFactoryBase() {
        super();
    }

    // --------------------------------------------------------- Public Methods

    /**
     * <p>Gets the default instance of Catalog associated with the factory
     * (if any); otherwise, return <code>null</code>.</p>
     *
     * @return the default Catalog instance
     */
    @Override
    public Catalog<K, V, C> getCatalog() {
        return this.catalog;
    }

    /**
     * <p>Sets the default instance of Catalog associated with the factory.</p>
     *
     * @param val the default Catalog instance
     */
    @Override
    public void setCatalog(final Catalog<K, V, C> val) {
        this.catalog = val;
    }

    /**
     * <p>Retrieves a Catalog instance by name (if any); otherwise
     * return <code>null</code>.</p>
     *
     * @param name the name of the Catalog to retrieve
     * @return the specified Catalog
     */
    @Override
    public Catalog<K, V, C> getCatalog(final String name) {
        return this.catalogs.get(name);
    }

    /**
     * <p>Adds a named instance of Catalog to the factory (for subsequent
     * retrieval later).</p>
     *
     * @param name    the name of the Catalog to add
     * @param val the Catalog to add
     */
    @Override
    public void addCatalog(final String name, final Catalog<K, V, C> val) {
        this.catalogs.put(name, val);
    }

    /**
     * <p>Return an <code>Stream</code> over the set of named
     * {@link Catalog}s known to this {@link CatalogFactoryBase}.
     * If there are no known catalogs, an empty Stream is returned.</p>
     *
     * @return An Stream of the names of the Catalogs known by this factory.
     */
    @Override
    public Stream<String> getNames() {
        return this.catalogs.keySet().stream();
    }

    /**
     * @see org.apache.commons.chain2.CatalogFactory#getCommand(java.lang.String)
     */
    @Override
    public <T extends Command<K, V, C>> T getCommand(final String commandID) {
        String commandName = commandID;
        String catalogName = null;

        if (commandID != null) {
            int splitPos = commandID.indexOf(DELIMITER.codePointAt(0));
            if (splitPos != -1) {
                catalogName = commandID.substring(0, splitPos);
                commandName = commandID.substring(splitPos + DELIMITER.length());
                if (commandName.contains(DELIMITER)) {
                    throw new IllegalArgumentException(String.format(
                            "commandID [%s] has too many delimiters (reserved for future use)",
                            commandID));
                }
            }
        }

        Catalog<K, V, C> ctlg;
        if (catalogName != null) {
            ctlg = getCatalog(catalogName);
            if (ctlg == null) {
                Log log = LogFactory.getLog(CatalogFactoryBase.class);
                log.warn("No catalog found for name: " + catalogName + ".");
                return null;
            }
        } else {
            ctlg = getCatalog();
            if (ctlg == null) {
                Log log = LogFactory.getLog(CatalogFactoryBase.class);
                log.warn("No default catalog found.");
                return null;
            }
        }

        return ctlg.getCommand(commandName);
    }

    // -------------------------------------------------------- Static Methods

    /**
     * <p>Return the singleton {@link CatalogFactoryBase} instance
     * for the relevant <code>ClassLoader</code>.  For applications
     * that use a thread context class loader (such as web applications
     * running inside a servlet container), this will return a separate
     * instance for each application, even if this class is loaded from
     * a shared parent class loader.</p>
     *
     * @param <K> Context key type
     * @param <V> Context value type
     * @param <C> Type of the context associated with this command
     * @return the per-application singleton instance of {@link CatalogFactoryBase}
     */
    public static <K, V, C extends Map<K, V>> CatalogFactory<K, V, C> getInstance() {

        CatalogFactoryBase<?, ?, ? extends Map<?, ?>> factory;
        ClassLoader cl = getClassLoader();

        synchronized (FACTORIES) {
            factory = FACTORIES.get(cl);
            if (factory == null) {
                factory = new CatalogFactoryBase<K, V, C>();
                FACTORIES.put(cl, factory);
            }
        }

        /* This should always convert cleanly because we are using the
         * base most generic for definition. */
        CatalogFactory<K, V, C> result = ChainUtil.cast(factory);

        return result;
    }

    /**
     * <p>Clear all references to registered catalogs, as well as to the
     * relevant class loader.  This method should be called, for example,
     * when a web application utilizing this class is removed from
     * service, to allow for garbage collection.</p>
     */
    public static void clear() {
        synchronized (FACTORIES) {
            FACTORIES.remove(getClassLoader());
        }
    }

    // ------------------------------------------------------- Private Methods

    /**
     * <p>Return the relevant <code>ClassLoader</code> to use as a Map key
     * for this request.  If there is a thread context class loader, return
     * that; otherwise, return the class loader that loaded this class.</p>
     * @return ClassLoader
     */
    private static ClassLoader getClassLoader() {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        if (cl == null) {
            cl = getClassLoader();
        }
        return cl;
    }

    /**
     * Check to see if we have an implementation of a valid configuration
     * parsing class loaded at runtime. If not, we throw an
     * IllegalStateException.
     */
    public static void checkForValidConfigurationModule() {
        try {
            ClassLoader cl = getClassLoader();
            cl.loadClass("org.apache.commons.chain2.config.ConfigParser");
        } catch (final ClassNotFoundException e) {
            String msg = "Couldn't not find a configuration implementation. "
                    + "Load a chain configuration module such as xml-configuration "
                    + "into the classpath and try again.";
            throw new IllegalStateException(msg, e);
        }
    }

}
