/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.web.servlet;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.commons.chain2.web.MapEntry;

/**
 * <p>Private implementation of <code>Map</code> for HTTP session
 * attributes.</p>
 * @version $Id$
 */
final class ServletSessionScopeMap implements Map<String, Object>, Serializable {

    /** serialVersionUID */
    private static final long serialVersionUID = 4102556809380028675L;

    /** HttpSession */
    private transient HttpSession session = null;
    /** HttpServletRequest */
    private transient HttpServletRequest request = null;

    /**
     * Constructor
     *
     * @param req HttpServletRequest
     */
    ServletSessionScopeMap(final HttpServletRequest req) {
        this.request = req;
        sessionExists();
    }

    /**
     * @see java.util.Map#clear()
     */
    @Override
    public void clear() {
        if (sessionExists()) {
            for (final String key : keySet()) {
                this.session.removeAttribute(key);
            }
        }
    }

    /**
     * @see java.util.Map#containsKey(java.lang.Object)
     */
    @Override
    public boolean containsKey(final Object key) {
        return sessionExists() && this.session.getAttribute(toKey(key)) != null;
    }

    /**
     * @see java.util.Map#containsValue(java.lang.Object)
     */
    @Override
    public boolean containsValue(final Object value) {
        if (value != null && sessionExists()) {
            for (final String key : Collections.list(this.session.getAttributeNames())) {
                Object next = this.session.getAttribute(key);
                if (value.equals(next)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @see java.util.Map#entrySet()
     */
    @Override
    public Set<Entry<String, Object>> entrySet() {
        Set<Entry<String, Object>> set = new HashSet<>();
        if (sessionExists()) {
            for (final String key : Collections.list(this.session.getAttributeNames())) {
                set.add(new MapEntry<>(key, this.session.getAttribute(key), true));
            }
        }
        return set;
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(final Object o) {
        if (o != this) {
            if (!ServletSessionScopeMap.class.isInstance(o)) {
                return false;
            }
            ServletSessionScopeMap map = (ServletSessionScopeMap) o;
            return isSame(this.session, map.session) && isSame(this.request, map.request);
        }
        return true;
    }

    /**
     * @see java.util.Map#get(java.lang.Object)
     */
    @Override
    public Object get(final Object key) {
        if (sessionExists()) {
            return this.session.getAttribute(toKey(key));
        }
        return null;
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        if (sessionExists()) {
            return this.session.hashCode();
        }
        return 0;
    }

    /**
     * @see java.util.Map#isEmpty()
     */
    @Override
    public boolean isEmpty() {
        return !sessionExists() || !this.session.getAttributeNames().hasMoreElements();
    }

    /**
     * @see java.util.Map#keySet()
     */
    @Override
    public Set<String> keySet() {
        Set<String> set = new HashSet<>();
        if (sessionExists()) {
            set.addAll(Collections.list(this.session.getAttributeNames()));
        }
        return set;
    }

    /**
     * @see java.util.Map#put(java.lang.Object, java.lang.Object)
     */
    @Override
    public Object put(final String key, final Object value) {
        if (value == null) {
            return remove(key);
        }

        // Ensure the Session is created, if it
        // doesn't exist
        if (this.session == null) {
            this.session = this.request.getSession();
            this.request = null;
        }

        Object previous = this.session.getAttribute(key);
        this.session.setAttribute(key, value);
        return previous;
    }

    /**
     * @see java.util.Map#putAll(java.util.Map)
     */
    @Override
    public void putAll(final Map<? extends String, ?> map) {
        for (final Entry<? extends String, ?> entry : map.entrySet()) {
            put(toKey(entry.getKey()), entry.getValue());
        }
    }

    /**
     * @see java.util.Map#remove(java.lang.Object)
     */
    @Override
    public Object remove(final Object key) {
        if (sessionExists()) {
            String skey = toKey(key);
            Object previous = this.session.getAttribute(skey);
            this.session.removeAttribute(skey);
            return previous;
        }
        return null;
    }

    /**
     * @see java.util.Map#size()
     */
    @Override
    public int size() {
        if (sessionExists()) {
            return Collections.list(this.session.getAttributeNames()).size();
        }
        return 0;
    }

    /**
     * @see java.util.Map#values()
     */
    @Override
    public Collection<Object> values() {
        List<Object> list = new ArrayList<>();
        if (sessionExists()) {
            for (final String key : Collections.list(this.session.getAttributeNames())) {
                list.add(this.session.getAttribute(key));
            }
        }
        return list;
    }

    /**
     * to Key String
     * @param key Key
     * @return Key String
     */
    private String toKey(final Object key) {
        if (key == null) {
            throw new IllegalArgumentException();
        }
        return key.toString();
    }

    /**
     * @return session exists or not
     */
    private boolean sessionExists() {
        if (this.session == null) {
            this.session = this.request.getSession(false);
            if (this.session != null) {
                this.request = null;
            }
        }
        return this.session != null;
    }

    /**
     * @param o1 Object1
     * @param o2 Object2
     * @return difference
     */
    private boolean isSame(final Object o1, final Object o2) {
        return o1 == o2 || (o1 != null && o1.equals(o2));
    }

}
