/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.web.servlet;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;

import org.apache.WebMock;
import org.junit.Test;
import org.mockito.Mockito;

/**
 *  Test case for {@link ChainProcessor}.
 */
public final class ChainProcessorTestCase {

    /**
     * Test serialization.
     */
    @Test
    public void testSerialize() {

        // Initialize a ChainProcessor
        ServletContext sctx = WebMock.createServletContext();
        ServletConfig sc = Mockito.mock(ServletConfig.class);
        Mockito.when(sc.getServletName()).thenReturn("test");
        Mockito.when(sc.getServletContext()).thenReturn(sctx);

        ChainProcessor processor = initServlet(new ChainProcessor(), sc);

        // Serialize/Deserialize & test value
        ChainProcessor process = (ChainProcessor)serializeDeserialize(processor);
        assertNotNull(process);

    }

    /**
     * Initialize the ChainProcessor.
     *
     * @param processor The chain processor instance
     * @param config The servlet config to initialize with
     * @return The chain processor
     */
    private ChainProcessor initServlet(final ChainProcessor processor, final ServletConfig config) {
        try {
            processor.init(config);
        } catch (final ServletException t) {
            t.printStackTrace();
            fail("ChainProcessor init() threw " + t);
        }
        return processor;
    }

    /**
     * Do serialization and deserialization.
     *
     * @param target Object
     * @return Object
     */
    private Object serializeDeserialize(final Object target) {

        // Serialize the test object
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (ObjectOutputStream oos = new ObjectOutputStream(baos)) {
            oos.writeObject(target);
            oos.flush();
        } catch (final IOException e) {
            fail("First Test: Exception during serialization: " + e);
        }

        // Deserialize the test object
        Object result = null;
        try (ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray())) {
            try (ObjectInputStream ois = new ObjectInputStream(bais)) {
                result = ois.readObject();
            }
        } catch (final ClassNotFoundException | IOException e) {
            fail("First Test: Exception during deserialization: " + e);
        }
        return result;

    }
}
