/*
 * $Id: TestModuleConfig.java 471754 2006-11-06 14:55:09Z husted $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.config;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.digester3.Digester;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.xml.sax.SAXException;

/**
 * Unit tests for the <code>org.apache.struts.config</code> package.
 *
 * @version $Rev: 471754 $ $Date: 2005-03-01 20:26:14 -0500 (Tue, 01 Mar 2005)
 *          $
 */
public class TestModuleConfig {

    // ----------------------------------------------------- Instance Variables

    /**
     * The ModuleConfig we are testing.
     */
    private ModuleConfig config = null;

    // ----------------------------------------------------------- Constructors

    // --------------------------------------------------------- Public Methods

    /**
     * Set up instance variables required by this test case.
     */
    @Before
    public void setUp() {
        ModuleConfigFactory factoryObject = ModuleConfigFactory.createFactory();

        this.config = factoryObject.createModuleConfig("");
    }

    /**
     * Tear down instance variables required by this test case.
     */
    @After
    public void tearDown() {
        this.config = null;
    }

    // ------------------------------------------------ Individual Test Methods

    /**
     * parseConfig
     * @param publicId String
     * @param entityURL String
     * @param strutsConfig String
     */
    private void parseConfig(final String publicId, final String entityURL,
            final String strutsConfig) {
        // Prepare a Digester for parsing a struts-config.xml file
        Digester digester = new Digester();

        digester.push(this.config);
        digester.setNamespaceAware(true);
        digester.setValidating(true);
        digester.addRuleSet(new ConfigRuleSet());
        digester.register(publicId,
                TestModuleConfig.class.getResource(entityURL).toString());

        // Parse the test struts-config.xml file
        try (InputStream input =
                TestModuleConfig.class.getResourceAsStream(strutsConfig)) {

            Assert.assertNotNull("Got an input stream for " + strutsConfig, input);
            digester.parse(input);
        } catch (final IOException | SAXException e) {
            e.printStackTrace(System.out);
            Assert.fail("Parsing threw exception:  " + e);
        }
    }

    /**
     * Test parsing of a struts-config.xml file.
     */
    @Test
    public void testParse() {
        testParseBase("-//Apache Software Foundation//DTD Struts Configuration 1.3//EN",
            "/org/apache/struts/resources/struts-config_1_3.dtd",
            "/org/apache/struts/config/struts-config.xml");
    }

    /**
     * @param publicId String
     * @param entityURL String
     * @param strutsConfig String
     */
    protected void testParseBase(final String publicId,
            final String entityURL, final String strutsConfig) {
        parseConfig(publicId, entityURL, strutsConfig);

        // Perform assertion tests on the parsed information
        FormBeanConfig[] configs = this.config.findFormBeanConfigs();

        Assert.assertNotNull("Found our form bean configurations", configs);
        Assert.assertEquals("Found three form bean configurations", 3, configs.length);

        ForwardConfig[] fcs = this.config.findForwardConfigs();

        Assert.assertNotNull("Found our forward configurations", fcs);
        Assert.assertEquals("Found three forward configurations", 3, fcs.length);

        ActionConfig logon = this.config.findActionConfig("/logon");

        Assert.assertNotNull("Found logon action configuration", logon);
        Assert.assertEquals("Found correct logon configuration", "logonForm",
            logon.getName());
    }

    /**
     * Tests a struts-config.xml that contains a custom mapping and property.
     */
    @Test
    public void testCustomMappingParse() {
        // Prepare a Digester for parsing a struts-config.xml file
        testCustomMappingParseBase(
                "-//Apache Software Foundation//DTD Struts Configuration 1.3//EN",
                "/org/apache/struts/resources/struts-config_1_3.dtd",
                "/org/apache/struts/config/struts-config-custom-mapping.xml");
    }

    /** Tests a struts-config.xml that contains a custom mapping and property.
     * @param publicId String
     * @param entityURL String
     * @param strutsConfig String */
    private void testCustomMappingParseBase(final String publicId, final String entityURL,
            final String strutsConfig) {
        parseConfig(publicId, entityURL, strutsConfig);

        // Perform assertion tests on the parsed information
        ActionConfig map = this.config.findActionConfig("/editRegistration");

        Assert.assertNotNull("Cannot find editRegistration mapping", map);
        Assert.assertTrue(map instanceof CustomMappingTest);
        Assert.assertTrue("The custom mapping attribute has not been set",
            ((CustomMappingTest) map).getPublic());
    }

    /**
     * Test order of action mappings defined preserved.
     */
    @Test
    public void testPreserveActionMappingsOrder() {
        parseConfig("-//Apache Software Foundation//DTD Struts Configuration 1.3//EN",
            "/org/apache/struts/resources/struts-config_1_3.dtd",
            "/org/apache/struts/config/struts-config.xml");

        String[] paths =
            new String[] {
                "/editRegistration", "/editSubscription", "/logoff", "/logon",
                "/saveRegistration", "/saveSubscription", "/tour"
            };

        ActionConfig[] actions = this.config.findActionConfigs();

        for (int x = 0; x < paths.length; x++) {
            Assert.assertEquals("Action config out of order:" + actions[x].getPath(),
                paths[x], actions[x].getPath());
        }
    }
}
