package project.check.attribute;

import java.util.Optional;
import java.util.regex.Pattern;

import core.util.MojiUtil;
import core.util.bean.Pair;
import online.context.check.ItemCheck;
import online.model.ModelUtil;
import project.check.BaseCheck;

/**
 * 入力文字列が変換可能かチェックする。
 *
 * @author Tadashi Nakayama
 * @version 1.0.0
 */
public final class MojiReferenceCheck extends BaseCheck implements ItemCheck {

	/** 数値文字参照10進 */
	private static final Pattern PATTERN_DEC = Pattern.compile("&#[0-9]{2,5};");
	/** 数値文字参照16進 */
	private static final Pattern PATTERN_HEX = Pattern.compile("&#x[0-9A-Fa-f]{4};");

	/** カナチェック */
	private final boolean k;
	/** 外字チェック */
	private final boolean e;
	/** メッセージID */
	private final String mid;
	/** メッセージパラメータ */
	private final String[] prm;

	/**
	 * コンストラクタ
	 */
	public MojiReferenceCheck() {
		this(true, true);
	}

	/**
	 * コンストラクタ
	 * @param kana 半角カナを含む場合 true
	 * @param ext ユーザ定義領域を含む場合 true
	 */
	public MojiReferenceCheck(final boolean kana, final boolean ext) {
		this.k = kana;
		this.e = ext;
		this.mid = null;
		this.prm = null;
	}

	/**
	 * コンストラクタ
	 *
	 * @param kana 半角カナを含む場合 true
	 * @param ext ユーザ定義領域を含む場合 true
	 * @param msgId メッセージID
	 * @param param メッセージパラメータ
	 */
	public MojiReferenceCheck(final boolean kana, final boolean ext,
					final String msgId, final String... param) {
		this.k = kana;
		this.e = ext;
		this.mid = msgId;
		this.prm = copyOf(param);
	}

	/**
	 * @see online.context.check.ItemCheck#check(java.lang.String...)
	 */
	@Override
	public void check(final String... items) {
		Optional<Pair<String, Integer>> check = Optional.empty();
		for (final var item : items) {
			final var val = getArrayParameter(item);
			for (var j = 0; val != null && j < val.length; j++) {
				if (!isTarget(j)) {
					val[j] = ModelUtil.getValueAsString(getUniModel(), item, j);
				} else if (isMojiReference(val[j])) {
					setMessage(item, j, this.mid, this.prm);
					check = Optional.of(check.orElse(Pair.of(item, j)));
				}
			}
		}

		check.ifPresent(this::throwKeepCheckException);
	}

	/**
	 * 数値文字参照か判断します。
	 * @param val 対象文字列
	 * @return 文字参照が存在する場合 true を返す。
	 */
	private boolean isMojiReference(final String val) {
		return MojiUtil.isShiftJIS(val, this.k, this.e)
				&& (PATTERN_DEC.matcher(val).matches() || PATTERN_HEX.matcher(val).matches());
	}
}
