package project.check.attribute;

import java.util.Date;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import core.util.DateUtil;
import core.util.bean.Pair;
import online.context.check.ItemCheck;
import online.context.check.ItemConvert;
import online.model.ModelUtil;
import project.check.BaseCheck;

/**
 * 時間チェック
 *
 * @author Tadashi Nakayama
 * @version 1.0.0
 */
public final class TimeCheck extends BaseCheck implements ItemCheck, ItemConvert<Date> {

	/** メッセージID */
	private final String mid;
	/** メッセージパラメータ */
	private final String[] prm;

	/**
	 * コンストラクタ
	 */
	public TimeCheck() {
		this.mid = null;
		this.prm = null;
	}

	/**
	 * コンストラクタ
	 *
	 * @param msgId メッセージID
	 * @param param メッセージパラメータ
	 */
	public TimeCheck(final String msgId, final String... param) {
		this.mid = msgId;
		this.prm = copyOf(param);
	}

	/**
	 * @see online.context.check.ItemCheck#check(java.lang.String...)
	 */
	@Override
	public void check(final String... items) {
		Optional<Pair<String, Integer>> check = Optional.empty();
		for (final var item : items) {
			final var val = getArrayParameter(item);
			for (var j = 0; val != null && j < val.length; j++) {
				if (!isTarget(j)) {
					val[j] = ModelUtil.getValueAsString(getUniModel(), item, j);
				} else if (!isTime(val[j])) {
					setMessage(item, j, this.mid, this.prm);
					check = Optional.of(check.orElse(Pair.of(item, j)));
				}
			}
		}

		check.ifPresent(this::throwKeepCheckException);
	}

	/**
	 * 時分確認
	 * @param val 文字列
	 * @return 時分の場合 true を返す。
	 */
	private boolean isTime(final String val) {
		final var str = DateUtil.removeSeparator(val);
		return Objects.toString(str, "").isEmpty()
				|| (str.length() % 2 == 0 && str.length() <= 6 && DateUtil.toTime(str) != null);
	}

	/**
	 * @see online.context.check.ItemConvert#convert(java.lang.String...)
	 */
	@Override
	public Date[] convert(final String... val) {
		return Stream.of(val).map(DateUtil::toTime).toArray(Date[]::new);
	}
}
