package common.db.dao.hibernate;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;

/**
 * EntityUtilTest
 *
 */
public class EntityUtilTest {

	/**
	 * エンティティから挿入SQLとパラメータリスト変換
	 */
	@Test
	@DisplayName("エンティティから挿入SQLとパラメータリスト変換")
	public void toInsertSqlTest() {
		final FooMaster fm = new FooMaster();
		fm.setCreateDateTime(new Timestamp(new Date().getTime()));
		fm.setCreateUserId("abcd");
		fm.setFooAddress("fooAddress");
		fm.setFooId("fooId");
		fm.setFooName("fooName");
		fm.setFooPhone("fooPhone");
		fm.setHooId("hooId");
		fm.setId(1L);
		fm.setTorokuDateTime(fm.getCreateDateTime());
		fm.setUpdateDateTime(fm.getTorokuDateTime());
		fm.setUpdateUserId("updateUserId");
		fm.setVersion(1);

		final List<Object> param = new ArrayList<>();
		final String sql = EntityUtil.toInsertSql(fm, param);

		Assertions.assertEquals(param.size(), 12);
		Assertions.assertEquals(sql, "INSERT INTO foo_master(create_date_time, create_user_id, "
				+ "foo_address, foo_id, foo_name, foo_phone, hoo_id, id, toroku_date_time, "
				+ "update_date_time, update_user_id, version) "
				+ "VALUES(:1, :2, :3, :4, :5, :6, :7, :8, :9, :10, :11, :12)");
	}

	/**
	 * エンティティから更新SQLとパラメータリストに変換
	 */
	@Test
	@DisplayName("エンティティから更新SQLとパラメータリストに変換")
	public void toUpdateSqlTest() {
		final FooMaster fm = new FooMaster();
		fm.setCreateDateTime(new Timestamp(new Date().getTime()));
		fm.setCreateUserId("abcd");
		fm.setFooAddress("fooAddress");
		fm.setFooId("fooId");
		fm.setFooName("fooName");
		fm.setFooPhone("fooPhone");
		fm.setHooId("hooId");
		fm.setId(1L);
		fm.setTorokuDateTime(fm.getCreateDateTime());
		fm.setUpdateDateTime(fm.getTorokuDateTime());
		fm.setUpdateUserId("updateUserId");
		fm.setVersion(1);

		final List<Object> param = new ArrayList<>();
		final String sql = EntityUtil.toUpdateSql(fm, param);

		Assertions.assertEquals(param.size(), 12);
		Assertions.assertEquals(sql,
				"UPDATE foo_master SET create_date_time = :1, create_user_id = :2,"
				+ " foo_address = :3, foo_id = :4, foo_name = :5, foo_phone = :6, hoo_id = :7, "
				+ "toroku_date_time = :8, update_date_time = :9, update_user_id = :10, "
				+ "version = :11 WHERE id = :12");
	}

	/**
	 * エンティティから削除SQLとパラメータリストに変換
	 */
	@Test
	@DisplayName("エンティティから削除SQLとパラメータリストに変換")
	public void toDeleteSqlTest() {
		final FooMaster fm = new FooMaster();
		fm.setCreateDateTime(new Timestamp(new Date().getTime()));
		fm.setCreateUserId("abcd");
		fm.setFooAddress("fooAddress");
		fm.setFooId("fooId");
		fm.setFooName("fooName");
		fm.setFooPhone("fooPhone");
		fm.setHooId("hooId");
		fm.setId(1L);
		fm.setTorokuDateTime(fm.getCreateDateTime());
		fm.setUpdateDateTime(fm.getTorokuDateTime());
		fm.setUpdateUserId("updateUserId");
		fm.setVersion(1);

		final List<Object> param = new ArrayList<>();
		final String sql = EntityUtil.toDeleteSql(fm, param);

		Assertions.assertEquals(sql, "DELETE FROM foo_master WHERE id = :1");
		Assertions.assertEquals(param.size(), 1);
	}

	/**
	 * マップからエンティティ変換
	 */
	@Test
	@DisplayName("マップからエンティティ変換")
	public void toEntityTest() {
		final Timestamp now = new Timestamp(new Date().getTime());
		final Map<String, Object> map = new HashMap<>();
		map.put("create_user_id", "abcd");
		map.put("id", 1L);
		map.put("create_date_time", now);

		final FooMaster fm = EntityUtil.toInstance(map, FooMaster.class);
		Assertions.assertEquals("abcd", fm.getCreateUserId());
		Assertions.assertEquals(Long.valueOf(1), fm.getId());
		Assertions.assertEquals(now, fm.getCreateDateTime());
	}
}
