/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.seasar.extension.dbcp.impl;

import java.sql.Connection;
import java.sql.SQLException;

import javax.transaction.xa.XAException;
import javax.transaction.xa.Xid;

import org.seasar.extension.dbcp.DbXAResource;
import org.seasar.extension.jta.xa.DefaultXAResource;

/**
 * {@link DbXAResource}の実装クラスです。
 *
 * @author higa
 *
 */
public class DbXAResourceImpl extends DefaultXAResource implements DbXAResource {

    /** コネクション */
    private final Connection connection;

    /**
     * {@link DbXAResourceImpl}を作成します。
     *
     * @param conn コネクション
     */
    public DbXAResourceImpl(final Connection conn) {
        this.connection = conn;
    }

    /**
     * @see org.seasar.extension.dbcp.DbXAResource#getConnection()
     */
    @Override
    public Connection getConnection() {
        return this.connection;
    }

    /**
     * @see org.seasar.extension.jta.xa.DefaultXAResource#doBegin(javax.transaction.xa.Xid)
     */
    @Override
    protected void doBegin(final Xid xid) throws XAException {
        try {
            if (this.connection.getAutoCommit()) {
                this.connection.setAutoCommit(false);
            }
        } catch (final SQLException ex) {
            throw new XAException(ex.getMessage());
        }
    }

    /**
     * @see org.seasar.extension.jta.xa.DefaultXAResource
     * #doCommit(javax.transaction.xa.Xid, boolean)
     */
    @Override
    protected void doCommit(final Xid xid, final boolean onePhase) throws XAException {
        try {
            this.connection.commit();
            this.connection.setAutoCommit(true);
        } catch (final SQLException ex) {
            throw new XAException(ex.getMessage());
        }
    }

    /**
     * @see org.seasar.extension.jta.xa.DefaultXAResource#doPrepare(javax.transaction.xa.Xid)
     */
    @Override
    protected int doPrepare(final Xid xid) throws XAException {
        try {
            if (this.connection.isClosed()) {
                return XA_RDONLY;
            }
            return XA_OK;
        } catch (final SQLException ex) {
            throw new XAException(ex.getMessage());
        }
    }

    /**
     * @see org.seasar.extension.jta.xa.DefaultXAResource#doRollback(javax.transaction.xa.Xid)
     */
    @Override
    protected void doRollback(final Xid xid) throws XAException {
        try {
            this.connection.rollback();
            this.connection.setAutoCommit(true);
        } catch (final SQLException ex) {
            throw new XAException(ex.getMessage());
        }
    }
}
