/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.seasar.extension.dbcp.impl;

import java.sql.SQLException;

import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

import org.seasar.extension.dbcp.ConnectionWrapper;

/**
 * {@link XAResource}のラッパーです。
 *
 * @author higa
 *
 */
public class XAResourceWrapperImpl implements XAResource {

    /** 物理的なXAリソース */
    private final XAResource physicalXAResource;
    /** コネクション */
    private final ConnectionWrapper connectionWrapper;

    /**
     * {@link XAResourceWrapperImpl}を作成します。
     *
     * @param pxa 物理的なXAリソース
     * @param conn コネクション
     */
    public XAResourceWrapperImpl(final XAResource pxa, final ConnectionWrapper conn) {
        this.physicalXAResource = pxa;
        this.connectionWrapper = conn;
    }

    /**
     * 物理的なXAリソースを返します。
     *
     * @return 物理的なXAリソース
     */
    public XAResource getPhysicalXAResource() {
        return this.physicalXAResource;
    }

    /**
     * コネクションを返します。
     *
     * @return コネクション
     */
    public ConnectionWrapper getConnectionWrapper() {
        return this.connectionWrapper;
    }

    /**
     * release connection
     *
     * @throws XAException XAException
     */
    private void release() throws XAException {
        try {
            getConnectionWrapper().release();
        } catch (final SQLException ex) {
            throw new XAException(ex.getMessage());
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#getTransactionTimeout()
     */
    @Override
    public int getTransactionTimeout() throws XAException {
        try {
            return getPhysicalXAResource().getTransactionTimeout();
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#setTransactionTimeout(int)
     */
    @Override
    public boolean setTransactionTimeout(final int arg0) throws XAException {
        try {
            return getPhysicalXAResource().setTransactionTimeout(arg0);
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#isSameRM(javax.transaction.xa.XAResource)
     */
    @Override
    public boolean isSameRM(final XAResource arg0) throws XAException {
        try {
            return getPhysicalXAResource().isSameRM(arg0);
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#recover(int)
     */
    @Override
    public Xid[] recover(final int arg0) throws XAException {
        try {
            return getPhysicalXAResource().recover(arg0);
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#prepare(javax.transaction.xa.Xid)
     */
    @Override
    public int prepare(final Xid arg0) throws XAException {
        try {
            return getPhysicalXAResource().prepare(arg0);
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#forget(javax.transaction.xa.Xid)
     */
    @Override
    public void forget(final Xid arg0) throws XAException {
        try {
            getPhysicalXAResource().forget(arg0);
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#rollback(javax.transaction.xa.Xid)
     */
    @Override
    public void rollback(final Xid arg0) throws XAException {
        try {
            getPhysicalXAResource().rollback(arg0);
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#end(javax.transaction.xa.Xid, int)
     */
    @Override
    public void end(final Xid arg0, final int arg1) throws XAException {
        try {
            getPhysicalXAResource().end(arg0, arg1);
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#start(javax.transaction.xa.Xid, int)
     */
    @Override
    public void start(final Xid arg0, final int arg1) throws XAException {
        try {
            getPhysicalXAResource().start(arg0, arg1);
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }

    /**
     * @see javax.transaction.xa.XAResource#commit(javax.transaction.xa.Xid, boolean)
     */
    @Override
    public void commit(final Xid arg0, final boolean arg1) throws XAException {
        try {
            getPhysicalXAResource().commit(arg0, arg1);
        } catch (final XAException ex) {
            release();
            throw ex;
        }
    }
}
