/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.seasar.extension.timer;

/**
 * タイムアウトを管理するクラスです。
 *
 * @author higa
 *
 */
public class TimeoutTask {

    /** ステータス */
    private enum Status {
        /** 有効 */
        ACTIVE,
        /** 停止 */
        STOPPED,
        /** キャンセル */
        CANCELED
    }

    /** TimeoutTarget */
    private final TimeoutTarget timeoutTarget;
    /** timeoutMillis */
    private final long timeoutMillis;
    /** permanent */
    private final boolean permanent;

    /** startTime */
    private long startTime;
    /** Status */
    private Status status = Status.ACTIVE;

    /**
     * Constructor
     * @param target TimeoutTarget
     * @param timeout タイムアウト
     * @param permanentFlg フラグ
     */
    TimeoutTask(final TimeoutTarget target, final int timeout, final boolean permanentFlg) {
        this.timeoutTarget = target;
        this.timeoutMillis = timeout * 1000L;
        this.permanent = permanentFlg;
        this.startTime = System.currentTimeMillis();
    }

    /**
     * 期限切れかどうかを返します。
     *
     * @return 期限切れかどうか
     */
    public boolean isExpired() {
        return System.currentTimeMillis() >= this.startTime + this.timeoutMillis;
    }

    /**
     * 永続的かどうかを返します。
     *
     * @return 永続的かどうか
     */
    public boolean isPermanent() {
        return this.permanent;
    }

    /**
     * キャンセルされているかどうかを返します。
     *
     * @return キャンセルされているか
     */
    public boolean isCanceled() {
        return Status.CANCELED.equals(this.status);
    }

    /**
     * キャンセルします。
     */
    public void cancel() {
        this.status = Status.CANCELED;
    }

    /**
     * 止まっているかどうか返します。
     *
     * @return 止まっているかどうか
     */
    public boolean isStopped() {
        return Status.STOPPED.equals(this.status);
    }

    /**
     * タイマーをとめます。
     */
    public void stop() {
        if (!Status.ACTIVE.equals(this.status)) {
            throw new IllegalStateException(this.status.name());
        }
        this.status = Status.STOPPED;
    }

    /**
     * タイマーを再開始します。
     */
    public void restart() {
        this.status = Status.ACTIVE;
        this.startTime = System.currentTimeMillis();
    }

    /**
     * expired
     */
    public void expired() {
        this.timeoutTarget.expired();
    }
}
