/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.seasar.extension.jta;

import javax.transaction.Status;
import javax.transaction.SystemException;

/**
 * {@link javax.transaction.TransactionManager}の実装クラスです。
 *
 * @author higa
 */
public class TransactionManagerImpl implements ExtendedTransactionManager {

    /** スレッドに関連づけられているトランザクション */
    private static final ThreadLocal<ExtendedTransaction> THREAD_ATTACH_TX = new ThreadLocal<>();

    /**
     * <code>TransactionManagerImpl</code>のインスタンスを構築します。
     */
    public TransactionManagerImpl() {
        super();
    }

    /**
     * 現在のスレッドに関連づけられているトランザクションを返します。
     *
     * @return 現在のスレッドに関連づけられているトランザクション
     */
    @Override
    public ExtendedTransaction getCurrent() {
        final ExtendedTransaction tx = THREAD_ATTACH_TX.get();
        try {
            if (tx != null && tx.getStatus() == Status.STATUS_NO_TRANSACTION) {
                setCurrent(null);
                return null;
            }
            return tx;
        } catch (final SystemException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * トランザクションを現在のスレッドに関連づけます。
     *
     * @param current 現在のスレッドに関連づけるトランザクション
     */
    @Override
    public void setCurrent(final ExtendedTransaction current) {
        THREAD_ATTACH_TX.set(current);
    }

    /**
     * 新しいトランザクションを作成して現在のスレッドに関連づけます。
     *
     * @return 現在のスレッドに関連づけられたトランザクション
     */
    @Override
    public ExtendedTransaction attachNewTransaction() {
        ExtendedTransaction tx = THREAD_ATTACH_TX.get();
        if (tx == null) {
            tx = createTransaction();
            setCurrent(tx);
        }
        return tx;
    }

    /**
     * トランザクションを作成して返します。
     *
     * @return トランザクション
     */
    @Override
    public ExtendedTransaction createTransaction() {
        return new TransactionImpl();
    }
}
