/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.seasar.extension.jta;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.transaction.RollbackException;
import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import javax.transaction.TransactionSynchronizationRegistry;

import org.apache.logging.log4j.LogManager;

/**
 * {@link TransactionSynchronizationRegistry}の実装クラスです。
 * <p>
 * J2EE1.4準拠のAPサーバが提供するJTA実装など、 S2JTA以外のJTA実装と組み合わせて使用することもできます。 その場合、
 * {@link TransactionSynchronizationRegistry#registerInterposedSynchronization(Synchronization)}
 * が規定する{@link Synchronization}呼び出しの順序は満たされません。
 * </p>
 *
 * @author nakamura
 */
public class TransactionSynchronizationRegistryImpl implements TransactionSynchronizationRegistry {

    /** transactionContexts */
    private final Map<Transaction, SynchronizationRegister> transactionContexts =
            Collections.synchronizedMap(new HashMap<>());

    /** TransactionManager */
    private TransactionManager manager;

    /**
     * インスタンスを構築します。
     */
    public TransactionSynchronizationRegistryImpl() {
        super();
    }

    /**
     * インスタンスを構築します。
     *
     * @param tm トランザクションマネージャ
     */
    public TransactionSynchronizationRegistryImpl(final TransactionManager tm) {
        this.manager = tm;
    }

    /**
     * トランザクションマネージャを設定します。
     *
     * @param tm トランザクションマネージャ
     */
    public void setTransactionManager(final TransactionManager tm) {
        this.manager = tm;
    }

    /**
     * @see javax.transaction.TransactionSynchronizationRegistry
     * #putResource(java.lang.Object, java.lang.Object)
     */
    @Override
    public void putResource(final Object key, final Object value) {
        if (key == null) {
            throw new NullPointerException("key");
        }
        assertActive();
        getContext().putResource(key, value);
    }

    /**
     * @see javax.transaction.TransactionSynchronizationRegistry#getResource(java.lang.Object)
     */
    @Override
    public Object getResource(final Object key) {
        if (key == null) {
            throw new NullPointerException("key");
        }
        assertActive();
        return getContext().getResource(key);
    }

    /**
     * @see javax.transaction.TransactionSynchronizationRegistry#setRollbackOnly()
     */
    @Override
    public void setRollbackOnly() {
        assertActive();
        setRollbackOnly(this.manager);
    }

    /**
     * @see javax.transaction.TransactionSynchronizationRegistry#getRollbackOnly()
     */
    @Override
    public boolean getRollbackOnly() {
        assertActive();
        final int status = getTransactionStatus();
        return Status.STATUS_MARKED_ROLLBACK == status || Status.STATUS_ROLLING_BACK == status;
    }

    /**
     * @see javax.transaction.TransactionSynchronizationRegistry#getTransactionKey()
     */
    @Override
    public Object getTransactionKey() {
        if (!isActive()) {
            return null;
        }
        return getTransaction();
    }

    /**
     * @see javax.transaction.TransactionSynchronizationRegistry#getTransactionStatus()
     */
    @Override
    public int getTransactionStatus() {
        return getStatus(this.manager);
    }

    /**
     * @see javax.transaction.TransactionSynchronizationRegistry
     * #registerInterposedSynchronization(javax.transaction.Synchronization)
     */
    @Override
    public void registerInterposedSynchronization(final Synchronization sync) {
        assertActive();
        getContext().registerInterposedSynchronization(sync);
    }

    /**
     * トランザクションを返します。
     *
     * @return トランザクション
     */
    protected Transaction getTransaction() {
        return getTransaction(this.manager);
    }

    /**
     * トランザクションがアクティブであることを表明します。
     *
     * @throws IllegalStateException アクティブでない場合
     */
    protected void assertActive() {
        if (!isActive()) {
            throw new IllegalStateException("transaction is not active.");
        }
    }

    /**
     * トランザクションがアクティブかどうかを返します。
     *
     * @return トランザクションがアクティブかどうか
     */
    protected boolean isActive() {
        return isActive(this.manager);
    }

    /**
     * 現在のトランザクションに関連づけられた{@link SynchronizationRegisterImpl コンテキスト情報}を返します。
     *
     * @return 現在のトランザクションに関連づけられた{@link SynchronizationRegisterImpl コンテキスト情報}
     */
    protected SynchronizationRegister getContext() {
        final Transaction tx = getTransaction();
        if (tx instanceof SynchronizationRegister) {
            return (SynchronizationRegister) tx;
        }
        SynchronizationRegisterImpl context = (SynchronizationRegisterImpl)
                this.transactionContexts.get(tx);
        if (context == null) {
            context = new SynchronizationRegisterImpl(tx, this.transactionContexts);
            registerSynchronization(tx, context);
            this.transactionContexts.put(tx, context);
        }
        return context;
    }

    /**
     * {@link Synchronization}を登録します。
     *
     * @param tx Transaction
     * @param sync Synchronization
     */
    private static void registerSynchronization(final Transaction tx, final Synchronization sync) {
        try {
            tx.registerSynchronization(sync);
        } catch (final SystemException | RollbackException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * トランザクションを返します。
     *
     * @param tm トランザクションマネージャ
     * @return トランザクション
     */
    private static Transaction getTransaction(final TransactionManager tm) {
        try {
            return tm.getTransaction();
        } catch (final SystemException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * トランザクションがアクティブかどうか返します。
     *
     * @param tm トランザクションマネージャ
     * @return トランザクションがアクティブかどうか
     */
    private static boolean isActive(final TransactionManager tm) {
        return getStatus(tm) != Status.STATUS_NO_TRANSACTION;
    }

    /**
     * ステータスを返します。
     *
     * @param tm トランザクションマネージャ
     * @return ステータス
     */
    private static int getStatus(final TransactionManager tm) {
        try {
            return tm.getStatus();
        } catch (final SystemException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * ロールバックオンリーに設定します。
     *
     * @param tm トランザクションマネージャ
     */
    private static void setRollbackOnly(final TransactionManager tm) {
        try {
            tm.setRollbackOnly();
        } catch (final SystemException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * トランザクションに関連づけられたコンテキスト情報を表すクラスです。
     *
     * @author koichik
     */
    private static final class SynchronizationRegisterImpl
            implements SynchronizationRegister, Synchronization {

        /** interposedSynchronizations */
        private final List<Synchronization> interposedSynchronizations = new ArrayList<>();
        /** resourceMap */
        private final Map<Object, Object> resourceMap = new HashMap<>();

        /** transactionContexts */
        private final Map<Transaction, SynchronizationRegister> contexts;
        /** Transaction */
        private final Transaction transaction;

        /**
         * インスタンスを構築します。
         *
         * @param tx トランザクション
         * @param ctx transactionContexts
         */
        SynchronizationRegisterImpl(final Transaction tx,
                final Map<Transaction, SynchronizationRegister> ctx) {
            this.transaction = tx;
            this.contexts = ctx;
        }

        /**
         * @see org.seasar.extension.jta.SynchronizationRegister
         * #registerInterposedSynchronization(javax.transaction.Synchronization)
         */
        @Override
        public void registerInterposedSynchronization(final Synchronization sync) {
            this.interposedSynchronizations.add(sync);
        }

        /**
         * @see org.seasar.extension.jta.SynchronizationRegister
         * #putResource(java.lang.Object, java.lang.Object)
         */
        @Override
        public void putResource(final Object key, final Object value) {
            this.resourceMap.put(key, value);
        }

        /**
         * @see org.seasar.extension.jta.SynchronizationRegister#getResource(java.lang.Object)
         */
        @Override
        public Object getResource(final Object key) {
            return this.resourceMap.get(key);
        }

        /**
         * @see javax.transaction.Synchronization#beforeCompletion()
         */
        @Override
        public void beforeCompletion() {
            for (final Synchronization sync : this.interposedSynchronizations) {
                sync.beforeCompletion();
            }
        }

        /**
         * @see javax.transaction.Synchronization#afterCompletion(int)
         */
        @Override
        public void afterCompletion(final int status) {
            for (final Synchronization sync : this.interposedSynchronizations) {
                try {
                    sync.afterCompletion(status);
                } catch (final Throwable t) {
                    LogManager.getLogger().info(t);
                }
            }
            this.contexts.remove(this.transaction);
        }
    }
}
