/*
 * $Id: ActionMessages.java 471754 2006-11-06 14:55:09Z husted $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.action;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * <p>A class that encapsulates messages. Messages can be either global or
 * they are specific to a particular bean property.</p>
 *
 * <p>Each individual message is described by an <code>ActionMessage</code>
 * object, which contains a message key (to be looked up in an appropriate
 * message resources database), and up to four placeholder arguments used for
 * parametric substitution in the resulting message.</p>
 *
 * <p><strong>IMPLEMENTATION NOTE</strong> - It is assumed that these objects
 * are created and manipulated only within the context of a single thread.
 * Therefore, no synchronization is required for access to internal
 * collections.</p>
 *
 * @version $Rev: 471754 $ $Date: 2005-08-26 21:58:39 -0400 (Fri, 26 Aug 2005)
 *          $
 * @since Struts 1.1
 */
public class ActionMessages implements Serializable {

    // ----------------------------------------------------- Manifest Constants

    /**
     * <p>The "property name" marker to use for global messages, as opposed to
     * those related to a specific property.</p>
     */
    public static final String GLOBAL_MESSAGE =
        "org.apache.struts.action.GLOBAL_MESSAGE";

    /** serialVersionUID */
    private static final long serialVersionUID = -4363118850236216381L;

    /**
     * <p>Compares ActionMessageItem objects.</p>
     */
    private static final Comparator<ActionMessageItem> ACTION_ITEM_COMPARATOR =
        new Comparator<ActionMessageItem>() {
            @Override
            public int compare(final ActionMessageItem o1, final ActionMessageItem o2) {
                return o1.getOrder() - o2.getOrder();

            }
        };

    // ----------------------------------------------------- Instance Variables

    /**
     * <p>Have the messages been retrieved from this object?</p>
     *
     * <p>The controller uses this property to determine if session-scoped
     * messages can be removed.</p>
     *
     * @since Struts 1.2
     */
    private boolean accessed = false;

    /**
     * <p>The accumulated set of <code>ActionMessage</code> objects
     * (represented as an ArrayList) for each property, keyed by property
     * name.</p>
     */
    private final Map<String, ActionMessageItem> messages = new HashMap<>();

    // --------------------------------------------------------- Public Methods

    /**
     * <p>Create an empty <code>ActionMessages</code> object.</p>
     */
    public ActionMessages() {
        super();
    }

    /**
     * <p>Create an <code>ActionMessages</code> object initialized with the
     * given messages.</p>
     *
     * @param val The messages to be initially added to this object. This
     *                 parameter can be <code>null</code>.
     * @since Struts 1.1
     */
    public ActionMessages(final ActionMessages val) {
        super();
        add(val);
    }

    /**
     * <p>Add a message to the set of messages for the specified property. An
     * order of the property/key is maintained based on the initial addition
     * of the property/key.</p>
     *
     * @param property Property name (or ActionMessages.GLOBAL_MESSAGE)
     * @param message  The message to be added
     */
    public void add(final String property, final ActionMessage message) {
        ActionMessageItem item = this.messages.get(property);
        if (item == null) {
            this.messages.put(property, new ActionMessageItem(
                    Arrays.asList(message), this.messages.size(), property));
        } else {
            item.getList().add(message);
        }
    }

    /**
     * <p>Adds the messages from the given <code>ActionMessages</code> object
     * to this set of messages. The messages are added in the order they are
     * returned from the <code>properties</code> method. If a message's
     * property is already in the current <code>ActionMessages</code> object,
     * it is added to the end of the list for that property. If a message's
     * property is not in the current list it is added to the end of the
     * properties.</p>
     *
     * @param actionMessages The <code>ActionMessages</code> object to be
     *                       added. This parameter can be <code>null</code>.
     * @since Struts 1.1
     */
    public void add(final ActionMessages actionMessages) {
        if (actionMessages != null) {
            // loop over properties
            for (final String property : new IteratorWrap<>(actionMessages.properties())) {
                // loop over messages for each property
                for (final ActionMessage msg : new IteratorWrap<>(actionMessages.get(property))) {
                    add(property, msg);
                }
            }
        }
    }

    /**
     * <p>Clear all messages recorded by this object.</p>
     */
    public void clear() {
        this.messages.clear();
    }

    /**
     * <p>Return <code>true</code> if there are no messages recorded in this
     * collection, or <code>false</code> otherwise.</p>
     *
     * @return <code>true</code> if there are no messages recorded in this
     *         collection; <code>false</code> otherwise.
     * @since Struts 1.1
     */
    public boolean isEmpty() {
        return this.messages.isEmpty();
    }

    /**
     * <p>Return the set of all recorded messages, without distinction by
     * which property the messages are associated with. If there are no
     * messages recorded, an empty enumeration is returned.</p>
     *
     * @return An iterator over the messages for all properties.
     */
    public Iterator<ActionMessage> get() {
        this.accessed = true;

        if (this.messages.isEmpty()) {
            return Collections.EMPTY_LIST.iterator();
        }

        ArrayList<ActionMessage> results = new ArrayList<>();

        ArrayList<ActionMessageItem> actionItems = new ArrayList<>(this.messages.values());

        // Sort ActionMessageItems based on the initial order the
        // property/key was added to ActionMessages.
        Collections.sort(actionItems, ACTION_ITEM_COMPARATOR);

        for (final ActionMessageItem ami : actionItems) {
            for (final ActionMessage msg : ami.getList()) {
                results.add(msg);
            }
        }

        return results.iterator();
    }

    /**
     * <p>Return the set of messages related to a specific property. If there
     * are no such messages, an empty enumeration is returned.</p>
     *
     * @param property Property name (or ActionMessages.GLOBAL_MESSAGE)
     * @return An iterator over the messages for the specified property.
     */
    public Iterator<ActionMessage> get(final String property) {
        this.accessed = true;

        ActionMessageItem ami = this.messages.get(property);
        if (ami == null) {
            return Collections.EMPTY_LIST.iterator();
        }
        return ami.getList().iterator();
    }

    /**
     * <p>Returns <code>true</code> if the <code>get()</code> or
     * <code>get(String)</code> methods are called.</p>
     *
     * @return <code>true</code> if the messages have been accessed one or
     *         more times.
     * @since Struts 1.2
     */
    public boolean isAccessed() {
        return this.accessed;
    }

    /**
     * <p>Return the set of property names for which at least one message has
     * been recorded. If there are no messages, an empty <code>Iterator</code>
     * is returned. If you have recorded global messages, the
     * <code>String</code> value of <code>ActionMessages.GLOBAL_MESSAGE</code>
     * will be one of the returned property names.</p>
     *
     * @return An iterator over the property names for which messages exist.
     */
    public Iterator<String> properties() {

        ArrayList<String> results = new ArrayList<>();

        ArrayList<ActionMessageItem> actionItems = new ArrayList<>(this.messages.values());

        // Sort ActionMessageItems based on the initial order the
        // property/key was added to ActionMessages.
        Collections.sort(actionItems, ACTION_ITEM_COMPARATOR);

        for (final ActionMessageItem ami : actionItems) {
            results.add(ami.getProperty());
        }

        return results.iterator();
    }

    /**
     * <p>Return the number of messages recorded for all properties (including
     * global messages). <strong>NOTE</strong> - it is more efficient to call
     * <code>isEmpty</code> if all you care about is whether or not there are
     * any messages at all.</p>
     *
     * @return The number of messages associated with all properties.
     */
    public int size() {
        int total = 0;

        for (final ActionMessageItem ami : this.messages.values()) {
            total += ami.getList().size();
        }

        return total;
    }

    /**
     * <p>Return the number of messages associated with the specified
     * property. </p>
     *
     * @param property Property name (or ActionMessages.GLOBAL_MESSAGE)
     * @return The number of messages associated with the property.
     */
    public int size(final String property) {
        ActionMessageItem ami = this.messages.get(property);

        return (ami == null) ? 0 : ami.getList().size();
    }

    /**
     * <p>Returns a String representation of this ActionMessages' property
     * name=message list mapping.</p>
     *
     * @return String representation of the messages
     * @see Object#toString()
     */
    @Override
    public String toString() {
        return this.messages.toString();
    }

    /**
     * <p>This class is used to store a set of messages associated with a
     * property/key and the position it was initially added to list.</p>
     */
    protected static class ActionMessageItem implements Serializable {

        /** serialVersionUID */
        private static final long serialVersionUID = 6569527708054508739L;

        /** <p>The list of <code>ActionMessage</code>s.</p> */
        private final List<ActionMessage> list;

        /**
         * <p>The position in the list of messages.</p>
         */
        private final int iOrder;

        /**
         * <p>The property associated with <code>ActionMessage</code>.</p>
         */
        private final String property;

        /**
         * <p>Construct an instance of this class.</p>
         *
         * @param msg     The list of ActionMessages.
         * @param order   The position in the list of messages.
         * @param prop The property associated with ActionMessage.
         */
        public ActionMessageItem(final List<ActionMessage> msg,
                final int order, final String prop) {
            this.list = new ArrayList<>(msg);
            this.iOrder = order;
            this.property = prop;
        }

        /**
         * <p>Retrieve the list of messages associated with this item.</p>
         *
         * @return The list of messages associated with this item.
         */
        public List<ActionMessage> getList() {
            return this.list;
        }

        /**
         * <p>Retrieve the position in the message list.</p>
         *
         * @return The position in the message list.
         */
        public int getOrder() {
            return this.iOrder;
        }

        /**
         * <p>Retrieve the property associated with this item.</p>
         *
         * @return The property associated with this item.
         */
        public String getProperty() {
            return this.property;
        }

        /**
         * <p>Construct a string representation of this object.</p>
         *
         * @return A string representation of this object.
         */
        @Override
        public String toString() {
            return this.list.toString();
        }
    }

    private static final class IteratorWrap<T> implements Iterable<T> {
        private final Iterator<T> iterator;

        IteratorWrap(final Iterator<T> it) {
            this.iterator = it;
        }

        @Override
        public Iterator<T> iterator() {
            return this.iterator;
        }
    }
}
