/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.web.servlet;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import java.util.Locale;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.chain2.Processing;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;


/**
 * Test case for org.apache.commons.chain2.web.servlet.ServletSetLocaleCommand
 */
public final class ServletSetLocaleCommandTestCase {


    // ----------------------------------------------------- Instance Variables


    private Locale locale = null;

    // Servlet API Objects
    private ServletContext scontext = null;
    private HttpServletRequest request = null;
    private HttpServletResponse response = null;
    private HttpSession session = null;

    // Chain API Objects
    private ServletWebContextBase context = null;
    private ServletSetLocaleCommand command = null;


    // -------------------------------------------------- Overall Test Methods


    /**
     * Set up instance variables required by this test case.
     */
    @Before
    public void setUp() {

        this.locale = new Locale("en", "US");

        // Set up Servlet API Objects
        this.scontext = new MockServletContext();
        this.session = new MockHttpSession(this.scontext);
        this.request = new MockHttpServletRequest("/context", "/servlet",
                                             "/path/info", "a=b&c=d",
                                             this.session);
        this.response = new MockHttpServletResponse();

        // Set up Chain API Objects
        this.context = new ServletWebContextBase(this.scontext, this.request, this.response);
        this.command = new ServletSetLocaleCommand();

    }


    /**
     * Tear down instance variables required by this test case.
     */
    @After
    public void tearDown() {

        this.scontext = null;
        this.session = null;
        this.request = null;
        this.response = null;

        this.context = null;
        this.command = null;

    }


    // ------------------------------------------------- Individual Test Methods


    /**
     * Test configured behavior
     */
    @Test
    public void testConfigured() {

        this.command.setLocaleKey("special");
        assertEquals("special", this.command.getLocaleKey());
        check(this.context, this.command);

    }


    /**
     * Test default behavior
     */
    @Test
    public void testDefault() {

        assertEquals("locale", this.command.getLocaleKey());
        check(this.context, this.command);

    }


    // --------------------------------------------------------- Support Methods


    protected void check(final ServletWebContextBase ctx, final ServletSetLocaleCommand cmd) {

        String localeKey = cmd.getLocaleKey();
        assertNotNull(localeKey);
        Object value = ctx.get(localeKey);
        assertNull(value);
        ctx.put(localeKey, this.locale);
        assertNotNull(ctx.get(localeKey));
        assertNull(this.response.getLocale());
        Processing result = cmd.execute(ctx);
        assertEquals(Processing.CONTINUE, result);
        assertNotNull(this.response.getLocale());
        assertEquals(this.locale, this.response.getLocale());
    }

}
