/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.calllogplugin;

import com.sonyericsson.eventstream.calllogplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.calllogplugin.PluginConstants.EventStream;
import com.sonyericsson.eventstream.calllogplugin.PluginConstants.ServiceIntentCmd;
import com.sonyericsson.eventstream.calllogplugin.EventStreamAdapter;


import android.app.Service;
import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.ContentObserver;
import android.database.Cursor;
import android.database.SQLException;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.IBinder;
import android.provider.CallLog;
import android.provider.ContactsContract;
import android.provider.CallLog.Calls;
import android.provider.ContactsContract.Contacts;
import android.provider.ContactsContract.PhoneLookup;
import android.util.Log;

import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

/**
 * @auth Martin Kretz <martin.kretz@sonyericsson.com>
 */
public class CallLogPluginService extends Service {

    private CallLogContentObserver mCallLogContentObserver = null;

    private ContactsContentObserver mContactsContentObserver = null;

    private HandlerThread mHandlerThread;

    private Handler mHandler;

    private int INVALID_SOURCE_ID = -1;

    private static final int BULK_INSERT_MAX_COUNT = 50;

    private static final int BULK_INSERT_DELAY = 20; //ms

    @Override
    public void onCreate() {
        super.onCreate();
        mHandlerThread = new HandlerThread("CallLogPluginHandler");
        mHandlerThread.start();
        mHandler = new Handler(mHandlerThread.getLooper());
    }

    private long getSourceId() {
        Context context = getApplicationContext();

        if (context != null) {
            return EventStreamAdapter.getSourceId(context);
        } else {
            return INVALID_SOURCE_ID;
        }
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        int result = super.onStartCommand(intent, flags, startId);

        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Service started with intent: " + intent.toString());
        }
        if(null == intent.getExtras()) {//guard against null extra
            return result;
        }
        String serviceCommand = intent.getExtras().getString(ServiceIntentCmd.SERVICE_COMMAND_KEY);
        if (serviceCommand != null) {
            if (ServiceIntentCmd.CALLLOG_REFRESH_REQUEST.equals(serviceCommand)) {
                Runnable runnable = new Runnable() {
                    public void run() {
                        long sourceId = getSourceId();

                        if (sourceId != INVALID_SOURCE_ID) {
                            addNewCallsToEventStream(sourceId);
                        }
                    }
                };
                mHandler.post(runnable);
            } else if (ServiceIntentCmd.CALLLOG_REGISTER_PLUGIN.equals(serviceCommand)) {
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Start CallLog service for registering plugin.");
                }
                Runnable runnable = new Runnable() {
                    public void run() {
                        long sourceId = EventStreamAdapter.getSourceId(getApplicationContext());
                        sourceId = registerOrUpdatePlugin(sourceId);

                        if (sourceId != INVALID_SOURCE_ID) {
                            addNewCallsToEventStream(sourceId);
                        }
                    }
                };
                mHandler.post(runnable);
            } else if (ServiceIntentCmd.CALLLOG_VIEW_EVENT.equals(serviceCommand)) {
                Bundle extras = intent.getExtras();
                String eventKey = null;
                if (extras != null) {
                    eventKey =
                        extras.getString(EventStream.EVENTSTREAM_VIEW_EVENT_KEY_EXTRA);
                }
                if (eventKey != null) {
                    viewCallLogEvent(getApplicationContext(), eventKey);
                }
            } else if (ServiceIntentCmd.CALLLOG_UPDATE_FRIENDS.equals(serviceCommand)) {
                Runnable runnable = new Runnable() {
                    public void run() {
                        updateFriends();
                    }
                };
                mHandler.post(runnable);
            } else if (ServiceIntentCmd.CALLLOG_UPDATE_CALLS.equals(serviceCommand)) {
                Runnable runnable = new Runnable() {
                    public void run() {
                        long sourceId = getSourceId();

                        if (sourceId != INVALID_SOURCE_ID) {
                            removedDeletedCalls();
                            addNewCallsToEventStream(sourceId);
                        }
                    }
                };
                mHandler.post(runnable);
            } else if (ServiceIntentCmd.CALLOG_CONFIGURATION_CHANGED.equals(serviceCommand)) {
                final Context context = this;
                Runnable runnable = new Runnable() {
                    public void run() {
                        EventStreamAdapter.updateRegistration(context);
                    }
                };
                mHandler.post(runnable);
            }
            else if (ServiceIntentCmd.CALLLOG_LOCALE_CHANGED.equals(serviceCommand)) {
                final Context context = this;
                Runnable runnable = new Runnable() {
                    public void run() {
                        EventStreamAdapter.handleLocaleChange(context);
                    }
                };
                mHandler.post(runnable);
            }
        }

        // Register Call log content observer
        if (mCallLogContentObserver == null) {
            mCallLogContentObserver = new CallLogContentObserver(mHandler);
        }
        getContentResolver().unregisterContentObserver(mCallLogContentObserver);
        getContentResolver().registerContentObserver(CallLog.Calls.CONTENT_URI, true,
                mCallLogContentObserver);

        // Register Contacts content observer
        if (mContactsContentObserver == null) {
            mContactsContentObserver = new ContactsContentObserver(mHandler);
        }
        getContentResolver().unregisterContentObserver(mContactsContentObserver);
        getContentResolver().registerContentObserver(Contacts.CONTENT_URI, true,
                mContactsContentObserver);
        return result;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Service#onDestroy:");
        }

        if (mContactsContentObserver != null) {
            getContentResolver().unregisterContentObserver(mContactsContentObserver);
        }
        if (mCallLogContentObserver != null) {
            getContentResolver().unregisterContentObserver(mCallLogContentObserver);
        }
    }

    /***
     * Open the external CallLog item viewer
     *
     * @param eventId the ID of the event to view.
     */
    public void viewCallLogEvent(Context context, String callId) {
        Intent callLogViewIntent = new Intent(Intent.ACTION_VIEW);
        callLogViewIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

        try {
            Integer i = Integer.parseInt(callId);
            Uri uri = ContentUris.withAppendedId(CallLog.Calls.CONTENT_URI, i);
            callLogViewIntent.setData(uri);

            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "CallLogViewLauncher, uri: " + uri.toString());
            }
        }
        catch(NumberFormatException e) {//if not an int, avoid crashing call log
            callLogViewIntent.setData(CallLog.Calls.CONTENT_URI);
            callLogViewIntent.setType(CallLog.Calls.CONTENT_TYPE);//go to call list
        }
        context.startActivity(callLogViewIntent);
    }

    /**
     * Register the CallLog plugin or update the registration if already
     * registered.
     * @param sourceId
     */
    private long registerOrUpdatePlugin(long sourceId) {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Start registration of plugin.");
        }
        Context context = getApplicationContext();

        if (context == null) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "No context!!!.");
            }

            return -1;
        } else {
            boolean isRegistered = EventStreamAdapter.isRegistered(context);

            if (!isRegistered) {
                EventStreamAdapter.register(context);
            } else {
                EventStreamAdapter.updateRegistration(context);
            }

            if (sourceId == -1) {
                sourceId = EventStreamAdapter.registerSource(context);
            } else {
                EventStreamAdapter.updateSource(context, sourceId);
            }

            return sourceId;
        }
    }

    /**
     * Update the friends table with the latest data from the contacts database
     */
    private void updateFriends() {
        Cursor cursor = null;
        Context context = getApplicationContext();

        if (context == null) {
            return ;
        }

        try {
            cursor = EventStreamAdapter.getFriends(context);
            while (cursor != null && cursor.moveToNext()) {
                String phoneNumber = cursor.getString(
                        cursor.getColumnIndex(EventStream.FriendTable.FRIEND_KEY));
                long friendId = cursor.getLong(
                        cursor.getColumnIndex(EventStream.FriendTable.ID_COLUMN));

                updateFriend(friendId, phoneNumber);
            }
        } catch (SQLException exception) {
            if (Config.DEBUG) {
                Log.w(Config.LOG_TAG, "Failed to query friends");
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    /**
     * Update a friend with with the latest data from the contacts database.
     *
     * @param friendId      the friendId
     * @param phonenumber   the friends phone number
     */
    private void updateFriend(long friendId, String phonenumber) {
        Cursor cursor = null;
        Context context = getApplicationContext();

        try {
            cursor = ContactAdapter.getContact(context, phonenumber);
            if (cursor != null && cursor.moveToFirst()) {
                String displayName = cursor.getString(
                        cursor.getColumnIndex(PhoneLookup.DISPLAY_NAME));
                Long contactId = cursor.getLong(cursor.getColumnIndex(PhoneLookup._ID));
                String contactString =
                    Uri.withAppendedPath(ContactsContract.Contacts.CONTENT_URI, String.valueOf(contactId)).toString();

                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Updating friend "+ friendId +
                            " with contact id: " + contactId);
                }

                EventStreamAdapter.updateFriend(
                        context,
                        friendId,
                        null,
                        contactString,
                        displayName);
            } else {
                EventStreamAdapter.updateFriend(
                        context,
                        friendId,
                        null, null, phonenumber);
            }
        } finally {
            if (cursor != null)
                cursor.close();
        }
    }

    private void insertFriend(String phonenumber, long sourceId) {
        Cursor cursor = null;
        Context context = getApplicationContext();

        try {
            cursor = ContactAdapter.getContact(context, phonenumber);
            if (cursor != null && cursor.moveToFirst()) {
                String displayName = cursor.getString(
                        cursor.getColumnIndex(PhoneLookup.DISPLAY_NAME));
                Long contactId = cursor.getLong(cursor.getColumnIndex(PhoneLookup._ID));
                String contactString =
                    Uri.withAppendedPath(ContactsContract.Contacts.CONTENT_URI, String.valueOf(contactId)).toString();

                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Inserting friend with contact uri:" + contactString);
                }

                EventStreamAdapter.insertFriend(
                        context,
                        null,
                        contactString,
                        displayName,
                        phonenumber,
                        sourceId);
            } else {
                EventStreamAdapter.insertFriend(
                        context, null, null, phonenumber, phonenumber, sourceId);
            }
        } finally {
            if (cursor != null)
                cursor.close();
        }
    }

    /**
     * Copy missed calls from CallLog to the events table
     */
    private void addNewCallsToEventStream(long sourceId) {
        Cursor cursor = null;
        int callId = 0;
        Context context = getApplicationContext();

        if (context == null) {
            return ;
        }

        long lastIndex = EventStreamAdapter.getLastInsertedCallId(context);

        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Adding new calls to EventStream, start id = " + lastIndex);
        }

        try {

            cursor = CallLogAdapter.getCalls(context, lastIndex);

            if (cursor == null || cursor.getCount() <= 0) {
                // nothing to do.
                return;
            }

            List<ContentValues> bulkValues = new ArrayList<ContentValues>();

            while (cursor.moveToNext()) {
                ContentValues values = new ContentValues();
                callId = cursor.getInt(
                        cursor.getColumnIndexOrThrow(CallLog.Calls._ID));
                String phoneNumber = cursor.getString(
                        cursor.getColumnIndexOrThrow(CallLog.Calls.NUMBER));
                long timestamp = cursor.getLong(
                        cursor.getColumnIndexOrThrow(CallLog.Calls.DATE));
                int type = cursor.getInt(
                        cursor.getColumnIndexOrThrow(CallLog.Calls.TYPE));

                // If null or empty phone number string is returned from Call Log, then we treat it
                // as Unknown number.
                if (phoneNumber == null || phoneNumber.length() == 0) {
                    phoneNumber = PluginConstants.CallerInfo.UNKNOWN_NUMBER;
                    Log.e(Config.LOG_TAG, "Got a null or empty phonenumber string from Call Log.");
                }
                // The phone-number +46123 is the same as 123
                // If a friend with a matching phone-number already exists
                // we link with the existing friend.
                // If no phone-number is given, then no friend will be created
                String matchingPhonenumber = phoneNumber;

                boolean realPhoneNumber = false;
                if (PluginConstants.CallerInfo.UNKNOWN_NUMBER.equals(phoneNumber)) {
                    matchingPhonenumber = context.getResources().getString(R.string.calllog_callerinfo_unknown);
                    realPhoneNumber = false;
                } else if (PluginConstants.CallerInfo.PRIVATE_NUMBER.equals(phoneNumber)) {
                    matchingPhonenumber = context.getResources().getString(R.string.calllog_callerinfo_private_num);
                    realPhoneNumber = false;
                } else if (PluginConstants.CallerInfo.PAYPHONE_NUMBER.equals(phoneNumber)) {
                    matchingPhonenumber = context.getResources().getString(R.string.calllog_callerinfo_payphone);
                    realPhoneNumber = false;
                } else {
                    realPhoneNumber = true;
                }

                if (realPhoneNumber) {
                    String friendKey = EventStreamAdapter.getFriend(context, phoneNumber);
                    if (friendKey != null) {
                        matchingPhonenumber = friendKey;
                    } else {
                        // No match, create a new friend.
                        insertFriend(phoneNumber, sourceId);
                    }
                }

                int iconId = -1;
                int handledType = 1;
                switch (type) {
                    case Calls.MISSED_TYPE:
                        iconId = R.drawable.missed_msg;
                        handledType = 0;
                        break;
                    case Calls.INCOMING_TYPE:
                        iconId = R.drawable.incoming_msg;
                        break;
                    case Calls.OUTGOING_TYPE:
                        iconId = R.drawable.outgoing_msg;
                        break;
                    default:
                        break;
                }

                if (iconId != -1) {
                    values.put(EventStream.EventTable.STATUS_ICON_URI,
                            new Uri.Builder().scheme(
                                    ContentResolver.SCHEME_ANDROID_RESOURCE)
                                    .authority(getPackageName())
                                    .appendPath(Integer.toString(iconId)).toString());
                }

                values.put(EventStream.EventTable.EVENT_KEY, callId);
                values.put(EventStream.EventTable.SOURCE_ID, sourceId);
                values.put(EventStream.EventTable.PUBLISHED_TIME, timestamp);
                values.put(EventStream.EventTable.HANDLED_TYPE, handledType);
                values.put(EventStream.EventTable.PERSONAL,
                        (type == Calls.MISSED_TYPE || type == Calls.INCOMING_TYPE) ? 1 : 0);
                values.put(EventStream.EventTable.OUTGOING,
                        (type == Calls.OUTGOING_TYPE) ? 1 : 0);
                // If a real phone-number is retrieved, then it is matched against a friend,
                // otherwise the appropriate string (unknown, withheld, payphone) is used as title
                if (realPhoneNumber) {
                    values.put(EventStream.EventTable.FRIEND_KEY, matchingPhonenumber);
                    values.put(EventStream.EventTable.MESSAGE, phoneNumber);
                } else {
                    values.put(EventStream.EventTable.TITLE, matchingPhonenumber);
                }

                bulkValues.add(values);

                if (bulkValues.size() >= BULK_INSERT_MAX_COUNT) {
                    EventStreamAdapter.bulkInsertEvents(context,
                            (ContentValues[])bulkValues.toArray(new ContentValues[bulkValues.size()]));
                    bulkValues.clear();
                    // Give eventstream some time to run queries
                    try {
                        Thread.sleep(BULK_INSERT_DELAY);
                    } catch (InterruptedException e) {
                        // Do nothing
                    }
                }
            }

            if (bulkValues.size() > 0) {
                EventStreamAdapter.bulkInsertEvents(context,
                        (ContentValues[])bulkValues.toArray(new ContentValues[bulkValues.size()]));
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    private void removedDeletedCalls() {
        Context context = getApplicationContext();

        if (context != null) {
            // Fetch a list of the id's of all call log entries
            List<Integer> callIds = CallLogAdapter.getCallIds(context);

            if (callIds.isEmpty()) {
                // Special use case when the user clears the call-log
                EventStreamAdapter.deleteAllEvents(context);
            } else {
                List<String> eventKeys =
                    EventStreamAdapter.getEventKeys(context);
                for (int i = 0; i < callIds.size(); i++) {
                    eventKeys.remove(callIds.get(i).toString());
                }
                EventStreamAdapter.deleteEvents(context, eventKeys);
            }
        }
    }

    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    class FriendUpdateTask extends TimerTask {

        @Override
        public void run() {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Friend update timer expired, updating...");
            }
            Intent serviceIntent = new Intent();
            serviceIntent.setComponent(new ComponentName(getBaseContext(),
                    CallLogPluginService.class));
            serviceIntent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.CALLLOG_UPDATE_FRIENDS);
            startService(serviceIntent);
        }
    }

    class ContactsContentObserver extends ContentObserver {

        private Timer mTimer;

        public ContactsContentObserver(Handler handler) {
            super(handler);
            mTimer = new Timer();
        }

        @Override
        public void onChange(boolean selfChange) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Change occured in Contacts provider.");
            }
            mTimer.cancel();
            mTimer.purge();

            // The timer can't be used once cancel is called...
            mTimer = new Timer();
            mTimer.schedule(new FriendUpdateTask(), Config.FRIEND_UPDATE_DELAY);
        }
    }

    class CallLogContentObserver extends ContentObserver {

        public CallLogContentObserver(Handler handler) {
            super(handler);
        }

        @Override
        public void onChange(boolean selfChange) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Change occured in CallLog provider.");
            }
            Intent serviceIntent = new Intent();
            serviceIntent.setComponent(new ComponentName(getBaseContext(),
                    CallLogPluginService.class));
            serviceIntent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.CALLLOG_UPDATE_CALLS);
            startService(serviceIntent);
        }
    }

}
