/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.calllogplugin;

import com.sonyericsson.eventstream.calllogplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.calllogplugin.PluginConstants.EventStream;
import com.sonyericsson.eventstream.calllogplugin.PluginConstants.ServiceIntentCmd;
import com.sonyericsson.eventstream.calllogplugin.utilities.CallLogPluginTestCase;

import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.database.SQLException;
import android.net.Uri;
import android.util.Log;

import junit.framework.Assert;

public class FPluginRegistrationTests extends CallLogPluginTestCase {

    public void testRegisterPlugin() {
        // Send registration intent to the service
        registerPlugin();
        // Registration can take some time so wait here for a while
        sleep(3);
        // Check if the plugin (and service) are registered
        assertTrue(isPluginRegistered());
        assertTrue(isServiceRegistered());
    }

    public void testRegisterPluginTwice() {
        // Make sure that nothing goes wrong if we receive the registration
        // intent while already registered.
        testRegisterPlugin();
        testRegisterPlugin();
    }

    /*
     * Check that database values are correctly updated on locale change:
     */
    public void testLocaleChange() {

        String junkPluginName = "junk plugin";
        String junkSourceName = "junk source";
        String expectedPluginName = getContext().getResources().getString(
                R.string.calllog_plugin_name);
        String expectedSourceName = getContext().getResources().getString(
                R.string.calllog_source_name);

        testRegisterPlugin();
        //write junk values to database:
        setLocaleColumns(junkPluginName, junkSourceName);

        //verify junk values written
        checkLocaleColumns(expectedPluginName, expectedSourceName, true);

        // trigger locale change
        Intent serviceIntent = new Intent();
        serviceIntent.setComponent(new ComponentName(getContext(), CallLogPluginService.class));
        serviceIntent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY,
                               ServiceIntentCmd.CALLLOG_LOCALE_CHANGED);
        getContext().startService(serviceIntent);

        sleep(3);//give the service a chance to write new values to the database..

        // check new values in database are as expected.
        checkLocaleColumns(expectedPluginName, expectedSourceName, false);
    }

    private void putStringValueInColumn (Uri tableUri, String columnName, String value) {
        ContentValues values = new ContentValues();
        int result;

        values.put(columnName, value);

        try {
            result = getContext().getContentResolver().update(
                    tableUri, values, null, null);
            if (result != 1) {
                Log.e(Config.LOG_TAG, "Failed to update column: " + columnName + "in: " + tableUri + "result= " + result);
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to update column: " + columnName);
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to update column: " + columnName);
            throw new RuntimeException(exception);
        }
    }
/*
 * Compares actual value in column with expected value. If shouldFail == true, then the function will throw an
 * assert if the values match and vice versa.
 */
    private void checkExpectedStringValue(Uri tableUri, String columnName, String expectedValue, boolean shouldFail) {

        Cursor cursor = null;

        try {
            ContentResolver contentResolver = getContext().getContentResolver();
            cursor = contentResolver.query(tableUri, null, null, null, null);

            if (cursor != null && cursor.moveToFirst() && cursor.getCount() == 1) {
                // Check the value in field columnName
                String tempValue = cursor.getString(cursor.getColumnIndexOrThrow(columnName));
                if (shouldFail) {
                    Assert.assertTrue(!expectedValue.equals(tempValue));
                } else {
                    Assert.assertTrue(expectedValue.equals(tempValue));
                }
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    /*
     * Write user-defined values to columns affected by locale change
     */
    private void setLocaleColumns(String pluginName, String sourceName) {
        ContentValues values = new ContentValues();

        putStringValueInColumn(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                               EventStream.PluginTable.NAME,
                               pluginName);


        putStringValueInColumn(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI,
                               EventStream.SourceTable.NAME,
                               sourceName);
    }

    private void checkLocaleColumns(String pluginName, String sourceName, boolean shouldFail) {
        Context context = getContext();
        Cursor cursor = null;

        checkExpectedStringValue(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                                 EventStream.PluginTable.NAME,
                                 pluginName,
                                 shouldFail);

        checkExpectedStringValue(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI,
                                 EventStream.SourceTable.NAME,
                                 sourceName,
                                 shouldFail);
    }

    private long getSourceId(Context context) {
        long sourceId = -1;
        Cursor cursor = null;

        try {
            cursor = context.getContentResolver().query(
                    EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI, new String[] {
                        EventStream.SourceTable.ID_COLUMN
                    }, null, null, null);
            if (cursor != null && cursor.moveToFirst()) {
                sourceId = cursor.getLong(cursor.getColumnIndex(EventStream.SourceTable.ID_COLUMN));
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to query source");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to query source");
            throw new RuntimeException(exception);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return sourceId;
    }

}
