/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file
 * @author Niklas Karlsson (niklas.karlsson@sonyericsson.com)
 */
package com.sonyericsson.eventstream.facebookplugin;

import static com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.PLUGIN_PROVIDER_URI;

import android.app.Instrumentation;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.provider.ContactsContract;
import android.provider.ContactsContract.RawContacts;
import android.test.InstrumentationTestCase;

import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.PluginTable;
import com.sonyericsson.eventstream.facebookplugin.Facebook.EventStorage.EventType;
import com.sonyericsson.eventstream.facebookplugin.util.DatabaseHelper;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;


/**
 * Asserted test towards the event stream framework.
 * All public methods of com.sonyericsson.eventstream.facebookplugin.Database should
 * be tested here.
 * All private methods that tests or create pre-conditions for tests, shall directly
 * use constants defined by the event stream framework
 */
public class UFacebookPluginTestDatabase extends InstrumentationTestCase {

    private static final String FACEBOOK_FRIEND_ID = "100010101010";
    private static final String DISPLAY_NAME = "John Doe";
    private static final String FACEBOOK_FRIEND_ID_2 = "100020202020";
    private static final String DISPLAY_NAME_2 = "Bill Lumbergh";
    private static final String PROFILE_IMAGE_URI = "android.resource://myfile.jpg";
    private static final String FACEBOOK_STATUS_ID = "3330303030303";
    private static final String FACEBOOK_PHOTO_LINK = "android.resource://myfile.jpg";

    Context mContext;
    int mSourceId;
    int mEventId;
    Instrumentation mInsrt;
    Database mDatabase;
    String mSyncAccountName;
    String mSyncAccountType;

    @Override
    protected void setUp() throws Exception {
        mInsrt = getInstrumentation();
        mContext = mInsrt.getTargetContext();
        mDatabase = new Database(mContext);
        deleteDatabaseContent();
        mSourceId = registerPlugin();

        super.setUp();
    }

    @Override
    protected void tearDown() throws Exception {
        deleteDatabaseContent();

        super.tearDown();
    }

    /**
     * Add a new facebook friend
     */
    public void testAddNewFriend() {
        mDatabase.addFriend(FACEBOOK_FRIEND_ID, PROFILE_IMAGE_URI, DISPLAY_NAME);
        int count = mDatabase.storeFriends();
        assertEquals("Couldn't store new friend", 1, count);
        assertEquals("Couldn't retrieve status", 1, getFriendCount());
    }

    /**
     * When adding an existing facebook friend, we don't
     * want to have an other row in the database; instead
     * the existing row shall be updated with the new information
     */
    public void testAddExistingFriend() {
        // Add a new friend
        addFriend(mSourceId, DISPLAY_NAME, PROFILE_IMAGE_URI, FACEBOOK_FRIEND_ID, null);

        // Re-initialize the database object to make sure that the known friend list is up to date
        mDatabase = new Database(mContext);
        mDatabase.addFriend(FACEBOOK_FRIEND_ID, PROFILE_IMAGE_URI, DISPLAY_NAME);
        int count = mDatabase.storeFriends();
        assertEquals("There should have been an update of the friend", 0, count);

        int totalCount = getFriendCount();
        assertEquals("There should should only be one friend", 1, totalCount);
    }

    /**
     * Add a facebook link from a friend (someone the user knowns)
     */
    public void testAddLinkFromFriend() {
        int rowId = addFriend(mSourceId, DISPLAY_NAME, PROFILE_IMAGE_URI, FACEBOOK_FRIEND_ID, null);

        mDatabase.addEvent(FACEBOOK_STATUS_ID, "Hello my friend", null, null, FACEBOOK_FRIEND_ID,
                System.currentTimeMillis(), EventType.LINK_EVENT);
        int count = mDatabase.storeEvents();

        assertTrue("Row id is invalid", rowId > 0);
        assertEquals("We should be able to store a link", 1, count);

        count = getEventCount();
        assertEquals("We should be able to find one event", 1, count);
    }

    /**
     * Add a facebook link with a title!
     */
    public void testAddLinkWithTitle() {
        mDatabase.addEvent(FACEBOOK_STATUS_ID, "Greetings all", "From: Mr Unknown", null, null,
                System.currentTimeMillis(), EventType.LINK_EVENT);
        mDatabase.storeEvents();

        int count = getEventCount();
        assertEquals("We should be able to find one event", 1, count);
    }

    /**
     * Add a facebook link from someone the isn't friends with
     * but has a picture
     */
    public void testAddLinkWithTitleAndPicture() {
        mDatabase.addEvent(FACEBOOK_STATUS_ID, "Greetings all", "From: Mr Unknown",
                "https://sonyericsson.com/image.jpg", null, System.currentTimeMillis(), EventType.LINK_EVENT);

        int count = mDatabase.storeEvents();

        assertEquals("We should be able to store a link", 1, count);

        count = getEventCount();
        assertEquals("We should be able to find one event", 1, count);
    }

    /**
     * Add a facebook status message
     */
    public void testAddStatus() {
        int rowId = addFriend(mSourceId, DISPLAY_NAME, PROFILE_IMAGE_URI, FACEBOOK_FRIEND_ID, null);

        mDatabase.addEvent(FACEBOOK_STATUS_ID, "New status message dude", null, null,
                FACEBOOK_FRIEND_ID, System.currentTimeMillis(), EventType.STATUS_EVENT);
        int count = mDatabase.storeEvents();

        assertTrue("Couldn't add row in friend table", rowId > 0);
        assertEquals("We should be able to store a link", 1, count);

        count = getEventCount();
        assertEquals("We should be able to find one event", 1, count);
    }

    /**
     * Add a facebook status message from someone unknown
     * (not friend) but has a picture
     */
    public void testAddStatusWithTitle() {
        mDatabase.addEvent(FACEBOOK_STATUS_ID, "You don't know me, man", "The title", null, null,
                System.currentTimeMillis(), EventType.STATUS_EVENT);

        int count = mDatabase.storeEvents();

        assertEquals("We should be able to store a link", 1, count);

        count = getEventCount();
        assertEquals("We should be able to find one event", 1, count);
    }

    /**
     * Test that the store methods can be called at any time
     */
    public void testStore() {
        int eventCount = getEventCount();
        int friendCount = getFriendCount();

        mDatabase.storeEvents();
        mDatabase.storeFriends();

        assertEquals("Somehow new events where added", eventCount, getEventCount());
        assertEquals("Somehow new friends where added", friendCount, getFriendCount());
    }

    /**
     * Test that we can register the plug-in
     */
    public void testPluginRegistration() {
        deleteDatabaseContent();

        long sourceId = mDatabase.registerPlugin();
        assertTrue("Couldn't register the plugin", sourceId != 0);
    }

    /**
     * Test that the plugin is registered with correct
     * status update max length..
     */
    public void testPluginRegistrationStatusMaxLength() {
        mDatabase.registerPlugin();
        Cursor pluginCursor = null;

        try {
            pluginCursor = mContext.getContentResolver().query(
                    PLUGIN_PROVIDER_URI,
                    new String[]{PluginTable.STATUS_TEXT_MAX_LENGTH},
                    null, null, null);
            if (pluginCursor != null && pluginCursor.moveToFirst()) {
                assertEquals("status update max length should be 420",
                        420, pluginCursor.getInt(0));
            }
        } finally {
            if (pluginCursor != null) {
                pluginCursor.close();
            }
        }
    }

    /**
     * Test that the plug-in is reinitialized with correct
     * status update max length.
     */
    public void testPluginReInitializeStatusMaxLength() {
        mDatabase.registerPlugin();
        mDatabase.clearData();
        Cursor pluginCursor = null;

        try {
            pluginCursor = mContext.getContentResolver().query(
                    PLUGIN_PROVIDER_URI,
                    new String[]{PluginTable.STATUS_TEXT_MAX_LENGTH},
                    null, null, null);
            if (pluginCursor != null && pluginCursor.moveToFirst()) {
                assertEquals("status update max length should be 420",
                        420, pluginCursor.getInt(0));
            }
        } finally {
            if (pluginCursor != null) {
                pluginCursor.close();
            }
        }
    }

    /**
     * Test that it's possible to find an existing friend
     */
    public void testFindFriend() {
        int rowId = addFriend(mSourceId, DISPLAY_NAME, PROFILE_IMAGE_URI, FACEBOOK_FRIEND_ID, null);

        assertTrue("Couldn't add row in friend table", rowId > 0);

        mDatabase = new Database(mContext);

        boolean found = mDatabase.isFriend(FACEBOOK_FRIEND_ID);
        assertTrue("Couldn't find stored friend", found);
    }

    /**
     * Test that we can't find a friend that doesn't exist
     */
    public void testFindFriendFails() {
        mDatabase = new Database(mContext);

        boolean found = mDatabase.isFriend(FACEBOOK_FRIEND_ID);

        assertFalse("Could find a friend but that is wrong", found);
    }

    /**
     * Test that we can add a photo event from a friend
     */
    public void testAddPhotoFromFriend() {
        int rowId = addFriend(mSourceId, DISPLAY_NAME, PROFILE_IMAGE_URI, FACEBOOK_FRIEND_ID, null);

        assertTrue("Couldn't add row in friend table", rowId > 0);

        mDatabase.addEvent(FACEBOOK_STATUS_ID, "Check this out!", null, null, FACEBOOK_FRIEND_ID,
                System.currentTimeMillis(), EventType.PHOTO_EVENT);
        mDatabase.storeEvents();

        int count = getEventCount();
        assertEquals("Couldn't add photo event", 1, count);
    }

    /**
     * Test that we can add a photo event from someone we don't know
     */
    public void testAddPhotoWithTitle() {
        mDatabase.addEvent(FACEBOOK_STATUS_ID, "Long time no see", "From: John Doe",
                FACEBOOK_PHOTO_LINK, null, System.currentTimeMillis(), EventType.PHOTO_EVENT);
        mDatabase.storeEvents();

        int count = getEventCount();
        assertEquals("Couldn't add photo event", 1, count);
    }

    public void testAutoLinking() {
        List<Long> rawIdList = new ArrayList<Long>();

        // Add friends
        mDatabase.addFriend(FACEBOOK_FRIEND_ID, null, DISPLAY_NAME);
        mDatabase.addFriend(FACEBOOK_FRIEND_ID_2, null, DISPLAY_NAME_2);
        mDatabase.storeFriends();

        try {
            // Create contacts
            createContact(FACEBOOK_FRIEND_ID, DISPLAY_NAME, rawIdList);
            createContact(null, DISPLAY_NAME_2, rawIdList);

            // Link contacts
            mDatabase.autoLinkFriends();

            // Check...
            assertEquals((Long) rawIdList.get(0), DatabaseHelper.getRawContactId(getInstrumentation().getContext(), FACEBOOK_FRIEND_ID));
            assertNull(DatabaseHelper.getRawContactId(getInstrumentation().getContext(), FACEBOOK_FRIEND_ID_2));
        } finally {
            for (Long rawId: rawIdList) {
                deleteRawContact(rawId);
            }
        }
    }

    public void testStoreDisplayname() {
        final String DISPLAY_NAME = "Kalle Kula";
        String storedName = "";

        storedName = DatabaseHelper.getDisplayName(mContext);
        assertTrue("We have a display name", (storedName == null || "".equals(storedName)));

        boolean stored = mDatabase.setConfigurationText(DISPLAY_NAME);
        assertTrue("Display name not stored", stored);

        storedName = DatabaseHelper.getDisplayName(mContext);
        assertEquals("Display name not what we expected", DISPLAY_NAME, storedName);
    }

    public void testSetConfiguration() {
        mDatabase.setConfigurationState(1);
        assertEquals("Configuration value not stored properly", 1, DatabaseHelper.getConfigurationState(mContext));
    }

    public void testGetEventId() {
        String databaseEventId = "5991693871_150863774949768;STATUS";
        String id = Database.getEventId(databaseEventId);
        assertEquals("5991693871_150863774949768", id);
    }

    public void testGetEventType() {
        String id = "5991693871_150863774949768";
        assertEquals(EventType.STATUS_EVENT, Database.getEventType(id + ";STATUS"));
        assertEquals(EventType.LINK_EVENT, Database.getEventType(id + ";LINK"));
        assertEquals(EventType.PHOTO_EVENT, Database.getEventType(id + ";PHOTO"));
        assertEquals(EventType.MESSAGE_EVENT, Database.getEventType(id + ";MESSAGE"));
    }

    public void testExtractFriendKey() {
        final String friendId = "222222";
        final String messageId = "44444";
        final String messageType = "STATUS";
        final String id1 = friendId + "_" + messageId + ";" + messageType;
        String result = null;

        result = Database.extractFriendKeyFromEventId(null);
        assertNull("Null shall be null...", result);

        result = Database.extractFriendKeyFromEventId(id1);
        assertEquals("Couldn't parse id", result, friendId);

        result = Database.extractFriendKeyFromEventId(messageId);
        assertNull("No friend part should give null", result);

        EventType eventType = Database.getEventType(id1);
        assertEquals("Errornous event type", eventType.ordinal(), EventType.STATUS_EVENT.ordinal());
    }

    @SuppressWarnings("unchecked")
    public void testRemoveFriend() throws SecurityException, NoSuchMethodException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
        mDatabase.addFriend(FACEBOOK_FRIEND_ID, PROFILE_IMAGE_URI, DISPLAY_NAME);
        assertEquals(1, mDatabase.storeFriends());
        mDatabase.removeFriend(FACEBOOK_FRIEND_ID);
        Method m = Database.class.getDeclaredMethod("getRegisteredFriendIds", long.class);
        m.setAccessible(true);
        Set<String> friends = (Set<String>)m.invoke(mDatabase, mSourceId);
        assertFalse(friends.contains(FACEBOOK_FRIEND_ID));
    }

    public void testClearData() {
        mDatabase.registerPlugin();
        List<Object> register_plugin = DatabaseHelper.readPluginData(mContext);
        List<Object> register_source = DatabaseHelper.readSourceData(mContext);

        mDatabase.setConfigurationText("John Doe");
        mDatabase.setConfigurationState(0);
        mDatabase.setOwnStatus("My status", 12332423L);
        mDatabase.addEvent("34234234", "Hello", "title", "url", "11111", 34234324L, EventType.STATUS_EVENT);
        mDatabase.addFriend("11111", "profile-url", "Kalle Kula");

        mDatabase.clearData();

        List<Object> logout_plugin = DatabaseHelper.readPluginData(mContext);
        List<Object> logout_source = DatabaseHelper.readSourceData(mContext);

        assertEquals("Some events still left?", 0, DatabaseHelper.getEventCount(mContext));
        assertEquals("Some friends still left?", 0, DatabaseHelper.getFriendCountFromEventStream(mContext));
        assertEquals("Not same data in plugin", register_plugin, logout_plugin);
        assertEquals("Not same data in source", register_source, logout_source);
    }

    private long createContact(String facebookId, String name, List<Long> rawIdList) {
        long id = -1;

        // Add a raw Contact
        ContentResolver cr = mContext.getContentResolver();
        ContentValues values = new ContentValues();
        values.put(RawContacts.ACCOUNT_TYPE, "com.facebook.auth.login");
        values.put(RawContacts.ACCOUNT_NAME, "facebookuser@gmail.com");
        values.put(RawContacts.SOURCE_ID, facebookId);
        Uri result =  cr.insert(ContactsContract.RawContacts.CONTENT_URI, values);
        long rawId = ContentUris.parseId(result);

        rawIdList.add(rawId);

        // Insert name
        values.clear();
        values.put(ContactsContract.Data.RAW_CONTACT_ID, rawId);
        values.put(ContactsContract.Data.MIMETYPE, ContactsContract.CommonDataKinds.StructuredName.CONTENT_ITEM_TYPE);
        values.put(ContactsContract.CommonDataKinds.StructuredName.DISPLAY_NAME, name);
        cr.insert(ContactsContract.Data.CONTENT_URI, values);

        // Look up the aggregated contact Id
        Cursor c = null;
        try {
            c = cr.query(RawContacts.CONTENT_URI, null, RawContacts._ID + "=" + rawId, null, null);
            if (c != null && c.moveToFirst()) {
                id = c.getLong(c.getColumnIndex(RawContacts.CONTACT_ID));
            }
            else {
                fail("Could not create contact");
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }

        return id;
    }

    private void deleteRawContact(Long rawId) {
        ContentResolver cr = mContext.getContentResolver();
        Uri uri = Uri.withAppendedPath(ContactsContract.RawContacts.CONTENT_URI, Long.toString(rawId));
        cr.delete(uri, null, null);
    }

    /**
     * Delete all content in the event stream database
     */
    private void deleteDatabaseContent() {
        DatabaseHelper.removePluginFromEventStream(mContext);
    }

    /**
     * How many friends are stored in the event stream database
     *
     * @return the number of friends stored in the event stream database
     */
    private int getFriendCount() {
        return DatabaseHelper.getFriendCountFromEventStream(mContext);
    }

    /**
     * Add a friend to the event stream database
     *
     * @param sourceId
     * @param displayName
     * @param imageUri
     * @param friendKey
     * @param rawContactUri
     * @return
     */
    private int addFriend(int sourceId, String displayName, String imageUri, String friendKey, String rawContactUri) {
        return DatabaseHelper.addFriend(mContext, sourceId, displayName, imageUri, friendKey, rawContactUri);
    }

    /**
     * How many events are stored in the event stream framework
     *
     * @return the number of events stored
     */
    private int getEventCount() {
        return DatabaseHelper.getEventCount(mContext);
    }

    /**
     * Register the plug-in
     *
     * @return the value of _id column in table source for this plug-in
     */
    private int registerPlugin() {
        return DatabaseHelper.registerPlugin(mContext);
    }

}