/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.utility;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.TwitterConf;

import android.content.ContentResolver;
import android.database.Cursor;
import android.net.Uri;
import android.provider.ContactsContract;
import android.provider.ContactsContract.RawContacts;
import android.util.Log;

public class ContactLookupHelper {

    /** Illegal id when used as Twitter id */
    private static final int ILLEGAL_TWITTER_USER_ID = -1;

    /**
     * Ensure that a raw contact exists in contacts database
     *
     * False will be returned if passing null as objects.
     *
     * @param contentResolver to do the query on
     * @param the raw contact uri to look for
     * @return false if null is passed as objects or key not found, true if found
     */
    public static boolean rawContactExist(ContentResolver contentResolver, String rawContactUri) {
        if (rawContactUri == null || contentResolver == null) {
            return false;
        }
        if (! rawContactUri.startsWith(ContactsContract.RawContacts.CONTENT_URI.toString())) {
            return false;
        }

        Cursor c = null;
        boolean found = false;
        try {
            c = contentResolver.query(Uri.parse(rawContactUri), null, null, null, null);
            if (c != null && c.getCount() == 1) {
                found = true;
            }
        } catch(Exception e) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Unexpected exception caught when checking if raw contact exist");
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }

        return found;
    }

    /**
     * Get the raw contact uri for this Twitter id in contacts database
     * Passing null as objects or -1 as Twitter id will result in
     * an empty string.
     *
     * @param contentResolver to operate on
     * @param accountName is the Twitter user account name (screen name)
     * @param twitterId to do the lookup on
     * @return the raw contact uri for the twitter id. If error has occurred, the string is empty
     */
    public static String getRawContactUriInContacts(ContentResolver contentResolver, long twitterId) {
        String result = "";
        Cursor rawCursor = null;

        if (contentResolver == null || twitterId == ILLEGAL_TWITTER_USER_ID) {
            return result;
        }

        try {
            String where = RawContacts.ACCOUNT_TYPE + " like ? and " +
                RawContacts.SOURCE_ID + " = ?" + " and " +
                RawContacts.DELETED + " = ?";

            String[] whereArgs = {
                    TwitterConf.ACCOUNT_TYPE_SEARCH,
                    String.valueOf(twitterId),
                    "0"
            };

            // Get the raw contact
            rawCursor = contentResolver.query(RawContacts.CONTENT_URI, new String[] {
                RawContacts._ID
            }, where, whereArgs, null);

            if (rawCursor != null && rawCursor.moveToFirst()) {
                long rawContactId = rawCursor.getLong(rawCursor.getColumnIndexOrThrow(RawContacts._ID));

                result = Uri.withAppendedPath(ContactsContract.RawContacts.CONTENT_URI, String.valueOf(rawContactId)).toString();
            }
        } catch(Exception e) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Unexpected exception caught when getting data from contacts provider");
            }
        } finally {
            if (rawCursor != null) {
                rawCursor.close();
            }
        }

        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Raw contact URI = " + result);
        }
        return result;
    }
}

