/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin;

import android.app.Instrumentation;
import android.content.Context;
import android.test.InstrumentationTestCase;

import com.sonyericsson.eventstream.twitterplugin.TwitterPluginApplication.State;

public class UApplicationTests extends InstrumentationTestCase implements TwitterPluginApplication.Listener {

    /** Timeout for state change */
    private static final int TIMEOUT = 2 * 1000;

    State mCurrentState;

    State mOldState;

    TwitterPluginApplication mApplication;

    @Override
    protected void setUp() throws Exception {
        mOldState = State.SERVICE_NOT_CONFIGURED;
        mCurrentState = State.SERVICE_NOT_CONFIGURED;

        Context targetContext = getInstrumentation().getTargetContext();

        try {
            mApplication = (TwitterPluginApplication) Instrumentation.newApplication(TwitterPluginApplication.class,
                    targetContext);
            mApplication.onCreate();
            mApplication.setState(State.SERVICE_NOT_CONFIGURED);
        } catch (Exception exception) {
            fail("Couldn't create application!");
        }

        super.setUp();
    }

    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    /**
    * Test to set and get the state
    */
   public void testSetGetState() {
       // Set the state
       mApplication.setState(State.SERVICE_AUTHENTICATED);
       assertEquals("State set is wrong", true, mApplication.getState() == State.SERVICE_AUTHENTICATED);
   }

    /**
     * Test if state change works
     */
    public void testSetStateChanged() {
        // Set the old state
        mApplication.setState(State.SERVICE_NOT_AUTHENTICATED);
        assertEquals("State set is wrong", true, mApplication.getState() == State.SERVICE_NOT_AUTHENTICATED);

        // Add a listener
        mApplication.addListener(this);

        // Set the state
        mApplication.setState(State.SERVICE_AUTHENTICATED);

        // Wait for state changed
        waitStateChanged(State.SERVICE_AUTHENTICATED, State.SERVICE_NOT_AUTHENTICATED);

        // Remove the listener
        mApplication.removeListener(this);
    }

    /**
     * Test to add a valid listener
     */
    public void testAddValidListener() {
        // Add a listener
        assertEquals("The listener should be added", true, mApplication.addListener(this));

        // Remove the listener
        mApplication.removeListener(this);
    }

    /**
     * Test to add a invalid listener
     */
    public void testAddInValidListener() {
        // Add a listener
        assertEquals("The listener should be not be added", false, mApplication.addListener(null));
    }

    /**
     * Test to remove a valid listener
     */
    public void testRemoveValidListener() {
        // Add the valid listener
        mApplication.addListener(this);

        // Remove the listener
        assertEquals("The listener should be removed", true, mApplication.removeListener(this));
    }

    /**
     * Test to remove a invalid listener
     */
    public void testRemoveInValidListener() {
        // Try to remove a invalid listener
        assertEquals("The listener should not be removed", false, mApplication.removeListener(null));
    }

    /**
     * Check if the visibility is set the to configuration view
     */
    public void testConfigurationVisibility() {
        mApplication.setConfigurationActivityVisible(true);
        assertEquals("Should be visible", true, mApplication.isConfigurationActivityVisible());
        mApplication.setConfigurationActivityVisible(false);
        assertEquals("Should not be visible", false, mApplication.isConfigurationActivityVisible());
    }


    /**
     * Helper method, will wait for state change to finish or timeout.
     * Will also check that the state is correct.
     */
    public void waitStateChanged(State currentState, State oldState) {
        int delay = 0;
        while (!getStateResult(currentState, oldState) && delay <= TIMEOUT) {
            delay += 200;
            try {
                Thread.sleep(200);
                Thread.yield();
            } catch (InterruptedException e) {
                assertTrue("Wait failed", false);
            }
        }
        assertEquals("Wrong state is set", true, getStateResult(currentState, oldState));
    }

    /**
     * Helper method to get the state result
     */
    public boolean getStateResult(State currentState, State oldState) {
        return ((mCurrentState == currentState) && (mOldState == oldState));
    }

    /**
     * Listener results
     */
    public void onStateChange(State oldState, State newState) {
        mCurrentState = newState;
        mOldState = oldState;
    }
}
