/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.utils;

import twitter4j.DirectMessage;
import twitter4j.PagableResponseList;
import twitter4j.Paging;
import twitter4j.ResponseList;
import twitter4j.Status;
import twitter4j.StatusUpdate;
import twitter4j.Twitter;
import twitter4j.TwitterException;
import twitter4j.User;
import twitter4j.http.AccessToken;
import twitter4j.internal.http.HttpResponseCode;

import junit.framework.Assert;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@SuppressWarnings({
        "serial", "deprecation"
})
public class MockTwitter extends Twitter {

    public static final long DEFAULT_PAGING_ID = 10;

    private String mMethodName = "";

    private int mMethodCallCount = 0;

    private boolean mReturnNull = false;

    private boolean mPerformThrowExceptionOnCall = false;

    private int mResponseCode = HttpResponseCode.BAD_REQUEST;

    public static final int DEFAULT_OWN_ID = 999999;

    private String DEFAULT_USER_SCREENNAME = "screenname";

    private String DEFAULT_USER_NAME = "name";

    private String DEFAULT_STATUS = "status";

    private String DEFAULT_TEXT = "text";

    private Date DEFAULT_DATE = new Date();

    //  Acc. to twitter4j getFriendsStatuses(long cursor) can return up to 100 friends at each call.
    //  Cursor is needed to fetch further friends >100. Friends are save at different index in a HashMap.
    private Map<Long, PagableResponseList<User>> friendLists = new HashMap<Long, PagableResponseList<User>>();

    public PagableResponseList<User> generateSomeUsers(int numberOf) {
        PagableResponseList<User> users = new PagableResponseListImpl();
        User user;
        URL url = null;
        try {
            url = new URL("http://www.sonyericsson.com");
        } catch (MalformedURLException e) {
            Assert.fail();
        }
        for (int i = 0; i < numberOf; i++) {
            user = new UserImpl(i, DEFAULT_USER_SCREENNAME + i, DEFAULT_USER_NAME, DEFAULT_STATUS, url);
            users.add(user);
        }
        return users;
    }

    //store a number of users, up to 100, at each index
    public void generateUsers(int numberOf) {
        long indexCounter = 0;
        PagableResponseList<User> users = new PagableResponseListImpl();
        User user;
        URL url = null;

        try {
            url = new URL("http://www.sonyericsson.com");
        } catch (MalformedURLException e) {
            Assert.fail();
        }

        //store bunch of users, 100 at the time
        do {
            if (numberOf>100){
                for (int i = 0; i < 100; i++) {
                    user = new UserImpl(i, DEFAULT_USER_SCREENNAME + i, DEFAULT_USER_NAME, DEFAULT_STATUS, url);
                    users.add(user);
                }
                friendLists.put(new Long(indexCounter), users);
                indexCounter++;
                numberOf=numberOf-100;
            }
        }
        while(numberOf > 100);

        //store users if number <=100
        if(numberOf<=100){
            for (int i = 0; i < numberOf; i++) {
                user = new UserImpl(i, DEFAULT_USER_SCREENNAME + i, DEFAULT_USER_NAME, DEFAULT_STATUS, url);
                users.add(user);
            }
            friendLists.put(new Long(indexCounter), users);
            indexCounter++;
        }
    }

    public PagableResponseList <User> getUsers(int indexCounter) {
        return friendLists.get(indexCounter);
    }

    public ResponseList<Status> generateStatuses(List<User> users, int nbrOf) {
        ResponseList<Status> list = new ResponseListImpl<Status>();
        Status status;
        for (int i = 0; i < users.size(); i++) {
            for (int j = 0; j < nbrOf; j++) {
                status = new StatusImpl(j, users.get(i), DEFAULT_TEXT, DEFAULT_DATE);
                list.add(status);
            }
        }
        return list;
    }

    public ResponseList<DirectMessage> generateMessages(List<User> users, long nbrOf) {
        ResponseList<DirectMessage> list = new ResponseListImpl<DirectMessage>();
        DirectMessage message;
        for (int i = 0; i < users.size(); i++) {
            for (long j = 0; j < nbrOf; j++) {
                message = new DirectMessageImpl(j, users.get(i), DEFAULT_TEXT, DEFAULT_DATE);
                list.add(message);
            }
        }
        return list;
    }

    public User generateOwnUser() {
        URL url = null;
        try {
            url = new URL("http://www.sonyericsson.com");
        } catch (MalformedURLException e) {
            Assert.fail();
        }
        return new UserImpl(DEFAULT_OWN_ID, DEFAULT_USER_SCREENNAME, DEFAULT_USER_NAME, DEFAULT_STATUS, url);
    }

    /**
     * Set a method to track for call count.
     * @param name Method name
     */
    public void setMethodCallCount(String methodName) {
        mMethodCallCount = 0;
        mMethodName = methodName;
    }

    /**
     * Get the call count for the set method
     * @return
     */
    public int getMethodCallCount() {
        return mMethodCallCount;
    }

    /**
     * Will check if the method is called and increase the call counter
     * @param name
     */
    private void checkTrackedMethodCalled(String name) {
        if (mMethodName.equals(name)) {
            ++mMethodCallCount;
        }
    }

    /**
     * Will set is null should be returned when Twitter is called.
     * If throw exception is set it has higher prio.
     * @param returnNull
     */
    public void returnNullOnCall(boolean returnNull) {
        mReturnNull = returnNull;
    }

    /**
     * Set if exception should be thrown or not when Twitter is called
     * @param throwException
     * @param httpResponseCode
     */
    public void performThrowExceptionOnCall(boolean throwException, int httpResponseCode) {
        mPerformThrowExceptionOnCall = throwException;
        mResponseCode = httpResponseCode;
    }

    /**
     * Will check if twitter exception should be thrown
     * @throws TwitterException
     */
    public void throwTwitterException() throws TwitterException {
        if (mPerformThrowExceptionOnCall) {
            throw new TwitterException("", new Exception(), mResponseCode);
        }
    }

    @Override
    public Status updateStatus(StatusUpdate status) throws TwitterException {
        checkTrackedMethodCalled("updateStatus");
        throwTwitterException();

        if (mReturnNull) {
            return null;
        }
        User user = generateOwnUser();
        return user.getStatus();
    }

    @Override
    public User showUser(int userId) throws TwitterException {
        checkTrackedMethodCalled("showUser");
        throwTwitterException();

        if (mReturnNull) {
            return null;
        }
        if (userId == DEFAULT_OWN_ID) {
            return generateOwnUser();
        } else {
            List<User> list = generateSomeUsers(1);
            return list.get(0);
        }
    }

    @Override
    public int getId() throws TwitterException, IllegalStateException {
        checkTrackedMethodCalled("getId");
        throwTwitterException();
        return DEFAULT_OWN_ID;
    }

    @Override
    public void shutdown() {
        checkTrackedMethodCalled("shutdown");
    }

    @Override
    public ResponseList<Status> getHomeTimeline(Paging paging) throws TwitterException {
        checkTrackedMethodCalled("getHomeTimeline");
        Assert.assertEquals("Paging default value wrong", DEFAULT_PAGING_ID, paging.getSinceId());
        throwTwitterException();

        if (mReturnNull) {
            return null;
        }

        User ownUser = generateOwnUser();
        List<User> users = generateSomeUsers(1);
        users.add(ownUser);
        return generateStatuses(users, 10);
    }

    @Override
    public ResponseList<DirectMessage> getDirectMessages(Paging paging) throws TwitterException {
        checkTrackedMethodCalled("getDirectMessages");
        Assert.assertEquals("Paging default value wrong", DEFAULT_PAGING_ID, paging.getSinceId());
        throwTwitterException();

        if (mReturnNull) {
            return null;
        }
        return generateMessages(generateSomeUsers(1), 10);
    }

    @Override
    public PagableResponseList<User> getFriendsStatuses() throws TwitterException {
        checkTrackedMethodCalled("getFriendsStatuses");
        throwTwitterException();

        if (mReturnNull) {
            return null;
        }
        return friendLists.get(new Long(0));
    }

    @Override
    public PagableResponseList<User> getFriendsStatuses(long cursor) throws TwitterException {
        checkTrackedMethodCalled("getFriendsStatuses");
        throwTwitterException();

        if (mReturnNull) {
            return null;
        }
        return friendLists.get(new Long(cursor));
    }

    @Override
    public synchronized void setOAuthConsumer(String arg0, String arg1) {
        checkTrackedMethodCalled("setOAuthConsumer");
    }

    @Override
    public synchronized AccessToken getOAuthAccessToken() throws TwitterException {
        checkTrackedMethodCalled("getOAuthAccessToken");
        throwTwitterException();

        if (mReturnNull) {
            return null;
        }
        return new AccessToken("token", "secret");
    }

    @Override
    public User verifyCredentials() throws TwitterException {
        checkTrackedMethodCalled("verifyCredentials");
        throwTwitterException();

        if (mReturnNull) {
            return null;
        }
        return generateOwnUser();
    }
}
