/*
 * Copyright (C) 2011 OgakiSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ogakisoft.android.gestureime;

import ogakisoft.android.gesture.reform.Gesture;
import ogakisoft.android.gesture.reform.GestureLibrary;
import ogakisoft.android.gesture.reform.GestureStroke;
import ogakisoft.android.gesture.reform.GestureUtilities;
import ogakisoft.android.gesture.reform.GestureView;
import ogakisoft.android.util.LOG;

import android.app.Activity;
import android.content.SharedPreferences;
import android.content.res.Configuration;
import android.graphics.Matrix;
import android.graphics.Path;
import android.graphics.RectF;
import android.os.Bundle;
import android.os.Environment;
import android.preference.PreferenceManager;
import android.util.Log;
import android.view.KeyEvent;
import android.view.View;
import android.view.View.OnFocusChangeListener;
import android.view.View.OnKeyListener;
import android.view.inputmethod.InputMethodManager;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.TextView.OnEditorActionListener;
import android.widget.Toast;

import java.io.File;
import java.util.List;

/**
 * Entry Gesture and Assign String
 * 
 * @author noritoshi ogaki
 * @version 1.1
 */
public class CreateGestureActivity extends Activity {
	@Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        Log.d(TAG, "onConfigurationChanged:"+newConfig.toString());
    }

    private static final String SaveInstanceStateId_Gesture = "gesture";
	private static final String TAG = "CreateGestureActivity";
	private TextView backText;
	private EditText labelText;
	private Gesture mGesture;
	private GestureView.OnGesturingListener mOnGesturingListener;
	private GestureView.OnSizeChangedListener mOnSizeListener;
	private View saveButton;
	private GestureView view;

	private void enableSaveButton() {
		final String str = labelText.getText().toString();
		if (null != mGesture && null != str && str.trim().length() > 0) {
			saveButton.setEnabled(true);
		} else {
			saveButton.setEnabled(false);
		}
	}

	private void hideKeyboard() {
		InputMethodManager imm = (InputMethodManager) getSystemService(INPUT_METHOD_SERVICE);
		imm.hideSoftInputFromWindow(labelText.getWindowToken(), 0);
	}

	public void onClickClearButton(View v) {
		mGesture = null;
		view.clear(false);
		saveButton.setEnabled(false);
	}

	public void onClickCloseButton(View v) {
		finish();
	}

	public void onClickSaveButton(View v) {
		final GestureLibrary lib = GestureLibrary.getInstance();
		String msg = null;
		if (null != mGesture) {
			final CharSequence label = labelText.getText();
			if (0 == label.length()) {
				labelText.setError(getString(R.string.error_illegal_value));
				return;
			}
			lib.addGesture(label.toString(), mGesture);
			if (lib.saveGestures()) {
				setResult(RESULT_OK);
				msg = getResources().getString(R.string.message_save_success,
						label.toString());
				Toast.makeText(CreateGestureActivity.this, msg,
						Toast.LENGTH_LONG).show();
			} else {
				setResult(RESULT_CANCELED);
				msg = getResources().getString(R.string.error_save_gesture,
						label.toString());
				Toast.makeText(CreateGestureActivity.this, msg,
						Toast.LENGTH_LONG).show();
			}
		} else {
			setResult(RESULT_CANCELED);
		}
		mGesture = null;
		labelText.setText("");
		backText.setText("");
		view.clear(false);
		saveButton.setEnabled(false);
	}

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		final SharedPreferences sp;
		
		super.onCreate(savedInstanceState);
		setContentView(R.layout.create_gesture_activity);
		
		backText = (TextView) findViewById(R.id.back_text);
		labelText = (EditText) findViewById(R.id.gesture_name);
		labelText.setSingleLine(false); // line wrap for long text
		saveButton = findViewById(R.id.saveButton);
		view = (GestureView) findViewById(R.id.gesture_view);
		view.setFadeEnabled(false);
		mOnGesturingListener = new GestureView.OnGesturingListener() {
			@Override
			public void onGesturingEnded(GestureView v) {
				mGesture = view.getGesture();
				enableSaveButton();
			}

			@Override
			public void onGesturingStarted(GestureView v) {
				mGesture = null;
				enableSaveButton();
			}
		};
		view.addOnGesturingListener(mOnGesturingListener);
		mOnSizeListener = new GestureView.OnSizeChangedListener() {
			public void onSizeChanged(int width, int height) {
				showData(width, height);
			}
		};
		view.addOnSizeChangedListener(mOnSizeListener);

		sp = PreferenceManager.getDefaultSharedPreferences(this);
		Settings.setFromPreferences(getResources(), sp, view);

		// check SDCARD
		final File f = Environment.getExternalStorageDirectory();
		if (!f.canRead() || !f.canWrite()) {
			LOG.e(TAG, "onCreate: can not access SDCARD");
			Toast.makeText(CreateGestureActivity.this,
					getResources().getString(R.string.error_sdcard),
					Toast.LENGTH_LONG).show();
			labelText.setEnabled(false);
			view.setEnabled(false);
		}

		// Show the input character on gesture-view.
		labelText.setOnEditorActionListener(new OnEditorActionListener() {
			@Override
			public boolean onEditorAction(TextView v, int actionId,
					KeyEvent event) {
				// LOG.d(TAG, "onEditorAction: actionId={0,number,#}",
				// actionId);
				backText.setText("");
				final int len = labelText.getText().length();
				if (len == 1) {
					// show background only first 1-char
					backText.setText(v.getText());
				}
				enableSaveButton();
				return false;
			}
		});
		labelText.setOnFocusChangeListener(new OnFocusChangeListener() {
			@Override
			public void onFocusChange(View v, boolean hasFocus) {
				if (!hasFocus) {
					enableSaveButton();
				}
			}
		});
		labelText.setOnKeyListener(new OnKeyListener() {
			@Override
			public boolean onKey(View v, int keyCode, KeyEvent event) {
				// LOG.d(TAG, "onKey: keyCode={0,number,#}, event={1}", keyCode,
				// event.toString());
				switch (keyCode) {
				case KeyEvent.KEYCODE_ENTER:
					enableSaveButton();
					hideKeyboard();
					return true; // for disable keyCode ENTER
				default:
					enableSaveButton();
					break;
				}
				return false;
			}
		});
	}

	@Override
	protected void onDestroy() {
		super.onDestroy();
		view.removeOnGesturingListener(mOnGesturingListener);
		view.removeOnSizeChangedListener(mOnSizeListener);
		mGesture = null;
		backText = null;
		labelText = null;
		saveButton = null;
		view = null;
	}

	@Override
	protected void onRestoreInstanceState(Bundle state) {
		mGesture = state.getParcelable(SaveInstanceStateId_Gesture);
		if (null != mGesture) {
			view.setGesture(mGesture);
			final NamedGesture g = GestureLibrary.getInstance()
					.getNamedGesture(mGesture.getId());
			if (null != g) {
				LOG.d(TAG,
						"restoreInstanceState: namedGesture id={0,number,#}, label={1}",
						mGesture.getId(), g.getName());
				labelText = (EditText) findViewById(R.id.gesture_name);
				labelText.setText(g.getName());
				enableSaveButton();
			}
		}
		super.onRestoreInstanceState(state);
	}

	@Override
	protected void onResume() {
		super.onResume();
		try {
			mGesture = GestureLibrary.loadGesture();
		} catch (Exception e) {
			LOG.e(TAG, "onResume: load_gesture: {0}", e.getMessage());
		}
		if (null != mGesture) {
			view.setGesture(mGesture);
			LOG.d(TAG, "onResume: Gesture id={0,number,#}", mGesture.getId());
			final GestureLibrary library = GestureLibrary.getInstance();
			final NamedGesture g = library.getNamedGesture(mGesture.getId());
			if (null != g) {
				LOG.d(TAG, "onResume: namedGesture id={0,number,#}, label={1}",
						mGesture.getId(), g.getName());
				labelText = (EditText) findViewById(R.id.gesture_name);
				labelText.setText(g.getName());
				enableSaveButton();
			}
		}
	}

	@Override
	protected void onSaveInstanceState(Bundle state) {
		super.onSaveInstanceState(state);
		if (null != mGesture) {
			state.putParcelable(SaveInstanceStateId_Gesture, mGesture);
		}
	}

	private void showData(int width, int height) {
		final Gesture gesture = view.getGesture();
		if (null != gesture) {
			final int w = (int) (width * 0.8);
			final int h = (int) (height * 0.8);
			final List<GestureStroke> strokes = GestureUtilities.scaling(
					gesture, GestureLibrary.GESTURE_WIDTH,
					GestureLibrary.GESTURE_HEIGHT, w, h);
			gesture.setStrokes(strokes);
			final RectF bounds = new RectF();
			final Path path = gesture.toPath();
			path.computeBounds(bounds, true);
			LOG.d(TAG, "showData: bounds w={0,number,#},h={1,number,#}", bounds.width(), bounds.height());
			Matrix matrix = new Matrix();
			matrix.mapRect(bounds);
			final int inset = 10;
			final float sx = (w - 2 * inset) / bounds.width();
			final float sy = (h - 2 * inset) / bounds.height();
			final float scale = (sx > sy) ? sy : sx;
			path.offset(-bounds.left + (w - bounds.width() * scale) / 2.0f,
					-bounds.top + (h - bounds.height() * scale) / 2.0f);
			final Path viewPath = view.getPath();
			viewPath.rewind();
			path.transform(matrix, viewPath);
			view.invalidate();
		}
	}
}