<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Native postgresql recordset.
 *
 * @package    core_dml
 * @copyright  2008 Petr Skoda (http://skodak.org)
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

require_once(__DIR__.'/moodle_recordset.php');

/**
 * pgsql specific moodle recordset class
 *
 * @package    core_dml
 * @copyright  2008 Petr Skoda (http://skodak.org)
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class pgsql_native_moodle_recordset extends moodle_recordset {

    protected $result;
    /** @var current row as array.*/
    protected $current;
    protected $blobs = array();

    /**
     * Build a new recordset to iterate over.
     *
     * @param resource $result A pg_query() result object to create a recordset from.
     */
    public function __construct($result) {
        $this->result = $result;

        // Find out if there are any blobs.
        $numfields = pg_num_fields($result);
        for ($i = 0; $i < $numfields; $i++) {
            $type = pg_field_type($result, $i);
            if ($type == 'bytea') {
                $this->blobs[] = pg_field_name($result, $i);
            }
        }

        $this->current = $this->fetch_next();
    }

    public function __destruct() {
        $this->close();
    }

    private function fetch_next() {
        if (!$this->result) {
            return false;
        }
        if (!$row = pg_fetch_assoc($this->result)) {
            pg_free_result($this->result);
            $this->result = null;
            return false;
        }

        if ($this->blobs) {
            foreach ($this->blobs as $blob) {
                $row[$blob] = $row[$blob] !== null ? pg_unescape_bytea($row[$blob]) : null;
            }
        }

        return $row;
    }

    public function current() {
        return (object)$this->current;
    }

    public function key() {
        // return first column value as key
        if (!$this->current) {
            return false;
        }
        $key = reset($this->current);
        return $key;
    }

    public function next() {
        $this->current = $this->fetch_next();
    }

    public function valid() {
        return !empty($this->current);
    }

    public function close() {
        if ($this->result) {
            pg_free_result($this->result);
            $this->result  = null;
        }
        $this->current = null;
        $this->blobs   = null;
    }
}
