/*
    Text maid for Windows
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "print.h"
#include <commctrl.h>
#include "abort.h"
#include "edit.h"
#include "resource.h"
#include "wcommon.h"


typedef struct _MARGINDLG
{
  int nWidth, nHeight, nMin, nMax, nMargin;
} MARGINDLG, *PMARGINDLG, *LPMARGINDLG;


/******************************************************************************
*                                                                             *
* ja:印刷関数群                                                               *
*                                                                             *
******************************************************************************/
static ABORTDLG AbortDlg;
static HWND hDlgCancel; /* ja:中断ダイアログボックス */


static BOOL CALLBACK
MarginDlgProc (HWND   hDlg,
               UINT   uMsg,
               WPARAM wParam,
               LPARAM lParam)
{
  switch (uMsg)
    {
      case WM_INITDIALOG:
        {
          LPMARGINDLG lpDlg;

          SetWindowLongPtr (hDlg, DWLP_USER, lParam);
          lpDlg = (LPMARGINDLG)lParam;
          /* ja:右マージンの最小値と最大値を計算する */
          lpDlg->nMin = max (lpDlg->nWidth * 2 / lpDlg->nHeight, 2);
          lpDlg->nMax = min (max (lpDlg->nWidth / 2, lpDlg->nMin), 512);
          /* ja:エディトコントロールの設定 */
          SetDlgItemInt (hDlg, IDC_EDIT31, lpDlg->nMargin, TRUE);
          /* ja:アップダウンコントロールの設定 */
          UpDownItem_SetRange (hDlg, IDC_UPDOWN31, lpDlg->nMin, lpDlg->nMax);
        }
        return TRUE;
      case WM_COMMAND:
        switch (LOWORD (wParam))
          {
            case IDOK:
              {
                LPMARGINDLG lpDlg;

                lpDlg = (LPMARGINDLG)GetWindowLongPtr (hDlg, DWLP_USER);
                lpDlg->nMargin = GetDlgItemInt (hDlg, IDC_EDIT31, NULL, TRUE);
              }
            case IDCANCEL:
              EndDialog (hDlg, LOWORD (wParam));
              return TRUE;
            case IDC_EDIT31:
              if (HIWORD (wParam) == EN_CHANGE)
                {
                  LPMARGINDLG lpDlg;

                  lpDlg = (LPMARGINDLG)GetWindowLongPtr (hDlg, DWLP_USER);
                  if (lpDlg)
                    {
                      int nMargin, nFontSize;

                      /* ja:スタティックコントロールの設定 */
                      nMargin = GetDlgItemInt (hDlg, IDC_EDIT31, NULL, TRUE);
                      if (lpDlg->nMin <= nMargin && nMargin <= lpDlg->nMax)
                        {
                          LPTSTR lpszFormat, lpszText;

                          nFontSize = (lpDlg->nWidth / nMargin * 2 + 1) / 2;
                          lpszFormat = LoadText (GetModuleHandle (NULL),
                                                            IDS_OTHER_LINE);
                          wasprintf (&lpszText, lpszFormat,
                            nFontSize > 0 ? lpDlg->nHeight / (nFontSize * 2)
                                          : 0);
                          MemoryFree (lpszFormat);
                          SetDlgItemText (hDlg, IDC_STATIC31, lpszText);
                          MemoryFree (lpszText);
                        }
                      else
                        {
                          nFontSize = 0;
                          SetDlgItemText (hDlg, IDC_STATIC31, _T(""));
                        }
                      /* ja:コントロールの表示状態 */
                      EnableDlgItem (hDlg, IDOK, nFontSize > 0);
                    }
                }
          }
        return TRUE;
    }
  return FALSE;
}


static BOOL CALLBACK
PrintProc (HDC hdc,
           int nError)
{
  MSG msg;

  while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
    if (!IsDialogMessage (hDlgCancel, &msg))
      {
        TranslateMessage (&msg);
        DispatchMessage (&msg);
      }
  return !AbortDlg.fSignal;
}


/*  ja:プリンタへ出力する
    hWnd,ウインドウ                                                         */
VOID
PrintOut (HWND hWnd)
{
  int i, nFontSize, nLength, nWidth;
  COLORREF crText, crCtrl;
  DOCINFO DocInfo;
  HFONT hFont = NULL, hFontsm = NULL;
  LOGFONT LogFont;
  LPTSTR lpszAbort = NULL, lpszDocName = NULL;
  HINSTANCE hInstance;
  LPTEXTWND ptw;
  MARGINDLG MarginDlg;
  POINT ptStart, ptEnd;
  PRINTDLG PrnDlg;

  ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
  hInstance = GetModuleHandle (NULL);
  /* ja:ダイアログ */
  MemorySet (&PrnDlg, 0, sizeof (PRINTDLG));
  PrnDlg.lStructSize = sizeof (PRINTDLG);
  PrnDlg.hwndOwner = hWnd;
  PrnDlg.Flags = PD_COLLATE | PD_HIDEPRINTTOFILE | PD_RETURNDC;
  PrnDlg.nCopies = 1;
  PrnDlg.Flags |= ptw->ptSelect.x < 0 ? PD_NOSELECTION : PD_SELECTION;
  if (!PrintDlg (&PrnDlg))
    goto final1;
  /* ja:右マージン */
  MarginDlg.nWidth = GetDeviceCaps (PrnDlg.hDC, HORZRES);
  MarginDlg.nHeight = GetDeviceCaps (PrnDlg.hDC, VERTRES);
  MarginDlg.nMargin = ptw->nMargin;
  if (DialogBoxParamGUI (hInstance, MAKEINTRESOURCE (DIALOG_3),
                            hWnd, MarginDlgProc, (LPARAM)&MarginDlg) != IDOK)
    goto final1;
  /* ja:フォント作成 */
  LogFont = ptw->LogFont;
  LogFont.lfHeight = MarginDlg.nWidth / MarginDlg.nMargin * 2;
  nFontSize = (LogFont.lfHeight + 1) / 2;
  hFont = CreateFontIndirect (&LogFont);
  LogFont.lfHeight /= 2;
  hFontsm = CreateFontIndirect (&LogFont);
  /* ja:桁数、行数を求める */
  if (MarginDlg.nHeight < nFontSize * 2)
    {
      LPTSTR lpszText;

      lpszText = LoadText (hInstance, IDS_OTHER_MARGIN);
      MessageBox (hWnd, lpszText,
                        APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      MemoryFree (lpszText);
      goto final1;
    }
  if (ptw->ptSelect.x < 0 || !(PrnDlg.Flags & PD_SELECTION))
    {
      /* ja:選択範囲なし、すべて印刷 */
      ptStart.x = ptStart.y = 0;
      ptEnd.x = GetWidth (&ptw->lpStart, &ptw->nOff, ptw->nMax - 1, ptw->nTab);
      ptEnd.y = ptw->nMax - 1;
    }
  else if (ptw->ptSelect.y < ptw->ptCursor.y
                                        || ptw->ptSelect.y == ptw->ptCursor.y
                                        && ptw->ptSelect.x < ptw->ptCursor.x)
    {
      ptStart = ptw->ptSelect;
      ptEnd = ptw->ptCursor;
    }
  else
    {
      ptStart = ptw->ptCursor;
      ptEnd = ptw->ptSelect;
    }
  if (ptw->fSysColor)
    {
      /* ja:システム色 */
      crText = RGB (  0,   0,   0);
      crCtrl = RGB (128, 128, 128);
    }
  else
    {
      /* ja:ユーザー設定 */
      crText = ptw->crColor[10];
      crCtrl = ptw->crColor[11];
    }
  EnableWindow (hWndMain, FALSE);
  AbortDlg.fSignal = FALSE;
  AbortDlg.lpszText = lpszAbort = LoadText (hInstance, IDS_JOB_PRINT);
  hDlgCancel = CreateDialogParamGUI (hInstance, MAKEINTRESOURCE (DIALOG_G),
                                hWndClient, AbortDlgProc, (LPARAM)&AbortDlg);
  if (SetAbortProc (PrnDlg.hDC, PrintProc) <= 0)
    {
      MessageBox (hWnd, _T("SetAbortProc"),
                        APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      goto final0;
    }
  MemorySet (&DocInfo, 0, sizeof (DOCINFO));
  DocInfo.cbSize =  sizeof (DOCINFO);
  DocInfo.lpszDocName = lpszDocName = GetWindowTextNew (hWnd);
  if (StartDoc (PrnDlg.hDC, &DocInfo) <= 0)
    {
      MessageBox (hWnd, _T("StartDoc"),
                        APPLICATION, MB_OK | MB_ICONEXCLAMATION);
      goto final0;
    }
  for (i = 0; i< (PrnDlg.Flags & PD_COLLATE ? PrnDlg.nCopies : 1); i++)
    {
      int nDataPos, nEndPos, nLine;
      LPLINEBUF p;

      p = GetLineBuffer (&ptw->lpStart, &ptw->nOff, ptStart.y);
      nEndPos = GetDataPos (&ptw->lpStart, &ptw->nOff,
                                        ptEnd.x, ptEnd.y, ptw->nTab, FALSE);
      nLine = ptStart.y;
      nDataPos = GetDataPos (&ptw->lpStart, &ptw->nOff,
                                    ptStart.x, ptStart.y, ptw->nTab, FALSE);
      while (nLine <= ptEnd.y)
        {
          /* ja:ページスタート */
          int x = 0, y = 0, nBkMode;
          COLORREF crTextColor;

          if (StartPage (PrnDlg.hDC) <= 0)
            {
              MessageBox (hWnd, _T("StartPage"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
              goto loop;
            }
          crTextColor = SetTextColor (PrnDlg.hDC, crText);
          nBkMode = SetBkMode (PrnDlg.hDC, TRANSPARENT);
          hFont = SelectObject (PrnDlg.hDC, hFont);
          while (y + nFontSize * 2 < MarginDlg.nHeight && nLine <= ptEnd.y)
            {
              while (x < MarginDlg.nWidth)
                {
                  int nWidth;

                  if (nDataPos >= (nLine == ptEnd.y ? nEndPos : p->nLength))
                    {
                      /* ja:行の終わり */
                      nDataPos = 0;
                      nLine++;
                      p = p->next;
                      if (!p || !p->fMargin)
                        break;
                    }
                  if (p->lpszText[nDataPos] == '\t')
                    nWidth = (x / nFontSize / ptw->nTab + 1)
                                                * ptw->nTab * nFontSize - x;
                  else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
                    nWidth = nFontSize;
                  else
                    nWidth = nFontSize * 2;
                  if (x > 0 && MarginDlg.nWidth < x + nWidth)
                    break;
                  if (p->lpszText[nDataPos] == '\t')
                    {
                      /* ja:タブ */
                      nDataPos++;
                    }
                  else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
                    {
                      /* ja:1バイト文字 */
                      if (!IsCharControl (p->lpszText[nDataPos]))
                        {
                          /* ja:通常の1文字 */
                          TextOutA (PrnDlg.hDC, x, y,
                                                    p->lpszText + nDataPos, 1);
                        }
                      else
                        {
                          int nLength;

                          nLength = lstrlenA (ppszCode[(BYTE)
                                                    p->lpszText[nDataPos]]);
                          if (nLength == 1)
                            {
                              /* en:ANK */
                              TextOutA (PrnDlg.hDC, x, y,
                                    ppszCode[(BYTE)p->lpszText[nDataPos]], 1);
                            }
                          else
                            {
                              SetTextColor (PrnDlg.hDC, crCtrl);/* ja:灰色 */
                              hFontsm = SelectObject (PrnDlg.hDC, hFontsm);
                              if (nLength == 2)
                                {
                                  if (IsDBCSLeadByteEx (CP_ACP,
                                            *ppszCode[(BYTE)
                                                    p->lpszText[nDataPos]]))
                                    {
                                      /* ja:全角1文字 */
                                      TextOutA (PrnDlg.hDC,
                                            x, y + nFontSize / 2,
                                            ppszCode[(BYTE)
                                                    p->lpszText[nDataPos]], 2);
                                    }
                                  else
                                    {
                                      /* ja:半角2文字 */
                                      TextOutA (PrnDlg.hDC, x, y,
                                            ppszCode[(BYTE)
                                                    p->lpszText[nDataPos]], 1);
                                      TextOutA (PrnDlg.hDC,
                                            x + nFontSize / 2, y + nFontSize,
                                            ppszCode[(BYTE)
                                                p->lpszText[nDataPos]] + 1, 1);
                                    }
                                }
                              else if (nLength == 3)
                                {
                                  /* ja:1バイト3文字 */
                                  if (!IsDBCSLeadByteEx (CP_ACP,
                                        *ppszCode[(BYTE)p->lpszText[nDataPos]])
                                    && IsDBCSLeadByteEx (CP_ACP,
                                        *(ppszCode[(BYTE)
                                                p->lpszText[nDataPos]] + 1)))
                                    {
                                      /* ja:半角+全角 */
                                      TextOutA (PrnDlg.hDC,
                                                    x, y, ppszCode[(BYTE)
                                                    p->lpszText[nDataPos]], 2);
                                      TextOutA (PrnDlg.hDC, x, y + nFontSize,
                                            ppszCode[(BYTE)
                                                p->lpszText[nDataPos]] + 2, 2);
                                    }
                                  else
                                    {
                                      /* ja:全角+半角 半角3文字 */
                                      TextOutA (PrnDlg.hDC,
                                                    x, y, ppszCode[(BYTE)
                                                    p->lpszText[nDataPos]], 2);
                                      TextOutA (PrnDlg.hDC,
                                            x + nFontSize / 2, y + nFontSize,
                                            ppszCode[(BYTE)
                                                p->lpszText[nDataPos]] + 2, 1);
                                    }
                                }
                              else
                                {
                                  /* ja:全角2文字 半角4文字 */
                                  TextOutA (PrnDlg.hDC, x, y, ppszCode[(BYTE)
                                                    p->lpszText[nDataPos]], 2);
                                  TextOutA (PrnDlg.hDC, x, y + nFontSize, 
                                                ppszCode[(BYTE)
                                                p->lpszText[nDataPos]] + 2, 2);
                                }
                              SetTextColor (PrnDlg.hDC, crText);
                              hFontsm = SelectObject (PrnDlg.hDC, hFontsm);
                            }
                        }
                        nDataPos++;
                    }
                  else
                    {
                      /* ja:全角文字 */
                      TextOutA (PrnDlg.hDC, x, y, p->lpszText + nDataPos, 2);
                      nDataPos += 2;
                    }
                  x += nWidth;
                }
              x = 0;
              y += nFontSize * 2;
            }
          SetTextColor (PrnDlg.hDC, crTextColor);
          SetBkMode (PrnDlg.hDC, nBkMode);
          hFont = SelectObject (PrnDlg.hDC, hFont);
          if (EndPage (PrnDlg.hDC) <= 0)
            {
              MessageBox (hWnd, _T("EndPage"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
              goto loop;
            }
          if (AbortDlg.fSignal)
            goto loop;
        }
    }
  loop:
  if (EndDoc (PrnDlg.hDC) <= 0)
    MessageBox (hWnd, _T("EndDoc"),
                        APPLICATION, MB_OK | MB_ICONEXCLAMATION);
  final0:
  EnableWindow (hWndMain, TRUE);
  DestroyWindow (hDlgCancel);
  MemoryFree (lpszAbort);
  final1:
  if (hFont)
    DeleteObject (hFont);
  if (hFontsm)
    DeleteObject (hFontsm);
  if (PrnDlg.hDC)
    DeleteDC (PrnDlg.hDC);
}
