/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"
#include <shlwapi.h>


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:文字が数字であるかどうかを判断する
     ch,文字
    RET,TRUE:数字,FALSE:その他                                              */
BOOL WINAPI
IsCharNumericW (WCHAR ch)
{
  return IsCharAlphaNumericW (ch) && !IsCharAlphaW (ch);
}


/*  ja:文字が数字であるかどうかを判断する
     ch,文字
    RET,TRUE:数字,FALSE:その他                                              */
BOOL WINAPI
IsCharNumericA (CHAR ch)
{
  return IsCharAlphaNumericA (ch) && !IsCharAlphaA (ch);
}


/*  ja:文字がコントロールコードであるかどうかを判断する
     ch,文字
    RET,TRUE:コントロールコード,FALSE:その他                                */
BOOL WINAPI
IsCharControlW (WCHAR ch)
{
  WORD ct;

  return !GetStringTypeExW (LOCALE_USER_DEFAULT, CT_CTYPE1, &ch, 1, &ct)
                                                || (ct & C1_CNTRL) || ct == 0;
}


/*  ja:文字がコントロールコードであるかどうかを判断する
     ch,文字
    RET,TRUE:コントロールコード,FALSE:その他                                */
BOOL WINAPI
IsCharControlA (CHAR ch)
{
  WORD ct;

  return !GetStringTypeExA (LOCALE_USER_DEFAULT, CT_CTYPE1, &ch, 1, &ct)
                                                || (ct & C1_CNTRL) || ct == 0;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
static LPCWSTR lpszNumeralW = L"0123456789abcdefghijklmnopqrstuvwxyz";
static LPCSTR  lpszNumeralA =  "0123456789abcdefghijklmnopqrstuvwxyz";


/*  ja:文字列→数値
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
       ppszEnd,終了文字
         nBase,基数
       dwFlags,フラグ
           RET,数値                                                         */
int WINAPI
StringToIntegerExW (LPCWSTR  lpszString,
                    SSIZE_T  nSize,
                    LPCWSTR *ppszEnd,
                    int      nBase,
                    DWORD    dwFlags)
{
  int nLength = 0, nResult = 0;

  if (lpszString)
    {
      int nElement;
      LPCWSTR lpszStart, lpszEnd;
      struct {
        int nBase;
        LPCWSTR lpszPrefix, lpszSuffix;
      } lpFormat[] = {{10, NULL, NULL}, {16, L"0x", NULL}, {16, NULL, L"h"},
                      {10, NULL, L"t"}, { 8, NULL,  L"o"}, { 8, NULL, L"q"},
                      { 2, NULL, L"b"}};

      lpszEnd = lpszString + (nSize >= 0 ? nSize : lstrlenW (lpszString));
      if (2 <= nBase && nBase <= lstrlenW (lpszNumeralW))
        lpFormat[0].nBase = nBase;
      nElement = dwFlags & WCOMMON_STRING_INTEGER_NUMERIC
                                                ? 1 : n_elements (lpFormat);
      for (lpszStart = lpszString; lpszStart < lpszEnd; lpszStart++)
        {
          int i;
          BOOL fSign = FALSE;
          LPCWSTR lpszDigit;

          lpszDigit = lpszStart;
          if (!(dwFlags & WCOMMON_STRING_INTEGER_UNSIGNED))
            switch (*lpszDigit)
              {
                case '-':
                  fSign = TRUE;
                case '+':
                  lpszDigit++;
              }
          for (i = 0; i < nElement; i++)
            {
              int nLen;
              BOOL fMatch;
              LPCWSTR p;

              p = lpszDigit;
              nLen = lstrlenW (lpFormat[i].lpszPrefix);
              fMatch = nLen == 0 || (lpszEnd - p > nLen
                        && StrCmpNIW (p, lpFormat[i].lpszPrefix, nLen) == 0);
              if (fMatch)
                {
                  int nValue = 0;
                  LPCWSTR q;

                  p += nLen;
                  for (q = p; q < lpszEnd; q++)
                    {
                      int nDelta;
                      LPWSTR r;

                      r = StrChrIW (lpszNumeralW, *q);
                      if (!r || (nDelta = r - lpszNumeralW)
                                                        >= lpFormat[i].nBase)
                        break;
                      nValue = nValue * lpFormat[i].nBase + nDelta;
                    }
                  if (q > p)
                    {
                      nLen = lstrlenW (lpFormat[i].lpszSuffix);
                      fMatch = nLen == 0 || (lpszEnd - q >= nLen
                          && StrCmpNIW (q, lpFormat[i].lpszSuffix, nLen) == 0);
                      if (fMatch)
                        {
                          q += nLen;
                          nLen = q - lpszString;
                          if (nLength < nLen)
                            {
                              nLength = nLen;
                              nResult = nValue;
                            }
                        }
                    }
                }
            }
          if (!(dwFlags & WCOMMON_STRING_INTEGER_NUMERIC)
                                                        && *lpszDigit == '\'')
            {
              LPCWSTR p, q;

              p = q = lpszDigit + 1;
              while (q < lpszEnd && *q != '\'')
                q += *q == '\\' ? 2 : 1;
              if (q < lpszEnd)
                {
                  LPWSTR lpszValue;

                  lpszValue = StringUnescapeExW (p, q - p);
                  if (lpszValue)
                    {
                      nResult = 0xD800 <= lpszValue[0] && lpszValue[0] < 0xDC00
                             && 0xDC00 <= lpszValue[1] && lpszValue[1] < 0xE000
                                    ? 0x10000 + (lpszValue[0] - 0xD800) * 0x400
                                              + (lpszValue[1] - 0xDC00)
                                    : lpszValue[0];
                      MemoryFree (lpszValue);
                      nLength = q + 1 - lpszString;
                    }
                }
            }
          if (!(dwFlags & WCOMMON_STRING_INTEGER_NUMERIC)
                                                        && *lpszDigit == '\"')
            {
              LPCWSTR p, q;

              p = q = lpszDigit + 1;
              while (q < lpszEnd && *q != '\"')
                q += *q == '\\' ? 2 : 1;
              if (q < lpszEnd)
                {
                  LPWSTR lpszValue;

                  lpszValue = StringUnescapeExW (p, q - p);
                  if (lpszValue)
                    {
                      for (i = 0; lpszValue[i] != '\0'
                                                    && i < sizeof (int); i++)
                        nResult |= (BYTE)lpszValue[i] << i * 8;
                      MemoryFree (lpszValue);
                      nLength = q + 1 - lpszString;
                    }
                }
            }
          if (fSign)
            nResult *= -1;
          if (dwFlags & WCOMMON_STRING_INTEGER_STRICT)
            {
              if (lpszString + nLength != lpszEnd)
                nLength = 0;
              break;
            }
          if (nLength > 0)
            break;
        }
    }
  if (ppszEnd)
    *ppszEnd = lpszString + nLength;
  return nResult;
}


/*  ja:文字列→数値
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
       ppszEnd,終了文字
         nBase,基数
       dwFlags,フラグ
           RET,TRUE:正常終了,FALSE:エラー                                   */
int WINAPI
StringToIntegerExA (LPCSTR   lpszString,
                    SSIZE_T  nSize,
                    LPCSTR  *ppszEnd,
                    int      nBase,
                    DWORD    dwFlags)
{
  int nLength = 0, nResult = 0;

  if (lpszString)
    {
      int nElement;
      LPCSTR lpszStart, lpszEnd;
      struct {
        int nBase;
        LPCSTR lpszPrefix, lpszSuffix;
      } lpFormat[] = {{10, NULL, NULL}, {16, "0x", NULL}, {16, NULL, "h"},
                      {10, NULL, "t"},  { 8, NULL, "o"},  { 8, NULL, "q"},
                      { 2, NULL, "b"}};

      lpszEnd = lpszString + (nSize >= 0 ? nSize : lstrlenA (lpszString));
      if (2 <= nBase && nBase <= lstrlenA (lpszNumeralA))
        lpFormat[0].nBase = nBase;
      nElement = dwFlags & WCOMMON_STRING_INTEGER_NUMERIC
                                                ? 1 : n_elements (lpFormat);
      for (lpszStart = lpszString; lpszStart < lpszEnd;
                            lpszStart += IsDBCSLeadByte (*lpszStart) ? 2 : 1)
        {
          int i;
          BOOL fSign = FALSE;
          LPCSTR lpszDigit;

          lpszDigit = lpszStart;
          if (!(dwFlags & WCOMMON_STRING_INTEGER_UNSIGNED))
            switch (*lpszDigit)
              {
                case '-':
                  fSign = TRUE;
                case '+':
                  lpszDigit++;
              }
          for (i = 0; i < nElement; i++)
            {
              int nLen;
              BOOL fMatch;
              LPCSTR p;

              p = lpszDigit;
              nLen = lstrlenA (lpFormat[i].lpszPrefix);
              fMatch = nLen == 0 || (lpszEnd - p > nLen
                        && StrCmpNIA (p, lpFormat[i].lpszPrefix, nLen) == 0);
              if (fMatch)
                {
                  int nValue = 0;
                  LPCSTR q;

                  p += nLen;
                  for (q = p; q < lpszEnd; q++)
                    {
                      int nDelta;
                      LPSTR r;

                      r = StrChrIA (lpszNumeralA, *q);
                      if (!r || (nDelta = r - lpszNumeralA)
                                                        >= lpFormat[i].nBase)
                        break;
                      nValue = nValue * lpFormat[i].nBase + nDelta;
                    }
                  if (q > p)
                    {
                      nLen = lstrlenA (lpFormat[i].lpszSuffix);
                      fMatch = nLen == 0 || (lpszEnd - q >= nLen
                          && StrCmpNIA (q, lpFormat[i].lpszSuffix, nLen) == 0);
                      if (fMatch)
                        {
                          q += nLen;
                          nLen = q - lpszString;
                          if (nLength < nLen)
                            {
                              nLength = nLen;
                              nResult = nValue;
                            }
                        }
                    }
                }
            }
          if (!(dwFlags & WCOMMON_STRING_INTEGER_NUMERIC)
                                                        && *lpszDigit == '\'')
            {
              LPCSTR p, q;

              p = q = lpszDigit + 1;
              while (q < lpszEnd && *q != '\'')
                q += *q == '\\' || IsDBCSLeadByte (*q) ? 2 : 1;
              if (q < lpszEnd)
                {
                  LPSTR lpszValue;

                  lpszValue = StringUnescapeExA (p, q - p);
                  if (lpszValue)
                    {
                      nResult = 0xD800 <= lpszValue[0] && lpszValue[0] < 0xDC00
                             && 0xDC00 <= lpszValue[1] && lpszValue[1] < 0xE000
                                    ? 0x10000 + (lpszValue[0] - 0xD800) * 0x400
                                              + (lpszValue[1] - 0xDC00)
                                    : lpszValue[0];
                      MemoryFree (lpszValue);
                      nLength = q + 1 - lpszString;
                    }
                }
            }
          if (!(dwFlags & WCOMMON_STRING_INTEGER_NUMERIC)
                                                        && *lpszDigit == '\"')
            {
              LPCSTR p, q;

              p = q = lpszDigit + 1;
              while (q < lpszEnd && *q != '\"')
                q += *q == '\\' || IsDBCSLeadByte (*q) ? 2 : 1;
              if (q < lpszEnd)
                {
                  LPSTR lpszValue;

                  lpszValue = StringUnescapeExA (p, q - p);
                  if (lpszValue)
                    {
                      for (i = 0; lpszValue[i] != '\0'
                                                    && i < sizeof (int); i++)
                        nResult |= (BYTE)lpszValue[i] << i * 8;
                      MemoryFree (lpszValue);
                      nLength = q + 1 - lpszString;
                    }
                }
            }
          if (fSign)
            nResult *= -1;
          if (dwFlags & WCOMMON_STRING_INTEGER_STRICT)
            {
              if (lpszString + nLength != lpszEnd)
                nLength = 0;
              break;
            }
          if (nLength > 0)
            break;
        }
    }
  if (ppszEnd)
    *ppszEnd = lpszString + nLength;
  return nResult;
}


/*  ja:文字列→数値
    lpszString,文字列
       ppszEnd,終了文字
         nBase,基数
           RET,TRUE:正常終了,FALSE:エラー                                   */
int WINAPI
StringToIntegerW (LPCWSTR  lpszString,
                  LPCWSTR *ppszEnd,
                  int      nBase)
{
  return StringToIntegerExW (lpszString, -1, ppszEnd, nBase, 0);
}


/*  ja:文字列→数値
    lpszString,文字列
       ppszEnd,終了文字
         nBase,基数
           RET,TRUE:正常終了,FALSE:エラー                                   */
int WINAPI
StringToIntegerA (LPCSTR   lpszString,
                  LPCSTR  *ppszEnd,
                  int      nBase)
{
  return StringToIntegerExA (lpszString, -1, ppszEnd, nBase, 0);
}


/*  ja:数値→文字列
    nValue,数値
     nBase,基数
     nWide,桁数
       RET,文字列,NULL:エラー                                               */
LPWSTR WINAPI
StringFromIntegerW (int nValue,
                    int nBase,
                    int nWide)
{
  int i = 0, j;
  LPWSTR lpszString;

  lpszString = MemoryAlloc
                        ((max (nWide, sizeof (int) * 8) + 1) * sizeof (WCHAR));
  if (nBase < 2 && lstrlenW (lpszNumeralW) < nBase)
    nBase = 10;
  if (nValue == 0)
    {
      lpszString[i++] = lpszNumeralW[0];
    }
  else
    {
      int nVal;

      nVal = nValue;
      while (nVal != 0)
        {
          lpszString[i++] = lpszNumeralW[abs (nVal % nBase)];
          nVal /= nBase;
        }
    }
  if (nValue < 0)
    lpszString[i++] = '-';
  if (nWide > 0)
    {
      for (j = i; j < nWide; j++)
        lpszString[j] = lpszNumeralW[0];
      if (lpszString[i - 1] == '-')
        {
          lpszString[i - 1] = lpszNumeralW[0];
          lpszString[j - 1] = '-';
        }
      i = j;
    }
  for (j = 0; j < i / 2; j++)
    {
      WCHAR c;

      c = lpszString[j];
      lpszString[j] = lpszString[i - j - 1];
      lpszString[i - j - 1] = c;
    }
  return lpszString;
}


/*  ja:数値→文字列
    nValue,数値
     nBase,基数
     nWide,桁数
       RET,文字列,NULL:エラー                                               */
LPSTR WINAPI
StringFromIntegerA (int nValue,
                    int nBase,
                    int nWide)
{
  int i = 0, j;
  LPSTR lpszString;

  lpszString = MemoryAlloc
                        ((max (nWide, sizeof (int) * 8) + 1) * sizeof (CHAR));
  if (nBase < 2 && lstrlenA (lpszNumeralA) < nBase)
    nBase = 10;
  if (nValue == 0)
    {
      lpszString[i++] = lpszNumeralA[0];
    }
  else
    {
      int nVal;

      nVal = nValue;
      while (nVal != 0)
        {
          lpszString[i++] = lpszNumeralA[abs (nVal % nBase)];
          nVal /= nBase;
        }
    }
  if (nValue < 0)
    lpszString[i++] = '-';
  if (nWide > 0)
    {
      for (j = i; j < nWide; j++)
        lpszString[j] = lpszNumeralA[0];
      if (lpszString[i - 1] == '-')
        {
          lpszString[i - 1] = lpszNumeralA[0];
          lpszString[j - 1] = '-';
        }
      i = j;
    }
  for (j = 0; j < i / 2; j++)
    {
      CHAR c;

      c = lpszString[j];
      lpszString[j] = lpszString[i - j - 1];
      lpszString[i - j - 1] = c;
    }
  return lpszString;
}


/*  ja:数値→文字列
    uValue,数値
     nBase,基数
     nWide,桁数
       RET,文字列,NULL:エラー                                               */
LPWSTR WINAPI
StringFromUnsignedW (unsigned uValue,
                     int      nBase,
                     int      nWide)
{
  int i = 0, j;
  LPWSTR lpszString;

  lpszString = MemoryAlloc
                ((max (nWide, sizeof (unsigned) * 8) + 1) * sizeof (WCHAR));
  if (nBase < 2 && lstrlenW (lpszNumeralW) < nBase)
    nBase = 10;
  if (uValue == 0)
    {
      lpszString[i++] = lpszNumeralW[0];
    }
  else
    {
      unsigned uVal;

      uVal = uValue;
      while (uVal != 0)
        {
          lpszString[i++] = lpszNumeralW[uVal % nBase];
          uVal /= nBase;
        }
    }
  if (nWide > 0)
    {
      for (j = i; j < nWide; j++)
        lpszString[j] = lpszNumeralW[0];
      i = j;
    }
  for (j = 0; j < i / 2; j++)
    {
      WCHAR c;

      c = lpszString[j];
      lpszString[j] = lpszString[i - j - 1];
      lpszString[i - j - 1] = c;
    }
  return lpszString;
}


/*  ja:数値→文字列
    uValue,数値
     nBase,基数
     nWide,桁数
       RET,文字列,NULL:エラー                                               */
LPSTR WINAPI
StringFromUnsignedA (unsigned uValue,
                     int      nBase,
                     int      nWide)
{
  int i = 0, j;
  LPSTR lpszString;

  lpszString = MemoryAlloc
                ((max (nWide, sizeof (unsigned) * 8) + 1) * sizeof (CHAR));
  if (nBase < 2 && lstrlenA (lpszNumeralA) < nBase)
    nBase = 10;
  if (uValue == 0)
    {
      lpszString[i++] = lpszNumeralA[0];
    }
  else
    {
      unsigned uVal;

      uVal = uValue;
      while (uVal != 0)
        {
          lpszString[i++] = lpszNumeralA[uVal % nBase];
          uVal /= nBase;
        }
    }
  if (nWide > 0)
    {
      for (j = i; j < nWide; j++)
        lpszString[j] = lpszNumeralA[0];
      i = j;
    }
  for (j = 0; j < i / 2; j++)
    {
      CHAR c;

      c = lpszString[j];
      lpszString[j] = lpszString[i - j - 1];
      lpszString[i - j - 1] = c;
    }
  return lpszString;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:8進数文字列を数値に変換する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
     dwDefault,デフォルト値
           RET,数値                                                         */
DWORD WINAPI
OctExW (LPCWSTR lpszString,
        SSIZE_T nSize,
        DWORD   dwDefault)
{
  DWORD dwResult;
  LPCWSTR lpszEnd;

  dwResult = StringToIntegerExW (lpszString, nSize, &lpszEnd, 8,
                                            WCOMMON_STRING_INTEGER_NUMERIC
                                          | WCOMMON_STRING_INTEGER_STRICT
                                          | WCOMMON_STRING_INTEGER_UNSIGNED);
  return lpszString != lpszEnd ? dwResult : dwDefault;
}


/*  ja:8進数文字列を数値に変換する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
     dwDefault,デフォルト値
           RET,数値                                                         */
DWORD WINAPI
OctExA (LPCSTR  lpszString,
        SSIZE_T nSize,
        DWORD   dwDefault)
{
  DWORD dwResult;
  LPCSTR lpszEnd;

  dwResult = StringToIntegerExA (lpszString, nSize, &lpszEnd, 8,
                                            WCOMMON_STRING_INTEGER_NUMERIC
                                          | WCOMMON_STRING_INTEGER_STRICT
                                          | WCOMMON_STRING_INTEGER_UNSIGNED);
  return lpszString != lpszEnd ? dwResult : dwDefault;
}


/*  ja:8進数文字列を数値に変換する
    lpszString,文字列
           RET,数値                                                         */
DWORD WINAPI
OctW (LPCWSTR lpszString)
{
  return OctExW (lpszString, -1, (DWORD)-1);
}


/*  ja:8進数文字列を数値に変換する
    lpszString,文字列
           RET,数値                                                         */
DWORD WINAPI
OctA (LPCSTR lpszString)
{
  return OctExA (lpszString, -1, (DWORD)-1);
}


/*  ja:10進数文字列を数値に変換する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
     dwDefault,デフォルト値
           RET,数値                                                         */
DWORD WINAPI
DecExW (LPCWSTR lpszString,
        SSIZE_T nSize,
        DWORD   dwDefault)
{
  DWORD dwResult;
  LPCWSTR lpszEnd;

  dwResult = StringToIntegerExW (lpszString, nSize, &lpszEnd, 10,
                                            WCOMMON_STRING_INTEGER_NUMERIC
                                          | WCOMMON_STRING_INTEGER_STRICT
                                          | WCOMMON_STRING_INTEGER_UNSIGNED);
  return lpszString != lpszEnd ? dwResult : dwDefault;
}


/*  ja:10進数文字列を数値に変換する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
     dwDefault,デフォルト値
           RET,数値                                                         */
DWORD WINAPI
DecExA (LPCSTR  lpszString,
        SSIZE_T nSize,
        DWORD   dwDefault)
{
  DWORD dwResult;
  LPCSTR lpszEnd;

  dwResult = StringToIntegerExA (lpszString, nSize, &lpszEnd, 10,
                                            WCOMMON_STRING_INTEGER_NUMERIC
                                          | WCOMMON_STRING_INTEGER_STRICT
                                          | WCOMMON_STRING_INTEGER_UNSIGNED);
  return lpszString != lpszEnd ? dwResult : dwDefault;
}


/*  ja:10進数文字列を数値に変換する
    lpszString,文字列
           RET,数値                                                         */
DWORD WINAPI
DecW (LPCWSTR lpszString)
{
  return DecExW (lpszString, -1, (DWORD)-1);
}


/*  ja:10進数文字列を数値に変換する
    lpszString,文字列
           RET,数値                                                         */
DWORD WINAPI
DecA (LPCSTR lpszString)
{
  return DecExA (lpszString, -1, (DWORD)-1);
}


/*  ja:16進数文字列を数値に変換する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
     dwDefault,デフォルト値
           RET,数値                                                         */
DWORD WINAPI
HexExW (LPCWSTR lpszString,
        SSIZE_T nSize,
        DWORD   dwDefault)
{
  DWORD dwResult;
  LPCWSTR lpszEnd;

  dwResult = StringToIntegerExW (lpszString, nSize, &lpszEnd, 16,
                                            WCOMMON_STRING_INTEGER_NUMERIC
                                          | WCOMMON_STRING_INTEGER_STRICT
                                          | WCOMMON_STRING_INTEGER_UNSIGNED);
  return lpszString != lpszEnd ? dwResult : dwDefault;
}


/*  ja:16進数文字列を数値に変換する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
     dwDefault,デフォルト値
           RET,数値                                                         */
DWORD WINAPI
HexExA (LPCSTR  lpszString,
        SSIZE_T nSize,
        DWORD   dwDefault)
{
  DWORD dwResult;
  LPCSTR lpszEnd;

  dwResult = StringToIntegerExA (lpszString, nSize, &lpszEnd, 16,
                                            WCOMMON_STRING_INTEGER_NUMERIC
                                          | WCOMMON_STRING_INTEGER_STRICT
                                          | WCOMMON_STRING_INTEGER_UNSIGNED);
  return lpszString != lpszEnd ? dwResult : dwDefault;
}


/*  ja:16進数文字列を数値に変換する
    lpszString,文字列
           RET,数値                                                         */
DWORD WINAPI
HexW (LPCWSTR lpszString)
{
  return HexExW (lpszString, -1, (DWORD)-1);
}


/*  ja:16進数文字列を数値に変換する
    lpszString,文字列
           RET,数値                                                         */
DWORD WINAPI
HexA (LPCSTR lpszString)
{
  return HexExA (lpszString, -1, (DWORD)-1);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:文字列のプレフィックスが一致するときには真
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
    lpszPrefix,プレフィックス
           RET,TRUE:一致,FALSE:不一致                                       */
BOOL WINAPI
StringHasPrefixExW (LPCWSTR lpszString,
                    SSIZE_T nSize,
                    LPCWSTR lpszPrefix)
{
  BOOL fResult = FALSE;

  if (lpszString)
    {
      int nLength;

      if (nSize < 0)
        nSize = lstrlenW (lpszString);
      nLength = lstrlenW (lpszPrefix);
      fResult = nLength == 0 || (nSize >= nLength
                        && StrCmpNW (lpszString, lpszPrefix, nLength) == 0);
    }
  return fResult;
}


/*  ja:文字列のプレフィックスが一致するときには真
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
    lpszPrefix,プレフィックス
           RET,TRUE:一致,FALSE:不一致                                       */
BOOL WINAPI
StringHasPrefixExA (LPCSTR  lpszString,
                    SSIZE_T nSize,
                    LPCSTR  lpszPrefix)
{
  BOOL fResult = FALSE;

  if (lpszString)
    {
      int nLength;

      if (nSize < 0)
        nSize = lstrlenA (lpszString);
      nLength = lstrlenA (lpszPrefix);
      fResult = nLength == 0 || (nSize >= nLength
                        && StrCmpNA (lpszString, lpszPrefix, nLength) == 0);
    }
  return fResult;
}


/*  ja:文字列のプレフィックスが一致するときには真
    lpszString,文字列
    lpszPrefix,プレフィックス
           RET,TRUE:一致,FALSE:不一致                                       */
BOOL WINAPI
StringHasPrefixW (LPCWSTR lpszString,
                  LPCWSTR lpszPrefix)
{
  return StringHasPrefixExW (lpszString, -1, lpszPrefix);
}


/*  ja:文字列のプレフィックスが一致するときには真
    lpszString,文字列
    lpszPrefix,プレフィックス
           RET,TRUE:一致,FALSE:不一致                                       */
BOOL WINAPI
StringHasPrefixA (LPCSTR  lpszString,
                  SSIZE_T nSize,
                  LPCSTR  lpszPrefix)
{
  return StringHasPrefixExA (lpszString, -1, lpszPrefix);
}


/*  ja:文字列のプレフィックスが一致するときには真
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
    lpszPrefix,プレフィックス(大文字小文字を区別しない)
           RET,TRUE:一致,FALSE:不一致                                       */
BOOL WINAPI
StringHasPrefixIExW (LPCWSTR lpszString,
                     SSIZE_T nSize,
                     LPCWSTR lpszPrefix)
{
  BOOL fResult = FALSE;

  if (lpszString)
    {
      int nLength;

      if (nSize < 0)
        nSize = lstrlenW (lpszString);
      nLength = lstrlenW (lpszPrefix);
      fResult = nLength == 0 || (nSize >= nLength
                        && StrCmpNIW (lpszString, lpszPrefix, nLength) == 0);
    }
  return fResult;
}


/*  ja:文字列のプレフィックスが一致するときには真
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
    lpszPrefix,プレフィックス(大文字小文字を区別しない)
           RET,TRUE:一致,FALSE:不一致                                       */
BOOL WINAPI
StringHasPrefixIExA (LPCSTR  lpszString,
                     SSIZE_T nSize,
                     LPCSTR  lpszPrefix)
{
  BOOL fResult = FALSE;

  if (lpszString)
    {
      int nLength;

      if (nSize < 0)
        nSize = lstrlenA (lpszString);
      nLength = lstrlenA (lpszPrefix);
      fResult = nLength == 0 || (nSize >= nLength
                        && StrCmpNIA (lpszString, lpszPrefix, nLength) == 0);
    }
  return fResult;
}


/*  ja:文字列のプレフィックスが一致するときには真
    lpszString,文字列
    lpszPrefix,プレフィックス(大文字小文字を区別しない)
           RET,TRUE:一致,FALSE:不一致                                       */
BOOL WINAPI
StringHasPrefixIW (LPCWSTR lpszString,
                   LPCWSTR lpszPrefix)
{
  return StringHasPrefixIExW (lpszString, -1, lpszPrefix);
}


/*  ja:文字列のプレフィックスが一致するときには真
    lpszString,文字列
    lpszPrefix,プレフィックス(大文字小文字を区別しない)
           RET,TRUE:一致,FALSE:不一致                                       */
BOOL WINAPI
StringHasPrefixIA (LPCSTR  lpszString,
                   SSIZE_T nSize,
                  LPCSTR  lpszPrefix)
{
  return StringHasPrefixIExA (lpszString, -1, lpszPrefix);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:文字列を書式化する
    ppszString,文字列
    lpszFormat,書式文字列
       arglist,引数
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wvasprintfW (LPWSTR  *ppszString,
             LPCWSTR  lpszFormat,
             va_list  arglist)
{
  SSIZE_T nResult = -1;

  if (lpszFormat && arglist)
    {
      LPWSTR lpszString = NULL;
      SSIZE_T nSize = 0;

      do
        {
          nSize += WCOMMON_BUFFER_SIZE;
          lpszString = MemoryReAlloc (lpszString, nSize * sizeof (WCHAR));
          nResult = wvnsprintfW (lpszString, nSize, lpszFormat, arglist);
        }
      while (nResult < 0 || nResult >= nSize - 1);
      if (ppszString)
        *ppszString = MemoryReAlloc (lpszString,
                                            (nResult + 1) * sizeof (WCHAR));
      else
        MemoryFree (lpszString);
    }
  return nResult;
}


/*  ja:文字列を書式化する
    ppszString,文字列
    lpszFormat,書式文字列
       arglist,引数
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wvasprintfA (LPSTR  *ppszString,
             LPCSTR  lpszFormat,
             va_list arglist)
{
  SSIZE_T nResult = -1;

  if (lpszFormat && arglist)
    {
      LPSTR lpszString = NULL;
      SSIZE_T nSize = 0;

      do
        {
          nSize += WCOMMON_BUFFER_SIZE;
          lpszString = MemoryReAlloc (lpszString, nSize * sizeof (CHAR));
          nResult = wvnsprintfA (lpszString, nSize, lpszFormat, arglist);
        }
      while (nResult < 0 || nResult >= nSize - 1);
      if (ppszString)
        *ppszString = MemoryReAlloc (lpszString,
                                            (nResult + 1) * sizeof (CHAR));
      else
        MemoryFree (lpszString);
    }
  return nResult;
}


/*  ja:文字列を書式化する
    ppszString,文字列
      lpFormat,書式文字列
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wasprintfW (LPWSTR  *ppszString,
            LPCWSTR  lpszFormat, ...)
{
  int nResult;
  va_list arglist;

  va_start (arglist, lpszFormat);
  nResult = wvasprintfW (ppszString, lpszFormat, arglist);
  va_end (arglist);
  return nResult;
}


/*  ja:文字列を書式化する
    ppszString,文字列
      lpFormat,書式文字列
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wasprintfA (LPSTR  *ppszString,
            LPCSTR  lpszFormat, ...)
{
  int nResult;
  va_list arglist;

  va_start (arglist, lpszFormat);
  nResult = wvasprintfA (ppszString, lpszFormat, arglist);
  va_end (arglist);
  return nResult;
}


/*  ja:文字列を複製する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,複製したメモリ,NULL:エラー                                   */
LPWSTR WINAPI
StringDuplicateExW (LPCWSTR lpszString,
                    SSIZE_T nSize)
{
  LPWSTR lpszResult = NULL;

  if (lpszString)
    {
      if (nSize > 0 && lpszString[nSize - 1] != '\0')
        nSize++;
      else if (nSize < 0)
        nSize = lstrlenW (lpszString) + 1;
      if (nSize > 0)
        {
          lpszResult = MemoryAlloc (nSize * sizeof (WCHAR));
          MemoryCopy (lpszResult, lpszString, (nSize - 1) * sizeof (WCHAR));
        }
    }
  return lpszResult;
}


/*  ja:文字列を複製する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,複製したメモリ,NULL:エラー                                   */
LPSTR WINAPI
StringDuplicateExA (LPCSTR  lpszString,
                    SSIZE_T nSize)
{
  LPSTR lpszResult = NULL;

  if (lpszString)
    {
      if (nSize > 0 && lpszString[nSize - 1] != '\0')
        nSize++;
      else if (nSize < 0)
        nSize = lstrlenA (lpszString) + 1;
      if (nSize > 0)
        {
          lpszResult = MemoryAlloc (nSize * sizeof (CHAR));
          MemoryCopy (lpszResult, lpszString, (nSize - 1) * sizeof (CHAR));
        }
    }
  return lpszResult;
}


/*  ja:文字列を複製する
    lpszString,文字列
           RET,複製したメモリ,NULL:エラー                                   */
LPWSTR WINAPI
StringDuplicateW (LPCWSTR lpszString)
{
  return StringDuplicateExW (lpszString, -1);
}


/*  ja:文字列を複製する
    lpszString,文字列
           RET,複製したメモリ,NULL:エラー                                   */
LPSTR WINAPI
StringDuplicateA (LPCSTR lpszString)
{
  return StringDuplicateExA (lpszString, -1);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:文字列をエスケープする
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,エスケープされた文字列                                       */
LPWSTR WINAPI
StringEscapeExW (LPCWSTR lpszString,
                 SSIZE_T nSize)
{
  LPWSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCWSTR p, r;
      LPWSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenW (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen * 6 + 1) * sizeof (WCHAR));
      while (p < r)
        {
          switch (*p)
            {
              case '\a': *q++ = '\\'; *q++ = 'a';  break;
              case '\b': *q++ = '\\'; *q++ = 'b';  break;
              case '\n': *q++ = '\\'; *q++ = 'n';  break;
              case '\r': *q++ = '\\'; *q++ = 'r';  break;
              case '\f': *q++ = '\\'; *q++ = 'f';  break;
              case '\t': *q++ = '\\'; *q++ = 't';  break;
              case '\v': *q++ = '\\'; *q++ = 'v';  break;
              case '\\': *q++ = '\\'; *q++ = '\\'; break;
              case '\?': *q++ = '\\'; *q++ = '\?'; break;
              case '\'': *q++ = '\\'; *q++ = '\''; break;
              case '\"': *q++ = '\\'; *q++ = '\"'; break;
              default:
                if (0x20 <= *p && *p <= 0x7e)
                  {
                    *q++ = *p;
                  }
                else
                  {
                    wsprintfW (q, L"\\u%04X", *p);
                    q += 6;
                  }
            }
          p++;
        }
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  ja:文字列をエスケープする
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,エスケープされた文字列                                       */
LPSTR WINAPI
StringEscapeExA (LPCSTR  lpszString,
                 SSIZE_T nSize)
{
  LPSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCSTR p, r;
      LPSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenA (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen * 4 + 1) * sizeof (CHAR));
      while (p < r)
        {
          switch (*p)
            {
              case '\a': *q++ = '\\'; *q++ = 'a';  break;
              case '\b': *q++ = '\\'; *q++ = 'b';  break;
              case '\n': *q++ = '\\'; *q++ = 'n';  break;
              case '\r': *q++ = '\\'; *q++ = 'r';  break;
              case '\f': *q++ = '\\'; *q++ = 'f';  break;
              case '\t': *q++ = '\\'; *q++ = 't';  break;
              case '\v': *q++ = '\\'; *q++ = 'v';  break;
              case '\\': *q++ = '\\'; *q++ = '\\'; break;
              case '\?': *q++ = '\\'; *q++ = '\?'; break;
              case '\'': *q++ = '\\'; *q++ = '\''; break;
              case '\"': *q++ = '\\'; *q++ = '\"'; break;
              default:
                if (0x20 <= *p && *p <= 0x7e)
                  {
                    *q++ = *p;
                  }
                else
                  {
                    wsprintfA (q, "\\x%02X", (BYTE)*p);
                    q += 4;
                  }
            }
          p++;
        }
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  ja:文字列をエスケープする
    lpszString,文字列
           RET,エスケープされた文字列                                       */
LPWSTR WINAPI
StringEscapeW (LPCWSTR lpszString)
{
  return StringEscapeExW (lpszString, -1);
}


/*  ja:文字列をエスケープする
    lpszString,文字列
           RET,エスケープされた文字列                                       */
LPSTR WINAPI
StringEscapeA (LPCSTR lpszString)
{
  return StringEscapeExA (lpszString, -1);
}


/*  ja:文字列をエスケープしない
    lpszString,エスケープされた文字列
         nSize,文字数(負:NULL終端文字列)
           RET,文字列                                                       */
LPWSTR WINAPI
StringUnescapeExW (LPCWSTR lpszString,
                   SSIZE_T nSize)
{
  LPWSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCWSTR p, r;
      LPWSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenW (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen + 1) * sizeof (WCHAR));
      while (p < r)
        if (*p == '\\')
          {
            switch (*++p)
              {
                case 'a':  *q++ = '\a';  p++; break;
                case 'b':  *q++ = '\b';  p++; break;
                case 'n':  *q++ = '\n';  p++; break;
                case 'r':  *q++ = '\r';  p++; break;
                case 'f':  *q++ = '\f';  p++; break;
                case 't':  *q++ = '\t';  p++; break;
                case 'v':  *q++ = '\v';  p++; break;
                case '\\': *q++ = '\\';  p++; break;
                case '\?': *q++ = '\?';  p++; break;
                case '\'': *q++ = '\'';  p++; break;
                case '\"': *q++ = '\"';  p++; break;
                case 'x':
                  {
                    int v;
                    WCHAR szString[3];

                    lstrcpynW (szString, p + 1, n_elements (szString));
                    if (lstrlenW (szString) == n_elements (szString) - 1
                                                && (v = HexW (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
                  break;
                case 'u':
                  {
                    int v;
                    WCHAR szString[5];

                    lstrcpynW (szString, p + 1, n_elements (szString));
                    if (lstrlenW (szString) == n_elements (szString) - 1
                                                && (v = HexW (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
                  break;
                default:
                  {
                    int v;
                    WCHAR szString[4];

                    lstrcpynW (szString, p, n_elements (szString));
                    if (lstrlenW (szString) == n_elements (szString) - 1
                                                && (v = OctW (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else if (*p == '0')
                      {
                        *q++ = '\0';
                        p++;
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
              }
          }
        else
          {
            *q++ = *p++;
          }
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  ja:文字列をエスケープしない
    lpszString,エスケープされた文字列
         nSize,文字数(負:NULL終端文字列)
           RET,文字列                                                       */
LPSTR WINAPI
StringUnescapeExA (LPCSTR  lpszString,
                   SSIZE_T nSize)
{
  LPSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCSTR p, r;
      LPSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenA (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen + 1) * sizeof (CHAR));
      while (p < r)
        if (*p == '\\')
          {
            switch (*++p)
              {
                case 'a':  *q++ = '\a';  p++; break;
                case 'b':  *q++ = '\b';  p++; break;
                case 'n':  *q++ = '\n';  p++; break;
                case 'r':  *q++ = '\r';  p++; break;
                case 'f':  *q++ = '\f';  p++; break;
                case 't':  *q++ = '\t';  p++; break;
                case 'v':  *q++ = '\v';  p++; break;
                case '\\': *q++ = '\\';  p++; break;
                case '\?': *q++ = '\?';  p++; break;
                case '\'': *q++ = '\'';  p++; break;
                case '\"': *q++ = '\"';  p++; break;
                case 'x':
                  {
                    int v;
                    CHAR szString[3];

                    lstrcpynA (szString, p + 1, n_elements (szString));
                    if (lstrlenA (szString) == n_elements (szString) - 1
                                                && (v = HexA (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
                  break;
                default:
                  {
                    int v;
                    CHAR szString[4];

                    lstrcpynA (szString, p, n_elements (szString));
                    if (lstrlenA (szString) == n_elements (szString) - 1
                                                && (v = OctA (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else if (*p == '0')
                      {
                        *q++ = '\0';
                        p++;
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
              }
          }
        else
          {
            if (IsDBCSLeadByte (*p))
              *q++ = *p++;
            *q++ = *p++;
          }
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  ja:文字列をエスケープしない
    lpszString,エスケープされた文字列
           RET,文字列                                                       */
LPWSTR WINAPI
StringUnescapeW (LPCWSTR lpszString)
{
  return StringUnescapeExW (lpszString, -1);
}


/*  ja:文字列をエスケープしない
    lpszString,エスケープされた文字列
           RET,文字列                                                       */
LPSTR WINAPI
StringUnescapeA (LPCSTR lpszString)
{
  return StringUnescapeExA (lpszString, -1);
}


/*  ja:文字列を正規化する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,正規化された文字列                                           */
LPWSTR WINAPI
StringCanonicalizeExW (LPCWSTR lpszString,
                       SSIZE_T nSize)
{
  LPWSTR lpszResult;

  lpszResult = StringEscapeExW (lpszString, nSize);
  return lpszResult ? lpszResult : StringDuplicateW (L"NULL");
}


/*  ja:文字列を正規化する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,正規化された文字列                                           */
LPSTR WINAPI
StringCanonicalizeExA (LPCSTR  lpszString,
                       SSIZE_T nSize)
{
  LPSTR lpszResult;

  lpszResult = StringEscapeExA (lpszString, nSize);
  return lpszResult ? lpszResult : StringDuplicateA ("NULL");
}


/*  ja:文字列を正規化する
    lpszString,文字列
           RET,正規化された文字列                                           */
LPWSTR WINAPI
StringCanonicalizeW (LPCWSTR lpszString)
{
  return StringCanonicalizeExW (lpszString, -1);
}


/*  ja:文字列を正規化する
    lpszString,文字列
           RET,正規化された文字列                                           */
LPSTR WINAPI
StringCanonicalizeA (LPCSTR lpszString)
{
  return StringCanonicalizeExA (lpszString, -1);
}
