/**@file
 *
 *@brief	QRL Maker
 *@date		Tue,13 Oct,2015 - Sun,18 Oct,2015
 *@date		Fri,23 Oct,2015
 *@date		Mon,26 Oct,2015
 *@date		Wed,28 Oct,2015
 *@date		Thu,29 Oct,2015
 *@date		Sat,31 Oct,2015
 *@date		Thu,11 Feb,2016
 *@date		Sun,16 Apr,2017
 *@date		Sun,09 Jul,2017
 *@date		Sat,15 Jul,2017
 *@date		Thu,27 Jul,2017
 *@date		Sat,17 Mar,2018
 *@date		Thu,29 Nov,2018
 *@date		Mon,24 Dec,2018
 *@date		Fri,28 Dec,2018
 *@date		Sat,29 Dec,2018
 *@date		Sun,30 Dec,2018
 *@date		Fri,25 Jan,2019
 *@date		Sat,26 Jan,2019
 *@date		Sun,27 Dec,2020
 *@date		Wed,17 Feb,2021
 *@date		Fri,19 Feb,2021
 *@date		Mon,22 Feb,2021
 *@date		Tue,23 Feb,2021
 *@date		Sat,29 May,2021
 *@date		Thu,03 Jun,2021
 *@date		Fri,04 Jun,2021
 *@date		Sat,05 Jun,2021
 *@date		Fri,28 Apr,2023
 *
 *@author	Copyright(C)2015-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <assert.h>
#include <cstdint>
#include "qrlmaker.h"
#include "mapeditor.h"
#include "form_mainform.h"
#include "qrl.h"
#include "qrlmap.h"
#include <QTranslator>
#include <QDateTime>
#include "xy_.h"

bool Form_MainForm::action_selectall(void) noexcept
{
	mapeditor->SelectedList_()->clear();

	for (QRL::QRLChipNumber i = 0; i < mapeditor->QRLMap().maxlist(); ++i) {
		const QRL::QRLChip* const ptr = mapeditor->QRLMap().ptr(i);
		assert (NULL != ptr);

		mapeditor->SelectedList_()->put(ptr);
	}

	this->needRefresh = true;
	buildImage(MapEditor_Type::MOUSE_NONE);

	{
		QDateTime dt = QDateTime::currentDateTime();
		QString str = dt.toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " " + tr("Select All");
		mainwindow->set_statusBar1(str);
	}
	return true;
}

bool Form_MainForm::action_unselectall(void) noexcept
{
	mapeditor->SelectedList_()->clear();

	this->needRefresh = true;
	buildImage(MapEditor_Type::MOUSE_NONE);

	{
		QDateTime dt = QDateTime::currentDateTime();
		QString str = dt.toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " " + tr("Unselect All");
		mainwindow->set_statusBar1(str);
	}
	return true;
}

bool Form_MainForm::action_undo(void) noexcept
{
	QString msg;
	mapeditor->SelectedList_()->clear();
	const int_fast8_t ptr = mapeditor->QRLMap_().undo(&msg);
	mainwindow->set_windowtitle();

	if (!msg.isEmpty()) {
		QDateTime dt = QDateTime::currentDateTime();
		QString str = dt.toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " " + tr("Undo");
		mainwindow->set_statusBar1(str + " (" + msg + ")");
	}

	for (QRL::QRLChipNumber i = 0; i < mapeditor->SelectedList()->maxlist(); ++i) {
		const QRL::QRLChipNumber chip_number = mapeditor->SelectedList()->ptr(i)->chip_number;
		if (0 < chip_number) {
			if (NULL == mapeditor->QRLMap().search(chip_number)) {
				mapeditor->SelectedList_()->remove(chip_number);
			}
		}
	}

	this->needRefresh = true;
	buildImage(MapEditor_Type::MOUSE_NONE);

	return (0 <= ptr);
}

bool Form_MainForm::action_redo(void) noexcept
{
	QString msg;
	const int_fast8_t ptr = mapeditor->QRLMap_().redo(&msg);
	mapeditor->SelectedList_()->clear();

	if (0 <= ptr) {
		QDateTime dt = QDateTime::currentDateTime();
		QString str = dt.toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " " + tr("Redo");
		mainwindow->set_statusBar1(str + " (" + msg + ")");
	}

	mainwindow->set_windowtitle();

	this->needRefresh = true;
	buildImage(MapEditor_Type::MOUSE_NONE);
	return (0 <= ptr);
}

bool Form_MainForm::action_cut(void) noexcept
{
	if (mapeditor->SelectedList()->numlist() < 1) {
		const QRL::QRLChipNumber chip_number = mapeditor->QRLMap().map(focus_addr);
		if (0 < chip_number) {
			for (QRL::QRLChipNumber i = 0; i < mapeditor->QRLMap().maxlist(); ++i) {
				const QRL::QRLChip* const ptr = mapeditor->QRLMap().ptr(i);
				assert (NULL != ptr);
				if (chip_number == ptr->chip_number) {
					mapeditor->SelectedList_()->put(ptr);
					break;
				}
			}
		}
	}

	(*(qrlmaker->Clipboard_()->list_())) = (*(mapeditor->SelectedList()));
	qrlmaker->Clipboard_()->reallocate(NULL);
	qrlmaker->Clipboard_()->reset_ofs();

	for (QRL::QRLChipNumber i = 0; i < mapeditor->SelectedList()->maxlist(); ++i) {
		const QRL::QRLChipNumber chip_number = mapeditor->SelectedList()->ptr(i)->chip_number;
		if (0 < chip_number) {
			mapeditor->QRLMap_().remove(chip_number);
		}
	}
	mapeditor->SelectedList_()->clear();

	this->needRefresh = true;
	buildImage(MapEditor_Type::MOUSE_NONE);

	{
		QDateTime dt = QDateTime::currentDateTime();
		QString str = dt.toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " " + tr("Cut");
		mainwindow->set_statusBar1(str);
	}
	{
		mainwindow->set_edited(mapeditor->QRLMap_().push_undo(tr("Cut")));
		mainwindow->set_windowtitle();
	}
	return true;
}

bool Form_MainForm::action_copy(void) noexcept
{
	if (mapeditor->SelectedList()->numlist() < 1) {
		const QRL::QRLChipNumber chip_number = mapeditor->QRLMap().map(focus_addr);
		if (0 < chip_number) {
			for (QRL::QRLChipNumber i = 0; i < mapeditor->QRLMap().maxlist(); ++i) {
				const QRL::QRLChip* const ptr = mapeditor->QRLMap().ptr(i);
				assert (NULL != ptr);
				if (chip_number == ptr->chip_number) {
					mapeditor->SelectedList_()->put(ptr);
					break;
				}
			}
		}
	}

	(*(qrlmaker->Clipboard_()->list_())) = (*(mapeditor->SelectedList()));
	qrlmaker->Clipboard_()->reallocate(NULL);
	qrlmaker->Clipboard_()->reset_ofs();

	mapeditor->SelectedList_()->clear();

	this->needRefresh = true;
	buildImage(MapEditor_Type::MOUSE_NONE);

	{
		QDateTime dt = QDateTime::currentDateTime();
		QString str = dt.toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " " + tr("Copy");
		mainwindow->set_statusBar1(str);
	}
	return true;
}

bool Form_MainForm::action_paste(void) noexcept
{
	const XY_<int_fast32_t> mouse_ptr = {
		.x = -1,
		.y = -1,
	};
	this->needRefresh = true;
	buildImage(MapEditor_Type::MOUSE_NONE, focus_addr, mouse_ptr, NULL);

	{
		QDateTime dt = QDateTime::currentDateTime();
		QString str = dt.toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " " + tr("Paste");
		mainwindow->set_statusBar1(str);
	}
	{
		mainwindow->set_edited(mapeditor->QRLMap_().push_undo(tr("Paste")));
		mainwindow->set_windowtitle();
	}
	return true;
}

bool Form_MainForm::action_delete(void) noexcept
{
	if (mapeditor->SelectedList()->numlist() < 1) {
		const QRL::QRLChipNumber chip_number = mapeditor->QRLMap().map(focus_addr);
		if (0 < chip_number) {
			for (QRL::QRLChipNumber i = 0; i < mapeditor->QRLMap().maxlist(); ++i) {
				const QRL::QRLChip* const ptr = mapeditor->QRLMap().ptr(i);
				assert (NULL != ptr);
				if (chip_number == ptr->chip_number) {
					mapeditor->SelectedList_()->put(ptr);
					break;
				}
			}
		}
	}

	for (QRL::QRLChipNumber i = 0; i < mapeditor->SelectedList()->maxlist(); ++i) {
		const QRL::QRLChipNumber chip_number = mapeditor->SelectedList()->ptr(i)->chip_number;
		if (0 < chip_number) {
			mapeditor->QRLMap_().remove(chip_number);
		}
	}
	mapeditor->SelectedList_()->clear();

	this->needRefresh = true;
	buildImage(MapEditor_Type::MOUSE_NONE);

	{
		QDateTime dt = QDateTime::currentDateTime();
		QString str = dt.toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " " + tr("Delete");
		mainwindow->set_statusBar1(str);
	}
	{
		mainwindow->set_edited(mapeditor->QRLMap_().push_undo(tr("Delete")));
		mainwindow->set_windowtitle();
	}
	return true;
}
// [ End of File ]
