/**@file
 *
 *@brief	QRL Maker
 *@date		Fri,28 Dec,2018
 *@date		Sat,29 Dec,2018 - Sun,30 Dec,2018
 *@date		Mon,31 Dec,2018
 *@date		Mon,21 Jan,2019
 *@date		Fri,25 Jan,2019
 *@date		Sat,26 Jan,2019
 *@date		Sun,27 Jan,2019
 *@date		Tue,29 Jan,2019
 *@date		Thu,31 Jan,2019
 *@date		Sun,27 Dec,2020
 *@date		Wed,17 Feb,2021
 *@date		Thu,18 Feb,2021
 *@date		Fri,19 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Mon,22 Feb,2021
 *@date		Tue,23 Feb,2021
 *@date		Sat,29 May,2021
 *@date		Thu,03 Jun,2021
 *@date		Fri,04 Jun,2021
 *@date		Sat,05 Jun,2021
 *@date		Wed,13 Jul,2022
 *@date		Fri,28 Apr,2023
 *
 *@author	Copyright(C)2018-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <cstdint>
#include <math.h>
#include "settings.h"
#include "mapeditor.h"
#include "imagebuilder.h"
#include "form_mainform.h"
#include "mytypes.h"
#include <QTranslator>
#include <QKeyEvent>
#include "xy_.h"

void Form_MainForm::keyPressEvent(QKeyEvent* const event) noexcept
{
	const Qt::KeyboardModifiers mod = event->modifiers();
	const bool mod_shift = (0 != (Qt::ShiftModifier & mod));
	const bool mod_ctrl = (0 != (Qt::ControlModifier & mod));

	if (0 != (Qt::AltModifier & mod)) {

		switch (event->key()) {
		case Qt::Key_Up:
			mainwindow->select_form_treeitem_prev();
			if (Settings::EditorCommand::NONE != this->mode) {
				buildImage(MapEditor_Type::MOUSE_NONE);
			}
			break;

		case Qt::Key_Down:
			mainwindow->select_form_treeitem_next();
			if (Settings::EditorCommand::NONE != this->mode) {
				buildImage(MapEditor_Type::MOUSE_NONE);
			}
			break;

		case Qt::Key_Left:
			if (mod_shift) {
				mainwindow->select_form_treeitem_collapse_All();
			} else {
				mainwindow->select_form_treeitem_collapse();
			}
			break;

		case Qt::Key_Right:
			if (mod_shift) {
				mainwindow->select_form_treeitem_expand_All();
			} else {
				mainwindow->select_form_treeitem_expand();
			}
			break;
		}

	} else {

		bool need_buildImage = false;
		const int step = (mod_ctrl ? 10 : 1 );
		QRL::ChipArrow arrow = QRL::ChipArrow_MIN;
		uint_fast8_t target_argument_num = 0;

		switch (event->key()) {
		case Qt::Key_Up: this->focus_addr.y -= step; goto KEY_DIR;
		case Qt::Key_Down: this->focus_addr.y += step; goto KEY_DIR;
		case Qt::Key_Left: this->focus_addr.x -= step; goto KEY_DIR;
		case Qt::Key_Right: this->focus_addr.x += step; goto KEY_DIR;
		case Qt::Key_H: if (mod_shift) { arrow = QRL::ChipArrow_LEFT;      goto KEY_ARROW; } else { this->focus_addr.x -= step; goto KEY_DIR; }
		case Qt::Key_J: if (mod_shift) { arrow = QRL::ChipArrow_DOWN;      goto KEY_ARROW; } else { this->focus_addr.y += step; goto KEY_DIR; }
		case Qt::Key_K: if (mod_shift) { arrow = QRL::ChipArrow_UP;        goto KEY_ARROW; } else { this->focus_addr.y -= step; goto KEY_DIR; }
		case Qt::Key_L: if (mod_shift) { arrow = QRL::ChipArrow_RIGHT;     goto KEY_ARROW; } else { this->focus_addr.x += step; goto KEY_DIR; }
		case Qt::Key_Y: if (mod_shift) { arrow = QRL::ChipArrow_UPLEFT;    goto KEY_ARROW; } else { this->focus_addr.x -= step; this->focus_addr.y -= step; goto KEY_DIR; }
		case Qt::Key_U: if (mod_shift) { arrow = QRL::ChipArrow_UPRIGHT;   goto KEY_ARROW; } else { this->focus_addr.x += step; this->focus_addr.y -= step; goto KEY_DIR; }
		case Qt::Key_B: if (mod_shift) { arrow = QRL::ChipArrow_DOWNLEFT;  goto KEY_ARROW; } else { this->focus_addr.x -= step; this->focus_addr.y += step; goto KEY_DIR; }
		case Qt::Key_N: if (mod_shift) { arrow = QRL::ChipArrow_DOWNRIGHT; goto KEY_ARROW; } else { this->focus_addr.x += step; this->focus_addr.y += step; goto KEY_DIR; }
		  KEY_DIR:
			if (this->focus_addr.x < 0) {
				this->focus_addr.x = 0;
			}
			if (Settings::BuildImage.Chip.Count.x <= this->focus_addr.x) {
				this->focus_addr.x = (Settings::BuildImage.Chip.Count.x - 1);
			}
			if (this->focus_addr.y < 0) {
				this->focus_addr.y = 0;
			}
			if (Settings::BuildImage.Chip.Count.y <= this->focus_addr.y) {
				this->focus_addr.y = (Settings::BuildImage.Chip.Count.y - 1);
			}
			need_buildImage = true;
			relocation_scrollbar();
			break;

		  KEY_ARROW:
			{
				const QRL::QRLChipNumber num = mapeditor->QRLMap().map(focus_addr);
				if (0 <= num) {
					QRL::QRLChip* const chip_ptr = mapeditor->QRLMap_().search_(num);

					QString msg;
					if (mod_ctrl) {
						if (mapeditor->set_arrow(chip_ptr, (focus_addr - chip_ptr->addr), arrow, false)) {
							msg = tr("Set Arrow False");
						} else {
							msg = tr("Cannot Be Set in Such a Arrow(False)");
						}
					} else {
						if (mapeditor->set_arrow(chip_ptr, (focus_addr - chip_ptr->addr), arrow, true)) {
							msg = tr("Set Arrow True");
						} else {
							msg = tr("Cannot Be Set in Such a Arrow(True)");
						}
					}

					this->needRefresh = true;
					buildImage(MapEditor_Type::MOUSE_NONE);

					mainwindow->set_statusBar1(msg);
					mainwindow->set_edited(mapeditor->QRLMap_().push_undo(msg));
					mainwindow->set_windowtitle();
				}
			}
			break;

		case Qt::Key_BracketLeft:
			{
				mapeditor->Set_Drag_ChipAddr(this->focus_addr);
				mapeditor->Set_SimpleCommand(MapEditor_Type::SIMPLECOMMAND_RECTANGULAR);

				const XY_<int_fast32_t> mouse_ptr = {
					.x = -1,
					.y = -1,
				};
				buildImage(MapEditor_Type::MOUSE_NONE, focus_addr, mouse_ptr, NULL);
			}
			break;

		case Qt::Key_BracketRight:
			{
				const XY_<int_fast32_t> mouse_ptr = {
					.x = -1,
					.y = -1,
				};
				buildImage(MapEditor_Type::MOUSE_FIN, focus_addr, mouse_ptr, NULL);
				mapeditor->Reset_Drag_ChipAddr();
			}
			break;

		case Qt::Key_PageUp:
			if (mod_ctrl) {
				mainwindow->select_scaling_dB_expand();
			} else {
				const int step = (mod_ctrl ? 10 : 1 );
				const XY_<int> sp = {
					.x = mainwindow->get_horizontalScrollBar(),
					.y = mainwindow->get_verticalScrollBar(),
				};

				if (mod_shift) {
					this->focus_addr.x -= step;
					if (this->focus_addr.x < 0) {
						this->focus_addr.x = 0;
					}
					mainwindow->set_horizontalScrollBar(sp.x - 1);
				} else {
					this->focus_addr.y -= step;
					if (this->focus_addr.y < 0) {
						this->focus_addr.y = 0;
					}
					mainwindow->set_verticalScrollBar(sp.y - 1);
				}
				relocation_scrollbar();
			}
			break;

		case Qt::Key_PageDown:
			if (mod_ctrl) {
				mainwindow->select_scaling_dB_shrink();
			} else {
				const int step = (mod_ctrl ? 10 : 1 );
				const XY_<int> sp = {
					.x = mainwindow->get_horizontalScrollBar(),
					.y = mainwindow->get_verticalScrollBar(),
				};

				if (mod_shift) {
					this->focus_addr.x += step;
					if (Settings::BuildImage.Chip.Count.x <= this->focus_addr.x) {
						this->focus_addr.x = (Settings::BuildImage.Chip.Count.x - 1);
					}
					mainwindow->set_horizontalScrollBar(sp.x + 1);
				} else {
					this->focus_addr.y += step;
					if (Settings::BuildImage.Chip.Count.y <= this->focus_addr.y) {
						this->focus_addr.y = (Settings::BuildImage.Chip.Count.y - 1);
					}
					mainwindow->set_verticalScrollBar(sp.y + 1);
				}
				relocation_scrollbar();
			}
			break;

		case Qt::Key_Return:
		case Qt::Key_Enter:
		case Qt::Key_Space:
			{
				QString msg;
				switch (this->mode) {
				case Settings::EditorCommand::NONE:
					break;
				case Settings::EditorCommand::NORMAL:
					break;
				case Settings::EditorCommand::SELECT:
					break;
				case Settings::EditorCommand::RECTANGULAR:
					break;
				case Settings::EditorCommand::CUT:
					break;
				case Settings::EditorCommand::PASTE:
					msg = tr("Paste");
					need_buildImage = true;
					break;
				case Settings::EditorCommand::DELETE:
					break;
				case Settings::EditorCommand::SET_ARROW_TRUE:
					break;
				case Settings::EditorCommand::SET_ARROW_FALSE:
					break;
				case Settings::EditorCommand::EDIT_ARGUMENT:
					msg = tr("Edit Argument");
					need_buildImage = true;
					break;
				case Settings::EditorCommand::PUT_CHIP:
					msg = tr("Put");
					need_buildImage = true;
					break;
				}
				if (need_buildImage) {
					const XY_<int_fast32_t> mouse_ptr = {
						.x = static_cast<int_fast32_t>(((focus_addr.x - this->cur_screen.ofs.x) * (Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize) + (Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize / 2)) * display_img->scaling() + Settings::BuildImage.GaugeSize),
						.y = static_cast<int_fast32_t>(((focus_addr.y - this->cur_screen.ofs.y) * (Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize) + (Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize / 2)) * display_img->scaling() + Settings::BuildImage.GaugeSize),
					};
					if (mod_ctrl) {
						buildImage(MapEditor_Type::MOUSE_LEFT_PRESS_W_CTRL, focus_addr, mouse_ptr, &msg);
					} else {
						buildImage(MapEditor_Type::MOUSE_LEFT_PRESS, focus_addr, mouse_ptr, &msg);
					}
					need_buildImage = false;

					mainwindow->set_statusBar1(msg);
					mainwindow->set_edited(mapeditor->QRLMap_().push_undo(msg));
					mainwindow->set_windowtitle();
				}
			}
			break;

		case Qt::Key_1: target_argument_num = 0; goto KEY_ARGUMENT;
		case Qt::Key_2: target_argument_num = 1; goto KEY_ARGUMENT;
		case Qt::Key_3: target_argument_num = 2; goto KEY_ARGUMENT;
		case Qt::Key_4: target_argument_num = 3; goto KEY_ARGUMENT;
		case Qt::Key_5: target_argument_num = 4; goto KEY_ARGUMENT;
		case Qt::Key_6: target_argument_num = 5; goto KEY_ARGUMENT;
		case Qt::Key_7: target_argument_num = 6; goto KEY_ARGUMENT;
		case Qt::Key_8: target_argument_num = 7; goto KEY_ARGUMENT;
		case Qt::Key_9: target_argument_num = 8; goto KEY_ARGUMENT;
		  KEY_ARGUMENT:
			{
				QRL::QRLChipNumber num = mapeditor->QRLMap().map(focus_addr);
				if (0 <= num) {
					QRL::QRLChip* const ptr = mapeditor->QRLMap_().search_(num);
					assert (NULL != ptr);
					const QRL::QRLChipLibrary* const chip = qrlmaker->QRLLibrary()->chipSearch(ptr->name);
					assert (NULL != chip);

					QString msg;
					if (mapeditor->edit_argument_of_chip(ptr, chip, target_argument_num)) {
						msg = tr("Edit Argument");
					} else {
						msg = tr("Cannot Be Edited Such an Argument");
					}

					this->needRefresh = true;
					buildImage(MapEditor_Type::MOUSE_NONE);

					mainwindow->set_statusBar1(msg);
					mainwindow->set_edited(mapeditor->QRLMap_().push_undo(msg));
					mainwindow->set_windowtitle();
				}
			}
			break;
		}

		if (need_buildImage) {
			const XY_<int_fast32_t> mouse_ptr = {
				.x = -1,
				.y = -1,
			};
			if ((0 <= mapeditor->Drag_ChipAddr().x) && (0 <= mapeditor->Drag_ChipAddr().y)) {
				mapeditor->Set_SimpleCommand(MapEditor_Type::SIMPLECOMMAND_RECTANGULAR);
				buildImage(MapEditor_Type::MOUSE_LEFT_DRAG, focus_addr, mouse_ptr, NULL);
			} else {
				if (Settings::EditorCommand::NONE != this->mode) {
					buildImage(MapEditor_Type::MOUSE_NONE, focus_addr, mouse_ptr, NULL);
				}
			}
		}
	}

	return;
}
// [ End of File ]
