/**@file
 *
 *@brief	QRL Maker
 *@date		Sun,31 May,2015 - Tue,08 Sep,2015
 *@date		Fri,11 Sep,2015 - Sat,19 Sep,2015
 *@date		Sun,20 Sep,2015
 *@date		Sat,03 Oct,2015 - Sat,10 Oct,2015
 *@date		Tue,13 Oct,2015 - Thu,15 Oct,2015
 *@date		Sun,18 Oct,2015
 *@date		Thu,22 Oct,2015 - Sun,25 Oct,2015
 *@date		Mon,26 Oct,2015
 *@date		Wed,28 Oct,2015
 *@date		Sat,31 Oct,2015
 *@date		Sat,07 Nov,2015
 *@date		Mon,11 Jan,2016
 *@date		Thu,11 Feb,2016 - Sun,14 Feb,2016
 *@date		Sun,16 Apr,2017
 *@date		Sun,09 Jul,2017
 *@date		Sat,15 Jul,2017
 *@date		Sun,16 Jul,2017 - Mon,17 Jul,2017
 *@date		Sun,23 Jul,2017
 *@date		Wed,26 Jul,2017
 *@date		Thu,27 Jul,2017
 *@date		Sat,05 Aug,2017
 *@date		Sat,17 Mar,2018
 *@date		Mon,16 Jul,2018
 *@date		Sat,21 Jul,2018
 *@date		Sat,27 Oct,2018
 *@date		Fri,23 Nov,2018
 *@date		Tue,27 Nov,2018 - Thu,29 Nov,2018
 *@date		Sat,15 Dec,2018
 *@date		Fri,21 Dec,2018
 *@date		Sat,22 Dec,2018
 *@date		Mon,24 Dec,2018
 *@date		Fri,28 Dec,2018
 *@date		Fri,25 Jan,2019
 *@date		Sun,27 Jan,2019
 *@date		Tue,29 Jan,2019
 *@date		Tue,19 Feb,2019
 *@date		Wed,17 Feb,2021
 *@date		Thu,18 Feb,2021
 *@date		Fri,19 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Tue,23 Feb,2021
 *@date		Sun,23 May,2021
 *@date		Sat,29 May,2021
 *@date		Wed,13 Jul,2022
 *@date		Wed,04 Jan,2023
 *@date		Sun,29 Jan,2023
 *@date		Sun,25 Jun,2023
 *
 *@author	Copyright(C)2015-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <assert.h>
#include <cstdint>
#include <stdio.h>
#include <string.h>
#include "settings.h"
#include "qrlmaker.h"
#include "mapeditor.h"
#include "imagebuilder.h"
#include "mytypes.h"
#include "qrl.h"
#include <QPainter>
#include <QImage>
#include <QString>
#include <QStringList>
#if defined(DEBUG)
# include <QPen>
# include <QColor>
#endif

void ImageBuilder_QRLMap_Base::draw_image_raw(QPainter& painter, const QRL::ChipAddrXY& ofs, const QImage* const drawing_img) noexcept
{
	const int_fast32_t ofs_coord_x = static_cast<int_fast32_t>(ofs.x) * this->screen_param.scs.x + this->screen_param.gap.x;
	const int_fast32_t ofs_coord_y = static_cast<int_fast32_t>(ofs.y) * this->screen_param.scs.y + this->screen_param.gap.y;

	painter.drawImage(ofs_coord_x, ofs_coord_y, *drawing_img);
	return;
}

void ImageBuilder_QRLMap_Base::draw_cursor_raw(QPainter& painter, const QRL::ChipAddrXY& ofs, const QImage* const drawing_img) noexcept
{
	const int_fast32_t ofs_coord_x = static_cast<int_fast32_t>(ofs.x) * this->screen_param.scs.x;
	const int_fast32_t ofs_coord_y = static_cast<int_fast32_t>(ofs.y) * this->screen_param.scs.y;

	painter.drawImage(ofs_coord_x, ofs_coord_y, *drawing_img);
	return;
}

void ImageBuilder_QRLMap_Base::draw_chip_raw(QPainter& painter, const int blink_color, const QRL::ChipAddrXY& ofs, const QRL::QRLChipLibrary* const chip, const QRL::QRLChip* const ptr) noexcept
{
	const int_fast32_t cw = static_cast<int_fast32_t>(chip->size.x) * this->screen_param.scs.x - this->screen_param.gap.x + 1;
	const int_fast32_t ch = static_cast<int_fast32_t>(chip->size.y) * this->screen_param.scs.y - this->screen_param.gap.y + 1;
	const QImage img_tmp = chip->image->scaled(cw, ch, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
	draw_image_raw(painter, ofs, &img_tmp);

	if (NULL != ptr) {
		const int_fast32_t ofs_coord_x = static_cast<int_fast32_t>(ofs.x) * this->screen_param.scs.x + this->screen_param.gap.x;
		const int_fast32_t ofs_coord_y = static_cast<int_fast32_t>(ofs.y) * this->screen_param.scs.y + this->screen_param.gap.y;
		for (uint_fast8_t i = 0; i < chip->argument_num; ++i) {
			const int_fast16_t px = static_cast<int_fast16_t>(ofs_coord_x + chip->argument[i].pos.x * this->screen_param.scaling);
			const int_fast16_t py = static_cast<int_fast16_t>(ofs_coord_y + chip->argument[i].pos.y * this->screen_param.scaling);
			const int_fast16_t box_x = static_cast<int_fast16_t>((chip->argument[i].end.x - chip->argument[i].pos.x) * this->screen_param.scaling);
			const int_fast16_t box_y = static_cast<int_fast16_t>((chip->argument[i].end.y - chip->argument[i].pos.y) * this->screen_param.scaling);
			switch (chip->argument[i].type) {
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR:
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::NUM:
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR_NUM:
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::STRING:
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LABEL:
				{
					painter.drawText(px, py, box_x, box_y, (Qt::AlignVCenter | Qt::AlignHCenter), (ptr->arg)[i]);
				}
				break;
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LIST:
				{
					const QString tmp_tr = QCoreApplication::translate(Settings::QRLChipLibrary.tr_context.ChipArgumentList, qPrintable((ptr->arg)[i]), Settings::QRLChipLibrary.tr_disambiguation.ArgumentList);
					painter.drawText(px, py, box_x, box_y, (Qt::AlignVCenter | Qt::AlignHCenter), tmp_tr);
				}
				break;
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::SENTINEL:
				break;
			}

		  #if defined(DEBUG)
			if (blink_color < 0) {
				// Disable blink.
			} else if (0 == blink_color) {
				// Show original image.
			} else if (0 <= blink_color) {
				// Blink.
				const QPen backup = painter.pen();

				painter.setPen(QPen(QColor(blink_color, blink_color, blink_color), 1, Qt::SolidLine));
				painter.drawRect(px, py, box_x, box_y);

				painter.setPen(backup);
			}
		  #endif
		}
	}

	return;
}

void ImageBuilder_QRLMap_Base::draw_arrow_raw(QPainter& painter, const QRL::ChipAddrXY& ofs, QRL::QRLChipArrowAddress arrow_addr, bool arrow_mode) noexcept
{
	if (0 < arrow_addr) {
		const QRL::ChipArrow arrow = static_cast<QRL::ChipArrow>(arrow_addr % 10);
		const int_fast16_t cw = this->screen_param.scs.x + this->screen_param.gap.x + (this->screen_param.overlapping.x * 2) + 1;
		const int_fast16_t ch = this->screen_param.scs.y + this->screen_param.gap.y + (this->screen_param.overlapping.y * 2) + 1;
		const QImage img_tmp = qrlmaker->ArrowImage(arrow_mode, arrow)->scaled(cw, ch, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
		const int_fast32_t ofs_coord_x = static_cast<int_fast32_t>(ofs.x) * this->screen_param.scs.x - this->screen_param.overlapping.x;
		const int_fast32_t ofs_coord_y = static_cast<int_fast32_t>(ofs.y) * this->screen_param.scs.y - this->screen_param.overlapping.y;
		const struct QRL::ChipAddrXY arrow_ofs = {
			.x = static_cast<QRL::ChipAddr>((arrow_addr / Settings::QRLChipLibrary.Limit.Address.Base / 10) % Settings::QRLChipLibrary.Limit.Address.Base),
			.y = static_cast<QRL::ChipAddr>((arrow_addr / 10) % Settings::QRLChipLibrary.Limit.Address.Base),
		};
		painter.drawImage((ofs_coord_x + static_cast<int_fast32_t>(arrow_ofs.x - 1) * this->screen_param.scs.x), (ofs_coord_y + static_cast<int_fast32_t>(arrow_ofs.y - 1) * this->screen_param.scs.y), img_tmp);
	}
	return;
}

void ImageBuilder_QRLMap_Base::draw_chip_w_arrow(QPainter& painter, const int blink_color, const QRL::ChipAddrXY& ofs, const QRL::QRLChip* const ptr) noexcept
{
	const QRL::QRLChipLibrary* const chip = qrlmaker->QRLLibrary()->chipSearch(ptr->name);
	assert (NULL != chip);

	const QRL::ChipAddrXY chip_ofs = (ofs + ptr->addr);
	draw_chip_raw(painter, blink_color, chip_ofs, chip, ptr);
	draw_arrow_raw(painter, chip_ofs, ptr->next_address_condition_true, true);
	draw_arrow_raw(painter, chip_ofs, ptr->next_address_condition_false, false);

	return;
}

void ImageBuilder_QRLMap_Base::draw_chip_wo_arrow(QPainter& painter, const int blink_color, const QRL::ChipAddrXY& ofs, const QRL::QRLChip* const ptr) noexcept
{
	const QRL::QRLChipLibrary* const chip = qrlmaker->QRLLibrary()->chipSearch(ptr->name);
	assert (NULL != chip);

	const QRL::ChipAddrXY chip_ofs = (ofs + ptr->addr);
	draw_chip_raw(painter, blink_color, chip_ofs, chip, ptr);

	return;
}

void ImageBuilder_QRLMap_Base::draw_chip_only_arrow(QPainter& painter, const QRL::ChipAddrXY& ofs, const QRL::QRLChip* const ptr) noexcept
{
	const QRL::ChipAddrXY chip_ofs = (ofs + ptr->addr);
	draw_arrow_raw(painter, chip_ofs, ptr->next_address_condition_true, true);
	draw_arrow_raw(painter, chip_ofs, ptr->next_address_condition_false, false);

	return;
}

void ImageBuilder_QRLMap_Base::draw_chipcursor_raw(QPainter& painter, const QRL::ChipAddrXY& ofs, const QRL::QRLChipLibrary* const chip, const QImage* const drawing_img) noexcept
{
	const struct QRL::ChipAddrXY max = chip->size;
	QStringList::const_iterator it = chip->footprint->begin();
	struct QRL::ChipAddrXY cur;
	for (cur.y = 0; cur.y < max.y; ++cur.y, ++it) {
		for (cur.x = 0; cur.x < max.x; ++cur.x) {
			if (QChar('#') == (*it)[static_cast<int>(cur.x)]) {
				draw_cursor_raw(painter, (ofs + cur), drawing_img);
			}
		}
	}

	return;
}

void ImageBuilder_QRLMap_Base::draw_chipcursor(QPainter& painter, const QRL::ChipAddrXY& ofs, const QRL::QRLChip* const ptr, const QImage* const drawing_img) noexcept
{
	const QRL::QRLChipLibrary* const chip = qrlmaker->QRLLibrary()->chipSearch(ptr->name);
	assert (NULL != chip);

	draw_chipcursor_raw(painter, (ofs + ptr->addr), chip, drawing_img);
	return;
}

void ImageBuilder_QRLMap_Base::draw_chipcursor_highlight(QPainter& painter, const QRL::ChipAddrXY& ofs, const QImage* const drawing_img) noexcept
{
	for (QRL::QRLChipNumber i = 0; i < mapeditor->SelectedList()->maxlist(); ++i) {
		const QRL::QRLChip* const ptr = mapeditor->SelectedList()->ptr(i);
		assert (NULL != ptr);

		if (ptr->chip_number <= 0) {
			continue;
		}

		draw_chipcursor(painter, ofs, ptr, drawing_img);
	}
	return;
}
// [ End of File ]
