/**@file
 *
 *@brief	QRL Maker
 *@date		Sun,11 Oct,2015
 *@date		Mon,12 Oct,2015
 *@date		Sat,17 Oct,2015
 *@date		Thu,22 Oct,2015
 *@date		Thu,29 Oct,2015 - Sat,31 Oct,2015
 *@date		Sat,07 Nov,2015
 *@date		Sun,09 Jul,2017
 *@date		Thu,27 Jul,2017
 *@date		Sun,02 Dec,2018
 *@date		Sat,22 Dec,2018
 *@date		Tue,01 Jan,2019
 *@date		Tue,29 Jan,2019
 *@date		Sun,18 Oct,2020
 *@date		Sun,01 Nov,2020
 *@date		Sun,15 Nov,2020
 *@date		Mon,16 Nov,2020
 *@date		Sun,27 Dec,2020
 *@date		Tue,23 Feb,2021
 *@date		Thu,25 Feb,2021
 *@date		Sat,27 Feb,2021
 *@date		Sun,28 Feb,2021
 *@date		Sun,23 May,2021
 *@date		Sat,29 May,2021
 *@date		Sun,30 May,2021
 *@date		Mon,14 Jun,2021
 *@date		Sun,17 Jul,2022
 *@date		Sat,18 Feb,2023
 *@date		Fri,28 Apr,2023
 *@date		Sat,27 May,2023
 *
 *@author	Copyright(C)2015-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <assert.h>
#include "settings.h"
#include "qrlmaker.h"
#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QString>
#include <QTranslator>
#include <QFileDialog>
#include <QMessageBox>
#include <QDateTime>

bool MainWindow::is_duplicate_dialog(const QString& strFName, bool check_this_whidow) const noexcept
{
	for (auto itr = qrlmaker->window_begin(); itr != qrlmaker->window_end(); ++itr) {
		if (!check_this_whidow && (this == (*itr))) {
		} else {
			if (strFName == (*itr)->filename) {
				QMessageBox msgBox;
				msgBox.setTextFormat(Qt::RichText);
				msgBox.setText(MainWindow::tr("A file '%1' is already being edited.").arg(strFName));
				msgBox.setStandardButtons(QMessageBox::Cancel);
				msgBox.setDefaultButton(QMessageBox::Cancel);

				const int _unused ret = msgBox.exec();
				return true;
			}
		}
	}
	return false;
}

void MainWindow::on_action_New_triggered(void) noexcept
{
	if (is_edited()) {
		const bool ret = question_triggered();
		if (!ret) {
			return;
		}
	}

	const bool ret = ui->form_mainform->action_new();
	if (!ret) {
		Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
						   + tr("Making new data was failed."));
	} else {
		if (0 < filename.length()) {
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("A file '%1' was closed.").arg(filename));
		}

		filename = "";
		set_windowtitle();
	}
	return;
}

void MainWindow::on_actionNew_Window_triggered(void) noexcept
{
	qrlmaker->open_window();
	return;
}

void MainWindow::on_action_Open_triggered(void) noexcept
{
	if (is_edited()) {
		const bool ret = question_triggered();
		if (!ret) {
			return;
		}
	}

	QString strSelectedFilter;
	QString strFName = QFileDialog::getOpenFileName(this, tr( "Filename to Open" ), (Settings::AppConfig.get_DefaultPath()), tr( "QRL-lua Script(QRL*.lua)" ), &strSelectedFilter);
	if (is_duplicate_dialog(strFName, false)){
		return;
	}

	if (!strFName.isEmpty() && QFile::exists(strFName)) {
		const bool ret = ui->form_mainform->action_open(strFName);
		if (!ret) {
			filename = "";
			set_windowtitle();
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("Failed to open a file '%1'.").arg(strFName));
		} else {
			if (0 < filename.length()) {
				Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
								   + tr("A file '%1' was closed.").arg(filename));
			}

			filename = strFName;
			set_windowtitle();
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("A file '%1' was opened.").arg(filename));
		}
	} else {
		Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
						   + tr("A file open was canceled."));
	}
	return;
}

void MainWindow::on_actionOpen_New_Window_triggered(void) noexcept
{
	QString strSelectedFilter;
	QString strFName = QFileDialog::getOpenFileName(this, tr( "Filename to Open in a New Window" ), (Settings::AppConfig.get_DefaultPath()), tr( "QRL-lua Script(QRL*.lua)" ), &strSelectedFilter);
	if (is_duplicate_dialog(strFName, true)){
		return;
	}

	if (!strFName.isEmpty() && QFile::exists(strFName)) {
		const bool ret_new = qrlmaker->open_window();
		if (!ret_new) {
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("A file open in a new window was failed."));
			return;
		}

		MainWindow* const new_w = qrlmaker->window_back_();
		if (NULL == new_w) {
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("A file open in a new window was failed."));
			return;
		}

		const bool ret = new_w->ui->form_mainform->action_open(strFName);
		if (!ret) {
			new_w->filename = "";
			new_w->set_windowtitle();
			new_w->Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
									  + tr("Failed to open a file '%1' in a new window.").arg(strFName));
		} else {
			new_w->filename = strFName;
			new_w->set_windowtitle();
			new_w->Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
																		   + tr("A file '%1' was opened.").arg(new_w->filename));
		}
	} else {
		Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
						   + tr("A file open in a new window was canceled."));
	}
	return;
}

void MainWindow::on_action_Import_triggered(void) noexcept
{
	QString strSelectedFilter;
	QString strFName = QFileDialog::getOpenFileName(this, tr( "Filename to Import" ), (Settings::AppConfig.get_DefaultPath()), tr( "QRL-lua Script(QRL*.lua)" ), &strSelectedFilter);

	if (!strFName.isEmpty() && QFile::exists(strFName)) {
		const bool ret = ui->form_mainform->action_import(strFName);
		if (!ret) {
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("Failed to import a file '%1'.").arg(strFName));
		} else {
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("A file '%1' was imported.").arg(strFName));
		}
	} else {
		Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
						   + tr("A file import was canceled."));
	}
	return;
}

void MainWindow::on_action_Save_triggered(void) noexcept
{
	if (filename.length() <= 0) {
		Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
						   + tr("No filename is specified."));

	  RETRY:
		QString strSelectedFilter;
		QString strFName = QFileDialog::getSaveFileName(this, tr( "Filename to Save" ), (Settings::AppConfig.get_DefaultPath()), tr( "QRL-lua Script(QRL*.lua)" ), &strSelectedFilter, QFileDialog::DontConfirmOverwrite);
		if (is_duplicate_dialog(strFName, false)){
			return;
		}
		{
			QFileInfo fi = QFileInfo(strFName);
			QString path = fi.path();
			QString baseName = fi.completeBaseName();
			QString suffix = fi.suffix();
			if (0 == baseName.length()) {
				Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
								   + tr("A file save was canceled."));
				return;
			}
			if (0 != baseName.indexOf(Settings::QRLData.File.Prefix)) {
				if (0 == suffix.length()) {
					strFName = path + "/" + Settings::QRLData.File.Prefix + baseName;
				} else {
					strFName = path + "/" + Settings::QRLData.File.Prefix + baseName + "." + suffix;
				}
			}
			if (Settings::QRLData.File.QRLMap_Ext != suffix) {
				strFName += "." + Settings::QRLData.File.QRLMap_Ext;
			}
		}
		if (QFile::exists(strFName)) {
			QMessageBox msgBox;
			msgBox.setTextFormat(Qt::RichText);
			msgBox.setText(tr("A file '%1' is already exists.<BR><H3>Are you sure you want to overwrite?</H3>").arg(strFName));
			msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Cancel);
			msgBox.setDefaultButton(QMessageBox::Cancel);

			const int ret = msgBox.exec();

			if (QMessageBox::Save != ret) {
				goto RETRY;
			}
		}

		filename = strFName;
	}

	if (QFile::exists(filename)) {
		QString backup_filename = filename;
		if (backup_filename.endsWith("." + Settings::QRLData.File.QRLMap_Ext)) {
			backup_filename.chop(1 + Settings::QRLData.File.QRLMap_Ext.length());
		}
		backup_filename += "." + Settings::QRLData.File.QRLMap_Backup_Ext;
		if (QFile::exists(backup_filename)) {
			QFile::remove(backup_filename);
		}
		QFile::rename(filename, backup_filename);
	}
	{
		Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
						   + tr("Begin saving to a file '%1'.").arg(filename));
		repaint();
		const bool ret = ui->form_mainform->action_save(filename);
		if (!ret) {
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("Failed to save a file '%1'.").arg(filename));
		} else {
			set_windowtitle();
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("A file '%1' was saved.").arg(filename));
		}
	}
	return;
}

void MainWindow::on_actionSave_as_triggered(void) noexcept
{
  RETRY:
	QString strSelectedFilter;
	QString strFName = QFileDialog::getSaveFileName(this, tr( "Filename to Save" ), (Settings::AppConfig.get_DefaultPath()), tr( "QRL-lua Script(QRL*.lua)" ), &strSelectedFilter, QFileDialog::DontConfirmOverwrite);
	if (is_duplicate_dialog(strFName, false)){
		return;
	}
	{
		QFileInfo fi = QFileInfo(strFName);
		QString path = fi.path();
		QString baseName = fi.completeBaseName();
		QString suffix = fi.suffix();
		if (0 == baseName.length()) {
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("A file save was canceled."));
			return;
		}
		if (0 != baseName.indexOf(Settings::QRLData.File.Prefix)) {
			if (0 == suffix.length()) {
				strFName = path + "/" + Settings::QRLData.File.Prefix + baseName;
			} else {
				strFName = path + "/" + Settings::QRLData.File.Prefix + baseName + "." + suffix;
			}
		}
		if (Settings::QRLData.File.QRLMap_Ext != suffix) {
			strFName += "." + Settings::QRLData.File.QRLMap_Ext;
		}
	}
	if (QFile::exists(strFName)) {
		QMessageBox msgBox;
		msgBox.setTextFormat(Qt::RichText);
		msgBox.setText(tr("A file '%1' is already exists.<BR><H3>Are you sure you want to overwrite?</H3>").arg(strFName));
		msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Cancel);
		msgBox.setDefaultButton(QMessageBox::Cancel);

		const int ret = msgBox.exec();

		if (QMessageBox::Save != ret) {
			goto RETRY;
		}
	}

	if (QFile::exists(filename)) {
		QString backup_filename = filename;
		if (backup_filename.endsWith("." + Settings::QRLData.File.QRLMap_Ext)) {
			backup_filename.chop(1 + Settings::QRLData.File.QRLMap_Ext.length());
		}
		backup_filename += "." + Settings::QRLData.File.QRLMap_Backup_Ext;
		if (QFile::exists(backup_filename)) {
			QFile::remove(backup_filename);
		}
		QFile::rename(filename, backup_filename);
	}
	{
		Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
						   + tr("Begin saving to a file '%1'.").arg(filename));
		repaint();
		const bool ret = ui->form_mainform->action_save(strFName);
		if (!ret) {
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("Failed to save a file '%1'.").arg(strFName));
		} else {
			filename = strFName;
			set_windowtitle();
			Write_ErrorConsole(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							   + tr("A file '%1' was saved.").arg(filename));
		}
	}
	return;
}
// [ End of File ]
