/**@file
 *
 *@brief	QRL Maker
 *@date		Thu,17 Sep,2015 - Sat,19 Sep,2015
 *@date		Sun,20 Sep,2015 - Tue,29 Sep,2015
 *@date		Sat,03 Oct,2015 - Mon,12 Oct,2015
 *@date		Tue,13 Oct,2015 - Thu,15 Oct,2015
 *@date		Sun,18 Oct,2015 - Mon,19 Oct,2015
 *@date		Thu,22 Oct,2015 - Sun,25 Oct,2015
 *@date		Wed,28 Oct,2015
 *@date		Sun,10 Jan,2016
 *@date		Thu,11 Feb,2016 - Sun,14 Feb,2016
 *@date		Sun,16 Apr,2017
 *@date		Sun,23 Jul,2017
 *@date		Fri,04 Aug,2017
 *@date		Fri,18 Aug,2017
 *@date		Sat,03 Mar,2018
 *@date		Sat,27 Oct,2018
 *@date		Tue,27 Nov,2018 - Thu,29 Nov,2018
 *@date		Fri,28 Dec,2018
 *@date		Sun,20 Jan,2019
 *@date		Sat,16 Feb,2019
 *@date		Sun,13 Sep,2020
 *@date		Mon,16 Nov,2020
 *@date		Sun,27 Dec,2020
 *@date		Wed,17 Feb,2021
 *@date		Thu,18 Feb,2021
 *@date		Fri,19 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Mon,22 Feb,2021
 *@date		Sat,29 May,2021
 *@date		Sun,10 Oct,2021
 *@date		Sat,23 Jul,2022
 *
 *@author	Copyright(C)2015-2022 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef QRL_H
#define QRL_H

#include <cstdint>
#include "qrl_type.h"
#include "settings.h"
#include "qrlmaker.h"
#include <QTranslator>
#include <QString>
#include <QStringList>
#include <QVector>
#include <QTextStream>
#include <QImage>
#include "xy_.h"

namespace QRL {

const QRLChipNumber QRLChipEmptyFlag = -1;
const QRLChipNumber QRLChipOutOfMapFlag = -2;

const struct ChipAddrXY AngDir[QRL::ChipArrow_MAX + 1] = {
	{ .x =  0, .y =  0 },
	{ .x = +1, .y =  0 },
	{ .x = +1, .y = +1 },
	{ .x =  0, .y = +1 },
	{ .x = -1, .y = +1 },
	{ .x = -1, .y =  0 },
	{ .x = -1, .y = -1 },
	{ .x =  0, .y = -1 },
	{ .x = +1, .y = -1 },
	{ .x =  0, .y =  0 },
};

struct QRLChipLibrary {
	QString* basename;
	QString* filename;

	QString* header;
	QString* body;

	QString* name;
	struct ChipAddrXY size;
	QStringList* footprint;
	QString* image_name;
	QImage* image;
	QString* chip_coordinates_tag;
	QString* chip_number_tag;
	QString* chip_number;
	QString* chip_footprint_tag;
	QString* next_address_condition_true_coordinates_tag;
	QString* next_address_condition_true_tag;
	QString* next_address_condition_true;
	QString* next_address_condition_false_coordinates_tag;
	QString* next_address_condition_false_tag;
	QString* next_address_condition_false;
	uint_fast8_t argument_num;
	struct Argument {
		QString* comment;
		QString* arg;
		QString* tag;
		QString* tag_raw;
		Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::ArgumentType_Kind type;
		QString* type_selected;
		QStringList* list;
		XY_<int_fast16_t> pos_raw, end_raw;
		XY_<int_fast16_t> pos, end;
	};
	struct Argument* argument;
};

struct QRLChipLibrary_List {
	enum DataType { NONE, LABEL, CHIP } type;

	union {
		struct {
			QString* basename;
			QString* filename;
			QString* label;
		} label;

		struct QRLChipLibrary chip;

	} data;
};


class QRLLibrary
{
	Q_DECLARE_TR_FUNCTIONS(QRLLibrary)

public:
	explicit QRLLibrary(QRLMaker* const q) noexcept;
	~QRLLibrary(void) noexcept;

	bool readLibrary(void) noexcept;
	QVector<QRLChipLibrary_List>::const_iterator libraryBegin(void) const noexcept;
	QVector<QRLChipLibrary_List>::const_iterator libraryEnd(void) const noexcept;
	const QRLChipLibrary* chipSearch(const QString& chip_name) const noexcept;
	const QRLChipLibrary* chipSearch(const QString* const chip_name) const noexcept;
	bool convert_arrowaddress(const QRL::QRLChipLibrary* const chip, const ChipAddrXY& ofs_arg, QRLChipArrowAddress arrow_addr, ChipAddrXY* const xy) const noexcept;
	bool check_arrowaddress(const QRL::QRLChipLibrary* const chip, QRLChipArrowAddress arrow_addr) const noexcept;
	bool check_argumenttype_freestring(Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::ArgumentType_Kind type, const QString& str) const noexcept;
	bool check_argumenttype_list(const QStringList& arg_list, const QString& str) const noexcept;
	Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::ArgumentType_Kind get_argumenttype(const QString& str) const noexcept;
	QString convert_argument_num_to_string(const QString& str) const noexcept;

	bool out_header(QTextStream& out) const noexcept;
	bool out_stamp(QTextStream& out) const noexcept;
	bool out_footer(QTextStream& out) const noexcept;

protected:

private:
	bool read_container(QString** const container, const QString& full_filename) noexcept;
	bool read_label(struct QRLChipLibrary_List* const chip, const QString& basename, const QString& filename, const QString& full_filename) noexcept;
	bool read_chip(struct QRLChipLibrary_List* const chip, const QString& basename, const QString& filename, const QString& full_filename) noexcept;

	QRLMaker* qrlmaker;

	struct {
		QString* header;
		QString* stamp;
		QString* footer;
	} compile_container;
	QVector<QRLChipLibrary_List> library;
};
};



inline QVector<QRL::QRLChipLibrary_List>::const_iterator QRL::QRLLibrary::libraryBegin(void) const noexcept
{
	return library.begin();
}

inline QVector<QRL::QRLChipLibrary_List>::const_iterator QRL::QRLLibrary::libraryEnd(void) const noexcept
{
	return library.end();
}

#endif // QRL_H
// [ End of File ]
