/**@file
 *
 *@brief	QRL Maker
 *@date		Thu,17 Sep,2015 - Sat,19 Sep,2015
 *@date		Sun,20 Sep,2015 - Tue,29 Sep,2015
 *@date		Sat,03 Oct,2015
 *@date		Mon,12 Oct,2015
 *@date		Tue,13 Oct,2015
 *@date		Mon,19 Oct,2015
 *@date		Thu,22 Oct,2015
 *@date		Mon,26 Oct,2015 - Thu,29 Oct,2015
 *@date		Sun,22 Nov,2015
 *@date		Sun,10 Jan,2016
 *@date		Thu,11 Feb,2016
 *@date		Sun,09 Jul,2017
 *@date		Fri,04 Aug,2017
 *@date		Fri,18 Aug,2017
 *@date		Sat,03 Mar,2018
 *@date		Sat,27 Oct,2018
 *@date		Thu,29 Nov,2018
 *@date		Sun,20 Jan,2019
 *@date		Sat,16 Feb,2019
 *@date		Sun,13 Sep,2020
 *@date		Wed,16 Sep,2020
 *@date		Wed,28 Oct,2020
 *@date		Sat,31 Oct,2020
 *@date		Sun,15 Nov,2020
 *@date		Mon,16 Nov,2020
 *@date		Sun,27 Dec,2020
 *@date		Fri,19 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Mon,22 Feb,2021
 *@date		Sat,29 May,2021
 *@date		Sun,10 Oct,2021
 *@date		Sat,30 Oct,2021
 *@date		Sat,23 Jul,2022
 *@date		Wed,04 Jan,2023
 *@date		Mon,09 Jan,2023
 *@date		Sun,26 Mar,2023
 *@date		Sat,22 Apr,2023
 *
 *@author	Copyright(C)2015-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#if defined(DEBUG)
# include <stdio.h>
# if !defined(__STDC_FORMAT_MACROS)
#  define    __STDC_FORMAT_MACROS
# endif
# include <inttypes.h>
#endif
#include <math.h>
#include <assert.h>
#include "settings.h"
#include "qrl.h"
#include <QString>
#include <QStringList>
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
# include <QRegularExpression>
#else
# include <QRegExp>
#endif
#include <QTranslator>
#include <QFile>
#include <QDir>
#include <QTextStream>
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
# include <QStringConverter>
#else
# include <QTextCodec>
#endif

bool QRL::QRLLibrary::read_container(QString** const container, const QString& full_filename) noexcept
{
	assert (NULL != container);

	bool err = false;

	QFile file(full_filename);
	if (!file.open(QIODevice::ReadOnly)) {
		qrlmaker->errmsg.push_back(tr("File Open Error : '%1'").arg(full_filename));

		err = true;
	} else {
		QTextStream in(&file);
	  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
		in.setEncoding(QStringConverter::Utf8);
	  #else
		QTextCodec* const codec = QTextCodec::codecForName("UTF-8");
		if (codec) {
			in.setCodec(codec);
		}
	  #endif

		(*container) = new(std::nothrow) QString("");
		assert (NULL != (*container));
		(**container) = in.readAll();
	}

	return !err;
}

bool QRL::QRLLibrary::read_label(struct QRL::QRLChipLibrary_List* const chip, const QString& basename, const QString& filename, const QString& full_filename) noexcept
{
	assert (NULL != chip);

	bool err = false;

	{
		chip->data.label.basename = new(std::nothrow) QString(basename);
		assert (NULL != chip->data.label.basename);
	}
	{
		chip->data.label.filename = new(std::nothrow) QString(filename);
		assert (NULL != chip->data.label.filename);
	}

	QFile file(full_filename);
	if (!file.open(QIODevice::ReadOnly)) {
		qrlmaker->errmsg.push_back(tr("File Open Error : '%1'").arg(full_filename));
		chip->type = QRL::QRLChipLibrary_List::NONE;

		err = true;
	} else {
		chip->type = QRL::QRLChipLibrary_List::LABEL;

		{
			QTextStream in(&file);
		  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
			in.setEncoding(QStringConverter::Utf8);
		  #else
			QTextCodec* const codec = QTextCodec::codecForName("UTF-8");
			if (codec) {
				in.setCodec(codec);
			}
		  #endif

			const QString str = in.readLine(0);
			chip->data.label.label = new(std::nothrow) QString(str);
			assert (NULL != chip->data.label.label);
		}
	}

	return !err;
}

bool QRL::QRLLibrary::read_chip(struct QRL::QRLChipLibrary_List* const chip, const QString& basename, const QString& filename, const QString& full_filename) noexcept
{
	assert (NULL != chip);

	bool err = false;

	{
		chip->data.chip.basename = new(std::nothrow) QString(basename);
		assert (NULL != chip->data.chip.basename);
	}
	{
		chip->data.chip.filename = new(std::nothrow) QString(filename);
		assert (NULL != chip->data.chip.filename);
	}

	QFile file(full_filename);
	if (!file.open(QIODevice::ReadOnly)) {
		qrlmaker->errmsg.push_back(tr("File Open Error : '%1'").arg(full_filename));
		chip->type = QRL::QRLChipLibrary_List::NONE;

		err = true;
	} else {
		chip->type = QRL::QRLChipLibrary_List::CHIP;

		chip->data.chip.header = NULL;
		chip->data.chip.body = NULL;

		chip->data.chip.name = NULL;

		chip->data.chip.size.x = 0;
		chip->data.chip.size.y = 0;

		chip->data.chip.footprint = NULL;

		chip->data.chip.image_name = NULL;
		{
			QFileInfo fi = QFileInfo(filename);
			const QString baseName = fi.completeBaseName();
			chip->data.chip.image_name = new(std::nothrow) QString(baseName + Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Image_ext);
			assert (NULL != chip->data.chip.image_name);
		}

		chip->data.chip.image = NULL;

		chip->data.chip.chip_coordinates_tag = NULL;
		chip->data.chip.chip_number_tag = NULL;
		chip->data.chip.chip_number = NULL;
		chip->data.chip.chip_footprint_tag = NULL;
		chip->data.chip.next_address_condition_true_coordinates_tag = NULL;
		chip->data.chip.next_address_condition_true_tag = NULL;
		chip->data.chip.next_address_condition_true = NULL;
		chip->data.chip.next_address_condition_false_coordinates_tag = NULL;
		chip->data.chip.next_address_condition_false_tag = NULL;
		chip->data.chip.next_address_condition_false = NULL;
		chip->data.chip.argument_num = 0;
		chip->data.chip.argument = NULL;

		{	// Read the File
			chip->data.chip.header = new(std::nothrow) QString("");
			assert (NULL != chip->data.chip.header);
			chip->data.chip.body = new(std::nothrow) QString("");
			assert (NULL != chip->data.chip.body);

			enum LineType { NONE, HEADER, BODY } type = NONE;
			QTextStream in(&file);
		  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
			in.setEncoding(QStringConverter::Utf8);
		  #else
			QTextCodec* const codec = QTextCodec::codecForName("UTF-8");
			if (codec) {
				in.setCodec(codec);
			}
		  #endif

			bool finished = false;
			while (!in.atEnd()) {
				const QString str = in.readLine(0);

				if (0 < str.indexOf(Settings::QRLChipLibrary.Syntax.Begin)) {
					type = HEADER;
				} else if (0 < str.indexOf(Settings::QRLChipLibrary.Syntax.Body)) {
					type = BODY;
				} else if (0 < str.indexOf(Settings::QRLChipLibrary.Syntax.End)) {
					type = NONE;
					*(chip->data.chip.body) += str + "\n";
					finished = true;
				} else {
				}
				switch (type) {
				case NONE:
					break;
				case HEADER:
					*(chip->data.chip.header) += str + "\n";
					break;
				case BODY:
					*(chip->data.chip.body) += str + "\n";
					break;
				}
			}

			if (0 == chip->data.chip.header->length()) {
				qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.Begin));
				err = true;
			}
			if (0 == chip->data.chip.body->length()) {
				qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.Body));
				err = true;
			}
			if (!finished) {
				qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.End));
				err = true;
			}
		}

		{	// Parse the Header
			QStringList list = chip->data.chip.header->split("\n");
			for (int n = 0; n < list.size(); ++n) {
			  RETRY_PARSE:
				bool hit = false;

				list[n] = list[n].trimmed();
				list[n] = list[n].simplified();

				if (!hit) {
					if (list[n].length() < 1) {
						hit = true;
					}
				}

				if (!hit) {
					if (0 < list[n].indexOf(Settings::QRLChipLibrary.Syntax.Begin)) {
						hit = true;
					}
				}

				if (!hit) {
					const int name_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.name);
					if (0 <= name_ptr) {
						chip->data.chip.name = new(std::nothrow) QString(list[n].mid(name_ptr + Settings::QRLChipLibrary.Syntax.Header.name.length()).trimmed());
						assert (NULL != chip->data.chip.name);
						hit = true;
					}
				}

				if (!hit) {
					const int size_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.size);
					if (0 <= size_ptr) {
						const QString str = list[n].mid(size_ptr + Settings::QRLChipLibrary.Syntax.Header.size.length()).trimmed();
						QStringList list2 = str.split(",");
						if (2 == list2.size()) {
							chip->data.chip.size.x = list2[0].toInt();
							chip->data.chip.size.y = list2[1].toInt();
							hit = true;
						}
					}
				}

				if (!hit) {
					const int footprint_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.footprint);
					if (0 <= footprint_ptr) {
						hit = true;
						if (list.size() <= (n + 1 + chip->data.chip.size.y)) {
							qrlmaker->errmsg.push_back(tr("%1 : There is too little linage of the footprint.").arg(full_filename));
							hit = false;
							err = true;

						} else {
							chip->data.chip.footprint = new(std::nothrow) QStringList();
							assert (NULL != chip->data.chip.footprint);

							for (QRL::ChipAddr i = 0; i < chip->data.chip.size.y; ++i) {
								++n;
								list[n] = list[n].trimmed();
								if (0 != list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.footprint_header_substance)) {
									qrlmaker->errmsg.push_back(tr("%1 : Syntax error at the footprint.").arg(full_filename));
									hit = false;
									err = true;
									goto RETRY_PARSE;
								}
								QString str2 = list[n].mid(Settings::QRLChipLibrary.Syntax.Header.footprint_header_substance.length());
								if (str2.length() < chip->data.chip.size.x) {
									for (QRL::ChipAddr j = str2.length(); j < chip->data.chip.size.x; ++j) {
										str2.append(' ');
									}
								} else if (chip->data.chip.size.x < str2.length()) {
									qrlmaker->errmsg.push_back(tr("%1 : The length of the footprint does not accord.").arg(full_filename));
									hit = false;
									err = true;
									goto RETRY_PARSE;
								}
								chip->data.chip.footprint->append(str2);
							}

						}
					}
				}

				if (!hit) {
					const int address_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.address);
					if (0 <= address_ptr) {
						hit = true;
					}
				}

				if (!hit) {
					const int chip_coordinates_tag_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.chip_coordinates_tag);
					if (0 <= chip_coordinates_tag_ptr) {
						chip->data.chip.chip_coordinates_tag = new(std::nothrow) QString(list[n].mid(chip_coordinates_tag_ptr + Settings::QRLChipLibrary.Syntax.Header.chip_coordinates_tag.length()).trimmed());
						assert (NULL != chip->data.chip.chip_coordinates_tag);
						if (chip->data.chip.chip_coordinates_tag->length() < 1) {
							delete chip->data.chip.chip_coordinates_tag;
							chip->data.chip.chip_coordinates_tag = NULL;
						}
						hit = true;
					}
				}

				if (!hit) {
					const int chip_number_tag_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.chip_number_tag);
					if (0 <= chip_number_tag_ptr) {
						chip->data.chip.chip_number_tag = new(std::nothrow) QString(list[n].mid(chip_number_tag_ptr + Settings::QRLChipLibrary.Syntax.Header.chip_number_tag.length()).trimmed());
						assert (NULL != chip->data.chip.chip_number_tag);
						if (chip->data.chip.chip_number_tag->length() < 1) {
							delete chip->data.chip.chip_number_tag;
							chip->data.chip.chip_number_tag = NULL;
						}
						hit = true;
					}
				}

				if (!hit) {
					const int chip_footprint_tag_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.chip_footprint_tag);
					if (0 <= chip_footprint_tag_ptr) {
						chip->data.chip.chip_footprint_tag = new(std::nothrow) QString(list[n].mid(chip_footprint_tag_ptr + Settings::QRLChipLibrary.Syntax.Header.chip_footprint_tag.length()).trimmed());
						assert (NULL != chip->data.chip.chip_footprint_tag);
						if (chip->data.chip.chip_footprint_tag->length() < 1) {
							delete chip->data.chip.chip_footprint_tag;
							chip->data.chip.chip_footprint_tag = NULL;
						}
						hit = true;
					}
				}

				if (!hit) {
					const int chip_number_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.chip_number);
					if (0 <= chip_number_ptr) {
						chip->data.chip.chip_number = new(std::nothrow) QString(list[n].mid(chip_number_ptr + Settings::QRLChipLibrary.Syntax.Header.chip_number.length()).trimmed());
						assert (NULL != chip->data.chip.chip_number);
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_true_coordinates_tag_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true_coordinates_tag);
					if (0 <= next_address_condition_true_coordinates_tag_ptr) {
						chip->data.chip.next_address_condition_true_coordinates_tag = new(std::nothrow) QString(list[n].mid(next_address_condition_true_coordinates_tag_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true_coordinates_tag.length()).trimmed());
						assert (NULL != chip->data.chip.next_address_condition_true_coordinates_tag);
						if (chip->data.chip.next_address_condition_true_coordinates_tag->length() < 1) {
							delete chip->data.chip.next_address_condition_true_coordinates_tag;
							chip->data.chip.next_address_condition_true_coordinates_tag = NULL;
						}
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_true_tag_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true_tag);
					if (0 <= next_address_condition_true_tag_ptr) {
						chip->data.chip.next_address_condition_true_tag = new(std::nothrow) QString(list[n].mid(next_address_condition_true_tag_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true_tag.length()).trimmed());
						assert (NULL != chip->data.chip.next_address_condition_true_tag);
						if (chip->data.chip.next_address_condition_true_tag->length() < 1) {
							delete chip->data.chip.next_address_condition_true_tag;
							chip->data.chip.next_address_condition_true_tag = NULL;
						}
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_true_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true);
					if (0 <= next_address_condition_true_ptr) {
						chip->data.chip.next_address_condition_true = new(std::nothrow) QString(list[n].mid(next_address_condition_true_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true.length()).trimmed());
						assert (NULL != chip->data.chip.next_address_condition_true);
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_false_coordinates_tag_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false_coordinates_tag);
					if (0 <= next_address_condition_false_coordinates_tag_ptr) {
						chip->data.chip.next_address_condition_false_coordinates_tag = new(std::nothrow) QString(list[n].mid(next_address_condition_false_coordinates_tag_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false_coordinates_tag.length()).trimmed());
						assert (NULL != chip->data.chip.next_address_condition_false_coordinates_tag);
						if (chip->data.chip.next_address_condition_false_coordinates_tag->length() < 1) {
							delete chip->data.chip.next_address_condition_false_coordinates_tag;
							chip->data.chip.next_address_condition_false_coordinates_tag = NULL;
						}
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_false_tag_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false_tag);
					if (0 <= next_address_condition_false_tag_ptr) {
						chip->data.chip.next_address_condition_false_tag = new(std::nothrow) QString(list[n].mid(next_address_condition_false_tag_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false_tag.length()).trimmed());
						assert (NULL != chip->data.chip.next_address_condition_false_tag);
						if (chip->data.chip.next_address_condition_false_tag->length() < 1) {
							delete chip->data.chip.next_address_condition_false_tag;
							chip->data.chip.next_address_condition_false_tag = NULL;
						}
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_false_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false);
					if (0 <= next_address_condition_false_ptr) {
						chip->data.chip.next_address_condition_false = new(std::nothrow) QString(list[n].mid(next_address_condition_false_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false.length()).trimmed());
						assert (NULL != chip->data.chip.next_address_condition_false);
						hit = true;
					}
				}

				if (!hit) {
					const int argument_num_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_num);
					if (0 <= argument_num_ptr) {
						const QString str = list[n].mid(argument_num_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_num.length()).trimmed();
						chip->data.chip.argument_num = str.toInt();
						chip->data.chip.argument = new(std::nothrow) QRL::QRLChipLibrary::Argument[chip->data.chip.argument_num];
						assert (NULL != chip->data.chip.argument);

						for (uint_fast8_t i = 0; i < chip->data.chip.argument_num; ++i) {
							chip->data.chip.argument[i].comment = NULL;
							chip->data.chip.argument[i].arg = NULL;
							chip->data.chip.argument[i].tag = NULL;
							chip->data.chip.argument[i].tag_raw = NULL;
							chip->data.chip.argument[i].type = Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::SENTINEL;
							chip->data.chip.argument[i].type_selected = NULL;
							chip->data.chip.argument[i].list = NULL;
							chip->data.chip.argument[i].pos.x = chip->data.chip.argument[i].pos_raw.x = 0;
							chip->data.chip.argument[i].pos.y = chip->data.chip.argument[i].pos_raw.y = 0;
							chip->data.chip.argument[i].end.x = chip->data.chip.argument[i].end_raw.x = 0;
							chip->data.chip.argument[i].end.y = chip->data.chip.argument[i].end_raw.y = 0;
						}
						hit = true;
					}
				}

				if (!hit) {
					const int argument_comment_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_comment);
					const int argument_closebracket_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_closebracket);
					if ((0 <= argument_comment_ptr) && (0 <= argument_closebracket_ptr)) {
						const QString str = list[n].mid(argument_comment_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_comment.length(), (argument_closebracket_ptr - (argument_comment_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_comment.length()))).trimmed();
						const uint_fast8_t i = str.toInt() - 1;
						if ((0 <= i) && (i < chip->data.chip.argument_num)) {
							chip->data.chip.argument[i].comment = new(std::nothrow) QString(list[n].mid(argument_closebracket_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket.length()).trimmed());
							assert (NULL != chip->data.chip.argument[i].comment);
							hit = true;
						}
					}
				}

				if (!hit) {
					const int argument_arg_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_arg);
					const int argument_closebracket_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_closebracket);
					if ((0 <= argument_arg_ptr) && (0 <= argument_closebracket_ptr)) {
						const QString str = list[n].mid(argument_arg_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_arg.length(), (argument_closebracket_ptr - (argument_arg_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_arg.length()))).trimmed();
						const uint_fast8_t i = str.toInt() - 1;
						if ((0 <= i) && (i < chip->data.chip.argument_num)) {
							chip->data.chip.argument[i].arg = new(std::nothrow) QString(list[n].mid(argument_closebracket_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket.length()).trimmed());
							assert (NULL != chip->data.chip.argument[i].arg);
							hit = true;
						}
					}
				}

				if (!hit) {
					const int argument_tag_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_tag);
					const int argument_closebracket_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_closebracket);
					if ((0 <= argument_tag_ptr) && (0 <= argument_closebracket_ptr)) {
						const QString str = list[n].mid(argument_tag_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_tag.length(), (argument_closebracket_ptr - (argument_tag_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_tag.length()))).trimmed();
						const uint_fast8_t i = str.toInt() - 1;
						if ((0 <= i) && (i < chip->data.chip.argument_num)) {
							chip->data.chip.argument[i].tag = new(std::nothrow) QString(list[n].mid(argument_closebracket_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket.length()).trimmed());
							assert (NULL != chip->data.chip.argument[i].tag);
							hit = true;
						}
					}
				}

				if (!hit) {
					const int argument_tag_raw_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_tag_raw);
					const int argument_closebracket_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_closebracket);
					if ((0 <= argument_tag_raw_ptr) && (0 <= argument_closebracket_ptr)) {
						const QString str = list[n].mid(argument_tag_raw_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_tag_raw.length(), (argument_closebracket_ptr - (argument_tag_raw_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_tag_raw.length()))).trimmed();
						const uint_fast8_t i = str.toInt() - 1;
						if ((0 <= i) && (i < chip->data.chip.argument_num)) {
							chip->data.chip.argument[i].tag_raw = new(std::nothrow) QString(list[n].mid(argument_closebracket_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket.length()).trimmed());
							assert (NULL != chip->data.chip.argument[i].tag_raw);
							hit = true;
						}
					}
				}

				if (!hit) {
					const int argument_type_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_type);
					const int argument_closebracket_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_closebracket);
					if ((0 <= argument_type_ptr) && (0 <= argument_closebracket_ptr)) {
						const QString str = list[n].mid(argument_type_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_type.length(), (argument_closebracket_ptr - (argument_type_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_type.length()))).trimmed();
						const uint_fast8_t i = str.toInt() - 1;
						if ((0 <= i) && (i < chip->data.chip.argument_num)) {
							const QString str2 = list[n].mid(argument_closebracket_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket.length()).trimmed();
							for (int j = 0; Settings::QRLChipLibrary.Syntax.Header.argument_type_kind[j].kind < Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::SENTINEL; ++j) {
								if (str2 == Settings::QRLChipLibrary.Syntax.Header.argument_type_kind[j].label) {
									chip->data.chip.argument[i].type = Settings::QRLChipLibrary.Syntax.Header.argument_type_kind[j].kind;
									hit = true;
									break;
								}
							}
						}
					}
				}

				if (!hit) {
					const int argument_type_selected_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_type_selected);
					const int argument_closebracket_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_closebracket);
					if ((0 <= argument_type_selected_ptr) && (0 <= argument_closebracket_ptr)) {
						const QString str = list[n].mid(argument_type_selected_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_type_selected.length(), (argument_closebracket_ptr - (argument_type_selected_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_type_selected.length()))).trimmed();
						const uint_fast8_t i = str.toInt() - 1;
						if ((0 <= i) && (i < chip->data.chip.argument_num)) {
							chip->data.chip.argument[i].type_selected = new(std::nothrow) QString(list[n].mid(argument_closebracket_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket.length()).trimmed());
							assert (NULL != chip->data.chip.argument[i].type_selected);
							hit = true;
						}
					}
				}

				if (!hit) {
					const int argument_list_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_list);
					const int argument_closebracket_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_closebracket);
					if ((0 <= argument_list_ptr) && (0 <= argument_closebracket_ptr)) {
						const QString str = list[n].mid(argument_list_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_list.length(), (argument_closebracket_ptr - (argument_list_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_list.length()))).trimmed();
						const uint_fast8_t i = str.toInt() - 1;
						if ((0 <= i) && (i < chip->data.chip.argument_num)) {
							chip->data.chip.argument[i].list = new(std::nothrow) QStringList();
							assert (NULL != chip->data.chip.argument[i].list);

							const QString str2 = list[n].mid(argument_closebracket_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket.length()).trimmed();
							QStringList list = str2.split(",");

							for (int j = 0; j < list.size(); ++j) {
								list[j] = list[j].trimmed();
								list[j] = list[j].simplified();
								chip->data.chip.argument[i].list->append(list[j]);
							}

							hit = true;
						}
					}
				}

				if (!hit) {
					const int argument_position_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_position);
					const int argument_closebracket_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_closebracket);
					if ((0 <= argument_position_ptr) && (0 <= argument_closebracket_ptr)) {
						const QString str = list[n].mid(argument_position_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_position.length(), (argument_closebracket_ptr - (argument_position_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_position.length()))).trimmed();
						const uint_fast8_t i = str.toInt() - 1;
						if ((0 <= i) && (i < chip->data.chip.argument_num)) {
							const QString str2 = list[n].mid(argument_closebracket_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket.length()).trimmed();
							QStringList list2 = str2.split(",");
							if (4 == list2.size()) {
								chip->data.chip.argument[i].pos.x = chip->data.chip.argument[i].pos_raw.x = list2[0].toInt();
								chip->data.chip.argument[i].pos.y = chip->data.chip.argument[i].pos_raw.y = list2[1].toInt();
								chip->data.chip.argument[i].end.x = chip->data.chip.argument[i].end_raw.x = list2[2].toInt();
								chip->data.chip.argument[i].end.y = chip->data.chip.argument[i].end_raw.y = list2[3].toInt();
								hit = true;
							}
						}
					}
				}
				if (!hit) {
					const int argument_geometry_ptr = list[n].indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_geometry);
					if (0 <= argument_geometry_ptr) {
					  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
						QRegularExpression geometry(Settings::QRLChipLibrary.Syntax.Header.argument_geometry_regex);
						auto match = geometry.match(list[n].mid(argument_geometry_ptr).toLower().trimmed());
						if (match.hasMatch()) {
							const uint_fast8_t i = match.captured(1).toInt() - 1;
							if ((0 <= i) && (i < chip->data.chip.argument_num)) {
								chip->data.chip.argument[i].pos.x = chip->data.chip.argument[i].pos_raw.x = match.captured(2).toInt();
								chip->data.chip.argument[i].pos.y = chip->data.chip.argument[i].pos_raw.y = match.captured(3).toInt();
								chip->data.chip.argument[i].end.x = chip->data.chip.argument[i].end_raw.x = (match.captured(2).toInt() + match.captured(4).toInt());
								chip->data.chip.argument[i].end.y = chip->data.chip.argument[i].end_raw.y = (match.captured(3).toInt() + match.captured(5).toInt());
								hit = true;
							}
						}
					  #else
						QRegExp geometry(Settings::QRLChipLibrary.Syntax.Header.argument_geometry_regex);
						if (geometry.exactMatch(list[n].mid(argument_geometry_ptr).toLower().trimmed())) {
							const uint_fast8_t i = geometry.cap(1).toInt() - 1;
							if ((0 <= i) && (i < chip->data.chip.argument_num)) {
								chip->data.chip.argument[i].pos.x = chip->data.chip.argument[i].pos_raw.x = geometry.cap(2).toInt();
								chip->data.chip.argument[i].pos.y = chip->data.chip.argument[i].pos_raw.y = geometry.cap(3).toInt();
								chip->data.chip.argument[i].end.x = chip->data.chip.argument[i].end_raw.x = (geometry.cap(2).toInt() + geometry.cap(4).toInt());
								chip->data.chip.argument[i].end.y = chip->data.chip.argument[i].end_raw.y = (geometry.cap(3).toInt() + geometry.cap(5).toInt());
								hit = true;
							}
						}
					  #endif
					}
				}

				if (!hit) {
					qrlmaker->errmsg.push_back(tr("%1 : Unknown Header : '%2'").arg(full_filename).arg(list[n]));
				}
			}
		}

	}

	{
		if (NULL != chip->data.chip.image_name) {
			const QString full_iconname = Settings::QRLChipLibrary.LibraryFile.LuaChipDir + *(chip->data.chip.image_name);

			chip->data.chip.image = new(std::nothrow) QImage();
			assert (NULL != chip->data.chip.image);
			const bool ret = chip->data.chip.image->load(full_iconname);
			if (!ret) {
				qrlmaker->errmsg.push_back(tr("ERROR: Failed to load '%1'.").arg(full_iconname));
				err = true;
			} else {
			  #if defined(DEBUG)
				const uint_fast16_t required_w = chip->data.chip.size.x * (Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize) - Settings::BuildImage.Chip.GapSize + 1;
				const uint_fast16_t required_h = chip->data.chip.size.y * (Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize) - Settings::BuildImage.Chip.GapSize + 1;
				const uint_fast16_t actual_w = chip->data.chip.image->width();
				const uint_fast16_t actual_h = chip->data.chip.image->height();
				if ((required_w != actual_w) || (required_h != actual_h)) {
					fprintf(stderr, "Image '%s': Size mismatch. Required:(%3" PRIu16 ",%3" PRIu16 "), Actual:(%3" PRIu16 ",%3" PRIu16 ").\n", qPrintable(*(chip->data.chip.image_name)), required_w, required_h, actual_w, actual_h);
				}
			  #endif
				assert (0 < chip->data.chip.size.x);
				assert (0 < chip->data.chip.size.y);
				assert (0 < chip->data.chip.image->width());
				assert (0 < chip->data.chip.image->height());
				const double w_ratio = static_cast<double>(chip->data.chip.size.x * (Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize) - Settings::BuildImage.Chip.GapSize) / chip->data.chip.image->width();
				const double h_ratio = static_cast<double>(chip->data.chip.size.y * (Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize) - Settings::BuildImage.Chip.GapSize) / chip->data.chip.image->height();
				for (uint_fast8_t i = 0; i < chip->data.chip.argument_num; ++i) {
					chip->data.chip.argument[i].pos.x = floor((chip->data.chip.argument[i].pos_raw.x - 1) * w_ratio) + 1;
					chip->data.chip.argument[i].pos.y = floor((chip->data.chip.argument[i].pos_raw.y - 1) * h_ratio) + 1;
					chip->data.chip.argument[i].end.x = ceil((chip->data.chip.argument[i].end_raw.x + 1) * w_ratio) - 1;
					chip->data.chip.argument[i].end.y = ceil((chip->data.chip.argument[i].end_raw.y + 1) * h_ratio) - 1;
				}
			}
		}
	}

	// Error Check
	if (!err) {
		if (NULL == chip->data.chip.name) {
			qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.Header.name));
			err = true;
		}
		if ((0 == chip->data.chip.size.x) || (0 == chip->data.chip.size.y)) {
			qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.Header.size));
			err = true;
		}
		if (NULL == chip->data.chip.footprint) {
			qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.Header.footprint));
			err = true;
		}

		for (uint_fast8_t i = 0; i < chip->data.chip.argument_num; ++i) {
			if (NULL == chip->data.chip.argument[i].arg) {
				qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.Header.argument_arg + QString::number(i + 1) + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket));
				err = true;
			}
			switch (chip->data.chip.argument[i].type) {
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR:
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::NUM:
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR_NUM:
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::STRING:
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LABEL:
				{
					if ((0 == chip->data.chip.argument[i].pos_raw.x)
						|| (0 == chip->data.chip.argument[i].pos_raw.y)
						|| (0 == chip->data.chip.argument[i].end_raw.x)
						|| (0 == chip->data.chip.argument[i].end_raw.y))
					{
						qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.Header.argument_geometry + QString::number(i + 1) + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket));
						err = true;
					}
				}
				break;
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LIST:
				{
					if (NULL == chip->data.chip.argument[i].list) {
						qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.Header.argument_list + QString::number(i + 1) + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket));
						err = true;
					}
				}
				break;
			case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::SENTINEL:
				{
					qrlmaker->errmsg.push_back(tr("%1 : Missing Header : '%2'").arg(full_filename).arg(Settings::QRLChipLibrary.Syntax.Header.argument_type + QString::number(i + 1) + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket));
					err = true;
				}
				break;
			}
		}
	}

	return !err;
}

bool QRL::QRLLibrary::readLibrary(void) noexcept
{
	bool err = false;

	{
		const QString full_filename = Settings::QRLChipLibrary.LibraryFile.LuaChipDir + Settings::QRLChipLibrary.LibraryFile.LuaContainer_Filename_Header;
		if (!read_container(&(this->compile_container.header), full_filename)) {
			err |= true;
		}
	}
	{
		const QString full_filename = Settings::QRLChipLibrary.LibraryFile.LuaChipDir + Settings::QRLChipLibrary.LibraryFile.LuaContainer_Filename_Stamp;
		if (!read_container(&(this->compile_container.stamp), full_filename)) {
			err |= true;
		}
	}
	{
		const QString full_filename = Settings::QRLChipLibrary.LibraryFile.LuaChipDir + Settings::QRLChipLibrary.LibraryFile.LuaContainer_Filename_Footer;
		if (!read_container(&(this->compile_container.footer), full_filename)) {
			err |= true;
		}
	}
	{
		QStringList filenameFilter;
		{
			const QString label_filename = Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Header + Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Wildcard + QString(Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Label_ext);
			filenameFilter << label_filename;

			const QString chip_filename = Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Header + Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Wildcard + QString(Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Chip_ext);
			filenameFilter << chip_filename;
		}
		QDir dataDir( Settings::QRLChipLibrary.LibraryFile.LuaChipDir );
		const QFileInfoList fileList = dataDir.entryInfoList( filenameFilter, QDir::Files );

		for (int i = 0; i < fileList.size(); ++i )
		{
			struct QRL::QRLChipLibrary_List chip;
			chip.type = QRL::QRLChipLibrary_List::NONE;

			const QString full_filename = Settings::QRLChipLibrary.LibraryFile.LuaChipDir + fileList[i].fileName();

			bool ret_err = false;
			if (0 == Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Label_ext.compare(fileList[i].fileName().right(Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Label_ext.length()), Qt::CaseInsensitive)) {
				// Separator Label
				ret_err |= !read_label(&chip, fileList[i].completeBaseName(), fileList[i].fileName(), full_filename);

			} else if (0 == Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Chip_ext.compare(fileList[i].fileName().right(Settings::QRLChipLibrary.LibraryFile.LuaChip_Filename_Chip_ext.length()), Qt::CaseInsensitive)) {
				// Chip Item
				ret_err |= !read_chip(&chip, fileList[i].completeBaseName(), fileList[i].fileName(), full_filename);

			} else {
				chip.type = QRL::QRLChipLibrary_List::NONE;
			}

			if (!ret_err) {
				library.push_back(chip);
			}
			err |= ret_err;
		}
	}

	return !err;
}
// [ End of File ]
