/**@file
 *
 *@brief	QRL Maker
 *@date		Thu,18 Feb,2021
 *@date		Fri,19 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Mon,22 Feb,2021
 *@date		Tue,23 Feb,2021
 *@date		Fri,26 Feb,2021
 *@date		Sun,23 May,2021
 *@date		Mon,24 May,2021
 *@date		Sat,29 May,2021
 *@date		Sun,30 May,2021
 *@date		Thu,03 Jun,2021
 *@date		Sun,26 Mar,2023
 *@date		Sat,27 May,2023
 *
 *@author	Copyright(C)2021-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#if defined(DEBUG)
# include <stdio.h>
# if !defined(__STDC_FORMAT_MACROS)
#  define    __STDC_FORMAT_MACROS
# endif
# include <inttypes.h>
#endif
#include <assert.h>
#include <cstdint>
#include <vector>
#include "settings.h"
#include "qrlmaker.h"
#include "qrl.h"
#include "mapeditor_clipboard.h"
#include "mainwindow.h"
#include <QString>
#include <QStringList>
#include <QTranslator>
#include <QMessageBox>

QRLMaker_Base::QRLMaker_Base(class QRLMaker* q) noexcept :
	qrlmaker(NULL)
{
	assert (NULL != q);

	qrlmaker = q;

	return;
}

QRLMaker_Base::QRLMaker_Base(void) noexcept :
	qrlmaker(NULL)
{
	assert (false);
	return;
}

QRLMaker_Base::~QRLMaker_Base(void) noexcept
{
	assert (NULL != qrlmaker);

	return;
}


QRLMaker_QRL::QRLMaker_QRL(void) noexcept(false) :
	qrllibrary(NULL)
{
	try {
		qrllibrary = new(std::nothrow) QRL::QRLLibrary(qrlmaker);
		assert (NULL != qrllibrary);
	}
	catch (QStringList msg) {
		QString allmsg = msg.join("\n");
		QMessageBox::critical(NULL, tr("QRL Maker"), allmsg);
		throw std::move(msg);
	}

	const bool ret = qrllibrary->readLibrary();
	if (!ret) {
		QString allmsg = errmsg.join("\n");
		allmsg.append("\n");
		allmsg.append(tr("ERROR: Failed to load required files."));
		QMessageBox::critical(NULL, tr("QRL Maker"), allmsg);
		throw errmsg;
	}

	return;
}

QRLMaker_QRL::~QRLMaker_QRL(void) noexcept
{
	assert (NULL != qrllibrary);

	delete qrllibrary;

	return;
}


QRLMaker_ClipBoard::QRLMaker_ClipBoard(void) noexcept :
	clipboard(NULL)
{
	clipboard = new(std::nothrow) MapEditor_ClipBoard(qrlmaker);
	assert (NULL != clipboard);

	return;
}

QRLMaker_ClipBoard::~QRLMaker_ClipBoard(void) noexcept
{
	assert (NULL != clipboard);

	if (NULL != clipboard) {
		delete clipboard;
	}
	return;
}


const QImage* QRLMaker_ImageBuilder::Cursor(enum Settings::OverlayImage_Info::CursorType t) const noexcept
{
	if ((Settings::OverlayImage_Info::CURSOR_TYPE_MIN <= t) && (t <= Settings::OverlayImage_Info::CURSOR_TYPE_MAX)) {
		return &(cursor[t]);
	}
	return NULL;
}

const QImage* QRLMaker_ImageBuilder::ArrowImage(bool mode, QRL::ChipArrow num) const noexcept
{
	if ((QRL::ChipArrow_MIN < num) && (num < QRL::ChipArrow_MAX)) {
		if (mode) {
			return &(arrowimage_true[num - QRL::ChipArrow_MIN - 1]);
		} else {
			return &(arrowimage_false[num - QRL::ChipArrow_MIN - 1]);
		}
	}
	return NULL;
}

QRLMaker_ImageBuilder::QRLMaker_ImageBuilder(void) noexcept(false) :
	arrowimage_true(NULL),
	arrowimage_false(NULL)
{
	QStringList errmsg;

	for (int i = Settings::OverlayImage_Info::CURSOR_TYPE_MIN; i <= Settings::OverlayImage_Info::CURSOR_TYPE_MAX; ++i) {
		const QString full_imagename = Settings::OverlayImage.File.ImageDir + Settings::OverlayImage.File.Cursor[i];

		const bool ret = this->cursor[i].load(full_imagename);
		if (!ret) {
			errmsg += tr("ERROR: Failed to load '%1'.").arg(full_imagename);
		} else {
		  #if defined(DEBUG)
			const int_fast16_t required_w = Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize + Settings::BuildImage.Chip.GapSize + 1;
			const int_fast16_t required_h = Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize + Settings::BuildImage.Chip.GapSize + 1;
			const int_fast16_t actual_w = this->cursor[i].width();
			const int_fast16_t actual_h = this->cursor[i].height();
			if ((required_w != actual_w) || (required_h != actual_h)) {
				fprintf(stderr, "Image '%s': Size mismatch. Required:(%3" PRId16 ",%3" PRId16 "), Actual:(%3" PRId16 ",%3" PRId16 ").\n", qPrintable(Settings::OverlayImage.File.Cursor[i]), required_w, required_h, actual_w, actual_h);
			}
		  #endif
		}
	}

	{
		arrowimage_true = new(std::nothrow) QImage[QRL::ChipArrow_MAX - QRL::ChipArrow_MIN + 1];
		assert (NULL != arrowimage_true);
		for (uint_fast8_t i = (QRL::ChipArrow_MIN + 1); i < QRL::ChipArrow_MAX; ++i) {
			const QString iconname = QString("%1%2.%3").arg(Settings::OverlayImage.File.ArrowImage_True).arg(i).arg(Settings::OverlayImage.File.ArrowImage_Ext);
			const QString full_iconname = QString("%1%2").arg(Settings::OverlayImage.File.ImageDir).arg(iconname);
			const bool ret = arrowimage_true[i - QRL::ChipArrow_MIN - 1].load(full_iconname);
			if (!ret) {
				errmsg += tr("ERROR: Failed to load '%1'.").arg(full_iconname);
			} else {
			  #if defined(DEBUG)
				 const int_fast16_t required_w = Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize + Settings::BuildImage.Chip.GapSize + (Settings::BuildImage.Chip.OverlappingSize * 2) + 1;
				 const int_fast16_t required_h = Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize + Settings::BuildImage.Chip.GapSize + (Settings::BuildImage.Chip.OverlappingSize * 2) + 1;
				 const int_fast16_t actual_w = arrowimage_true[i - QRL::ChipArrow_MIN - 1].width();
				 const int_fast16_t actual_h = arrowimage_true[i - QRL::ChipArrow_MIN - 1].height();
				 if ((required_w != actual_w) || (required_h != actual_h)) {
					 fprintf(stderr, "Image '%s': Size mismatch. Required:(%3" PRId16 ",%3" PRId16 "), Actual:(%3" PRId16 ",%3" PRId16 ").\n", qPrintable(iconname), required_w, required_h, actual_w, actual_h);
				 }
			  #endif
			}
		}

		arrowimage_false = new(std::nothrow) QImage[QRL::ChipArrow_MAX - QRL::ChipArrow_MIN + 1];
		assert (NULL != arrowimage_false);
		for (uint_fast8_t i = (QRL::ChipArrow_MIN + 1); i < QRL::ChipArrow_MAX; ++i) {
			const QString iconname = QString("%1%2.%3").arg(Settings::OverlayImage.File.ArrowImage_False).arg(i).arg(Settings::OverlayImage.File.ArrowImage_Ext);
			const QString full_iconname = QString("%1%2").arg(Settings::OverlayImage.File.ImageDir).arg(iconname);
			const bool ret = arrowimage_false[i - QRL::ChipArrow_MIN - 1].load(full_iconname);
			if (!ret) {
				errmsg += tr("ERROR: Failed to load '%1'.").arg(full_iconname);
			} else {
			  #if defined(DEBUG)
				const int_fast16_t required_w = Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize + Settings::BuildImage.Chip.GapSize + (Settings::BuildImage.Chip.OverlappingSize * 2) + 1;
				const int_fast16_t required_h = Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize + Settings::BuildImage.Chip.GapSize + (Settings::BuildImage.Chip.OverlappingSize * 2) + 1;
				const int_fast16_t actual_w = arrowimage_false[i - QRL::ChipArrow_MIN - 1].width();
				const int_fast16_t actual_h = arrowimage_false[i - QRL::ChipArrow_MIN - 1].height();
				if ((required_w != actual_w) || (required_h != actual_h)) {
					fprintf(stderr, "Image '%s': Size mismatch. Required:(%3" PRId16 ",%3" PRId16 "), Actual:(%3" PRId16 ",%3" PRId16 ").\n", qPrintable(iconname), required_w, required_h, actual_w, actual_h);
				}
			  #endif
			}
		}
	}

	if (!errmsg.isEmpty()) {
		QString allmsg = errmsg.join("\n");
		QMessageBox::critical(NULL, tr("QRL Maker"), allmsg);
		throw errmsg;
	}

	return;
}

QRLMaker_ImageBuilder::~QRLMaker_ImageBuilder(void) noexcept
{
	assert (NULL != arrowimage_true);
	assert (NULL != arrowimage_false);

	if (NULL != arrowimage_true) {
		delete[] arrowimage_true;
	}
	if (NULL != arrowimage_false) {
		delete[] arrowimage_false;
	}

	return;
}


bool QRLMaker_MainWindow::open_window(void) noexcept(false)
{
	try {
		w.emplace_back(new(std::nothrow) MainWindow(qrlmaker));
		assert (NULL != w.back());
	}
	catch (QStringList msg) {
		QString allmsg = msg.join("\n");
		QMessageBox::critical(NULL, tr("QRL Maker"), allmsg);
		throw std::move(msg);
	}

	w.back()->setAttribute(Qt::WA_DeleteOnClose);
	w.back()->show();

	return true;
}

bool QRLMaker_MainWindow::close_window(const class MainWindow* const target) noexcept
{
	bool ret = false;

	for (auto itr = w.begin(); itr != w.end(); ++itr) {
		if (target == (*itr)) {
			w.erase(itr);
			ret = true;
			break;
		}
	}

	return ret;
}

QRLMaker_MainWindow::QRLMaker_MainWindow(const QStringList* const filelist) noexcept(false)
{
	if (!filelist || (filelist->size() <= 0)) {
		open_window();

	} else {
		QStringList errmsg;
		errmsg.clear();

		for (int i = 0; i < filelist->size(); ++i) {
			QString strFName = filelist->at(i);

			if (!strFName.isEmpty() && QFile::exists(strFName)) {
				const bool ret_new = this->open_window();
				if (!ret_new) {
					errmsg += tr("ERROR: A file open in a new window was failed.");
					break;
				}

				MainWindow* const new_w = this->window_back_();
				if (NULL == new_w) {
					errmsg += tr("ERROR: A file open in a new window was failed.");
					break;
				}

				const bool ret = new_w->open(strFName);
				if (!ret) {
					errmsg += tr("ERROR: Failed to open a file '%1' in a new window.").arg(strFName);
				}
			} else {
				errmsg += tr("ERROR: Failed to open a file '%1'.").arg(strFName);
			}
		}

		if (!errmsg.isEmpty()) {
			QString allmsg = errmsg.join("\n");
			QMessageBox::critical(NULL, tr("QRL Maker"), allmsg);
			throw errmsg;
		}
	}

	return;
}

QRLMaker_MainWindow::~QRLMaker_MainWindow(void) noexcept
{
	while (!w.empty()) {
		const auto itr = w.begin();
		(*itr)->close();
		delete (*itr);
		// w.erase(); is called from close_window();
	}

	return;
}


QRLMaker::QRLMaker(const QStringList* const filelist) noexcept(false) :
	QRLMaker_Base(this),
	QRLMaker_MainWindow(filelist)
{
	return;
}

QRLMaker::~QRLMaker(void) noexcept
{
	return;
}
// [ End of File ]
