/**@file
 *
 *@brief	QRL Maker
 *@date		Thu,08 Oct,2015 - Mon,12 Oct,2015
 *@date		Tue,13 Oct,2015 - Wed,14 Oct,2015
 *@date		Sun,18 Oct,2015
 *@date		Thu,22 Oct,2015
 *@date		Wed,28 Oct,2015
 *@date		Sat,31 Oct,2015
 *@date		Sun,10 Jan,2016
 *@date		Thu,11 Feb,2016
 *@date		Thu,29 Nov,2018
 *@date		Fri,28 Dec,2018
 *@date		Fri,04 Jan,2019
 *@date		Sat,19 Sep,2020
 *@date		Wed,17 Feb,2021
 *@date		Fri,19 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Sat,29 May,2021
 *
 *@author	Copyright(C)2015-2021 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <assert.h>
#include "settings.h"
#include "qrlmap.h"
#include <QStringList>

bool QRL::QRLChipList::put(const QRL::QRLChip* const chip) noexcept
{
	assert (NULL != chip);

	QRL::QRLChip* ptr = NULL;
	bool done = false;
	for (QRL::QRLChipNumber i = 0; i < this->maxlist(); ++i) {
		ptr = this->ptr_(i);
		assert (NULL != ptr);
		if (QRLChipEmptyFlag != ptr->chip_number) {
			continue;
		}

		{ // Found
			(*ptr) = (*chip);
			done = true;
			break;
		}
	}
	if (!done) {
		return false;
	}
	return true;
}

QRL::QRLChipNumber QRL::QRLChipList::put(const ChipAddrXY& addr, const QRL::QRLChipLibrary* const chip, const QRL::QRLChip* const ptr) noexcept
{
	assert (NULL != chip);

	QRL::QRLChipNumber buf[this->maxlist() + 1];
	buf[0] = QRL::QRLChipOutOfMapFlag;
	for (QRL::QRLChipNumber i = 1; i <= this->maxlist(); ++i) {
		buf[i] = QRL::QRLChipEmptyFlag;
	}

	for (QRL::QRLChipNumber i = 0; i < this->maxlist(); ++i) {
		const QRL::QRLChip* const ptr_tmp = this->ptr(i);
		if (QRL::QRLChipEmptyFlag != ptr_tmp->chip_number) {
			buf[ptr_tmp->chip_number] = i;
		}
	}
	QRL::QRLChipNumber chip_number = QRLChipOutOfMapFlag;
	for (QRL::QRLChipNumber i = 0; i <= this->maxlist(); ++i) {
		if (QRL::QRLChipEmptyFlag == buf[i]) {
			chip_number = i;
			break;
		}
	}
	if (chip_number <= 0) {
		return QRLChipOutOfMapFlag;
	}

	bool done = false;
	for (QRL::QRLChipNumber i = 0; i < this->maxlist(); ++i) {
		QRL::QRLChip* const ptr_ = this->ptr_(i);
		assert (NULL != ptr_);
		if (QRLChipEmptyFlag != ptr_->chip_number) {
			continue;
		}

		{ // Found
			ptr_->chip_number = chip_number;
			ptr_->addr = addr;
			(ptr_->name) = *(chip->name);
			if (NULL != chip->next_address_condition_true) {
				if (NULL != ptr) {
					ptr_->next_address_condition_true = ptr->next_address_condition_true;
				} else {
					ptr_->next_address_condition_true = chip->next_address_condition_true->toLong();
				}
			} else {
				ptr_->next_address_condition_true = 0;
			}
			if (NULL != chip->next_address_condition_false) {
				if (NULL != ptr) {
					ptr_->next_address_condition_false = ptr->next_address_condition_false;
				} else {
					ptr_->next_address_condition_false = chip->next_address_condition_false->toLong();
				}
			} else {
				ptr_->next_address_condition_false = 0;
			}

			while (0 < ptr_->arg.size()) {
				ptr_->arg.removeFirst();
			}
			if (NULL != ptr) {
				for (uint_fast8_t i = 0; i < chip->argument_num; ++i) {
					ptr_->arg.append((ptr->arg)[i]);
				}
			} else {
				for (uint_fast8_t i = 0; i < chip->argument_num; ++i) {
					ptr_->arg.append(*(chip->argument[i].arg));
				}
			}
			done = true;
			break;
		}
	}
	if (!done) {
		return QRLChipOutOfMapFlag;
	}
	return chip_number;
}

bool QRL::QRLMap::put_to_map(const QRL::ChipAddrXY& addr, const QRL::QRLChipLibrary* const chip, const QRLChipNumber chip_number) noexcept
{
	assert (NULL != chip);

	const struct QRL::ChipAddrXY max = chip->size;
	QStringList::const_iterator it = chip->footprint->begin();
	for (QRL::ChipAddr y = 0; y < max.y; ++y, ++it) {
		for (QRL::ChipAddr x = 0; x < max.x; ++x) {
			if (QChar('#') == (*it)[static_cast<int>(x)]) {
				const QRL::ChipAddrXY ptr = {
					.x = static_cast<QRL::ChipAddr>(addr.x + x),
					.y = static_cast<QRL::ChipAddr>(addr.y + y),
				};
				set_map(ptr, chip_number);
			}
		}
	}
	return true;
}

bool QRL::QRLMap::put(const ChipAddrXY& addr, const QRL::QRLChipLibrary* const chip) noexcept
{
	assert (NULL != chip);

	const bool ret = is_put(addr, chip);
	if (!ret) {
		return false;
	}

	const QRLChipNumber chip_number = qrlchiplist->put(addr, chip, NULL);
	if (chip_number <= 0) {
		return false;
	}

	put_to_map(addr, chip, chip_number);

	return true;
}

bool QRL::QRLMap::put(const ChipAddrXY& addr, const QRL::QRLChip* const ptr, bool* const chip_number_changed) noexcept
{
	assert (NULL != ptr);

	const QRL::QRLChipLibrary* const chip = qrlmaker->QRLLibrary()->chipSearch(ptr->name);
	assert (NULL != chip);

	const bool ret = is_put(addr, chip);
	if (!ret) {
		return false;
	}

	const QRLChipNumber chip_number = qrlchiplist->put(addr, chip, ptr);
	if (chip_number <= 0) {
		return false;
	}
	if (ptr->chip_number != chip_number) {
		if (NULL != chip_number_changed) {
			*chip_number_changed = true;
		}
	}

	put_to_map(addr, chip, chip_number);

	return true;
}
// [ End of File ]
