// HLServer is the main part of the server. It listens for new connections,
// keeps a list of clients connected and private chats. It has functions for
// sending packets to multiple clients. It also has an upkeep thread that
// runs at an interval of SLEEP_TIME to check for dropped connections and idle
// clients. Inside HLServer.cpp are a few other classes for registering the
// server on trackers and listening for file transfer connections.

#ifndef HL_SERVER_H
#define HL_SERVER_H

#include "SocketSelector.h"
#include "AsyncTCPSocket.h"
#include "BMutex.h"
#include "HLPacket.h"
#include "HLDataBase.h"
#include "HLTransfer.h"
#include "ServerConf.h"
#include "InspectSOCKS.h"
#include <list>

#if defined(CONFIG_RENDEZVOUS)
	#include "DNSServiceRegistrar.h"
#endif

#if defined(CONFIG_LINKING)
	#include "ServerLink.h"
#endif

//2003/08/01 added by ortana.
#ifdef WIN32
#include "WinProject\\CBan.h"
#endif//WIN32

#define KICK_STRIKES				  3 // 3 strikes and you're out
#define SLEEP_TIME					 20 // 20 seconds
#define LOGIN_TIMEOUT				 30 // 30 seconds

class TCPSocket;
class HLFileServer;
class HLChat;
class HLClient;
class HLTrackerRegister;
#if defined(_HTTPD_)
class WebServer;
#endif

typedef list< HLClient* > ClientList;
typedef list< HLChat* > ChatList;

struct kick_entry
{
	time_t time;
	string nick;
	u_int32_t host;
	u_int32_t count;
};
typedef struct kick_entry kick_entry;
typedef list<kick_entry> KickList; // list of users recently kicked

typedef bool (*PacketFilterProc)(HLUser &inUser);

class HLServer : public AsyncTCPSocket, public InspectSOCKSCallback
#if defined(CONFIG_RENDEZVOUS)
    , public DNSServiceRegistrarCallback
#endif // CONFIG_RENDEZVOUS
{
public:
	HLServer(ServerConf &inConf);
	~HLServer();
	
	void Start();
	void Stop();
	bool IsRunning();

	//2003/08/01 added by ortana.
#ifdef WIN32
	NTF::CBan& getHLBan()
	{
		return this->hlBan;
	}
#endif//WIN32
	
	time_t Uptime(time_t inCurrentTime)
		{ return (inCurrentTime - mStartTime); }
	
	void SetSubject(const string &inSubject);
	void GetSubject(string &outSubject);
	
	static void FormatNewsPost(const string &inName, time_t inTime,
							const string &inPost, string &outFormattedPost);
	static void FormatChat(HLClient &inClient, unsigned int inStyle,
							string &ioChatString, unsigned int &outLineCount);
	
	void ReloadConf();
	
	ServerConf &Config()
		{ return mConf; }
	
	void AddClientToKickList(HLClient &inClient);

	//2003/07/31 added by ortana.
	ClientList& getClentList();
	
#if defined(CONFIG_LINKING)
	LinkManager &Link() { return mLinkManager; }
#endif
	
	HLUser serverUser; // this is basically a null user used for banning code

	//2003/07/27 added by ortana.
	HLDataBase * getDataBase();
	
	
protected:
	void FoundSOCKSHost(const string &inAddress, int inVersion);
	//void Run();
	void OnAccept();
	
	void SendToAll(HLPacket &inPacket, PacketFilterProc inFilter = NULL);
	void SendToOthers(HLPacket &inPacket, HLClient &inClient, PacketFilterProc inFilter = NULL);
	void SendTo(HLPacket &inPacket, u_int16_t inUserID);
	void KeepAliveConnections();
	void UpdateAccountForClients(const HLAccount &inAccount);
	void KickClientsWithAccount(const string &inAccount);

	static bool ReadNewsFilter(HLUser &inUser);
	static bool ReadChatFilter(HLUser &inUser);
	static bool CanKickFilter(HLUser &inUser);
	
	u_int16_t CreateNewUserID();
	HLClient *GetClientForID(u_int16_t inUserID);
#if defined(CONFIG_LINKING)
	void RemoveClientWithGlobalID(u_int32_t inGlobalID);
	HLClient *GetClientWithGlobalID(u_int32_t inGlobalID);
	void RemoveRemoteClients();
#endif
	bool IsValidChat(HLChat &inChat);
	bool IsClientInChat(HLClient &inClient, HLChat &inChat);
	u_int8_t CountChatsForClient(HLClient &inClient);
	u_int8_t TotalChats();
	
	u_int32_t mTotalConnections;
	time_t mStartTime;
	bool mRunning;
	SocketSelector mSelector;
	//TCPSocket *mSocket;
	HLFileServer *mFileServer;
	HLTransferManager *mTransAM;
	HLTrackerRegister *mTrackerReg;
#if defined(_HTTPD_)
	WebServer *mWebServer;
#endif

#if defined(CONFIG_LINKING)
	LinkManager mLinkManager;
#endif

	HLDataBase mDataBase;
	
	BMutex mLock;
	// mLock applies to:
	string mSubject;
	ClientList mClientList;
	ChatList mChatList;
	ServerConf mConf;
	u_int16_t mUserIDCounter;
	KickList mKickList;

#if defined(CONFIG_RENDEZVOUS)
	void RegisterReply(const string &inServiceName, int inErrorCode);
	DNSServiceRegistrar mDNSServiceRegistrar;
#endif
	
friend class HLClient;
friend class HLChat;
friend class HLFileServer;
friend class HLTransfer;
friend class HLTransferManager;
friend class HLTransferSocket;
friend class HLTrackerRegister;
friend class WebServer;
friend class LinkManager;

//2003/08/01 added by ortana.
#ifdef WIN32
NTF::CBan hlBan;
#endif//WIN32
};

extern HLServer *gServer;

#endif // HL_SERVER_H


