#include "HLSocket.h"
#include "HLProtocol.h"

HLSocket::HLSocket(int inDescriptor, struct sockaddr_in* inRemoteAddress)
    : BufferedSocket(inDescriptor, inRemoteAddress),
    mCallback(NULL), mPacketCounter(0), mGotMagic(false), mWaitingHeader(false)
{
    DEBUG_CALL(printf("connected!\n"); fflush(stdout));
}

HLSocket::~HLSocket()
{
    Shutdown();
    Close();
}

void HLSocket::DataAvailable()
{
    bool done = false;
    while (!done && mReadBuf.Size())
    {
        if (mGotMagic)
        {
			HL_PACKET_HEADER *hdr = (HL_PACKET_HEADER *)mReadBuf.Ptr();
            if (mWaitingHeader)
            {
				if (mReadBuf.Size() >= sizeof(HL_PACKET_HEADER))
                {
					mPacket.SetHeader(hdr); // this fixes the hdr byte ordering also
                    if (hdr->type != 0 && hdr->size1 == hdr->size2 &&
						hdr->size1 < 0x0000FFFF)
                    {
                        mWaitingHeader = false;
                    }
                    else
                    {
                        // close connection?
                        DEBUG_CALL(
							printf("bad packet header, terminating connection\n");
							printf("type: 0x%X id: 0x%X size1: %u size2: %u flags: 0x%X", hdr->type,
								hdr->id, hdr->size1, hdr->size2, hdr->flag);
							fflush(stdout));
                        done = true;
                        Close();
                    }
                }
                else
                    done = true;
            }
            else
            {
				if (mReadBuf.Size() >= (hdr->size1 + sizeof(HL_PACKET_HEADER)))
				{
					// got the rest of the packet
					char *packetData = mReadBuf.Ptr(sizeof(HL_PACKET_HEADER));
					mPacket.Instantiate(packetData, hdr->size1);
					// remove data from buffer
					mReadBuf.RemoveFrom(hdr->size1 + sizeof(HL_PACKET_HEADER));
					if (mCallback != NULL)
						mCallback->HandlePacket(mPacket);
					mPacket.Clear();
					mWaitingHeader = true;
				}
				else
					done = true;
			}
        }
        else if (mReadBuf.Size() >= HTLC_MAGIC_LEN)
        {
            if (memcmp(mReadBuf.Ptr(), HTLC_MAGIC, HTLC_MAGIC_LEN) == 0)
            {
                mGotMagic = true;
                mWaitingHeader = true;
                // remove magic data from buffer
                mReadBuf.RemoveFrom(HTLC_MAGIC_LEN);
				BufferedSend(HTLS_MAGIC, HTLS_MAGIC_LEN);
			}
            else
            {
                DEBUG_CALL(
                    printf("bad magic, terminating connection\n");
                
                    char hexBuf[513];
                    u_int32_t counter = 0;
                    while (counter < mReadBuf.Size() && counter < 256)
                    {
                        sprintf(&hexBuf[counter * 2], "%.2x", (u_int8_t)(mReadBuf.Ptr())[counter]);
                        counter++;
                    }
                    printf("data in hex(%u): %s\n", (unsigned int)mReadBuf.Size(), hexBuf);
                    fflush(stdout);
                );
                
                done = true;
                Close();
            }
        }
        else
            done = true;
    }
}

void HLSocket::OnClose()
{
    DEBUG_CALL(printf("HLSocket::OnClose\n"));
    BufferedSocket::OnClose();
    if (mCallback != NULL)
        mCallback->OnClose();
}

void HLSocket::SendPacket(HLPacket &inPacket)
{
	try
	{
		// first we must resize the BufferedSocket's write dynamic buffer
		// to hold the flattened packet data
		u_int32_t packetSize = inPacket.GetSize();
		size_t bufSize = mWriteBuf.Size();
		char *writeBuf = mWriteBuf.Resize(bufSize + packetSize);
		writeBuf += bufSize;
		
		// now we let HLPacket write the packet into the writeBuffer
		if (inPacket.GetType() != HTLS_HDR_TASK)
			inPacket.Flatten(writeBuf, packetSize, mPacketCounter++);
		else
			inPacket.Flatten(writeBuf, packetSize, 0);
		
		// if the socket is already waiting to send data, then do nothing
		// OnSend will get called and our data will get sent, otherwise we
		// do what OnSend would do, which is try to send the data in the buffer
		// and then remove the sent data from the buffer
		if (!mWaitWrite)
		{
			int sentSize = AsyncTCPSocket::Send(mWriteBuf.Ptr(), mWriteBuf.Size());
			if ((size_t)sentSize != mWriteBuf.Size())
			{
				mWriteBuf.RemoveFrom(sentSize);
			}
			else
			{
				mWriteBuf.Resize(0);
			}
		}
	}
	catch (socket_error &exp)
	{
		DEBUG_CALL(printf("exception in HLSocket::SendPacket %s\n", exp.what()));
	}
}

