﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Drawing;

namespace com.ast8.fw.forms
{
    /// <summary>
    /// WitchPaper専用キャレット
    /// </summary>
    /// <remarks>
    /// 本クラスではフォントサイズ修正に伴うキャレットサイズの修正や、
    /// カーソル位置の変更に伴うキャレット位置の修正まで自動で行う事はない。
    /// それぞれSizeやPositionプロパティを通して、
    /// 外部から別途指定する必要がある。
    /// なお、キャレットの点滅の際WM_PAINTメッセージが送出されないため、
    /// 本クラスを使用する場合は既存描画領域と重ならないよう十分注意する必要がある。
    /// </remarks>
    public class WitchCaret : IDisposable
    {
        #region API
        // ここの詳細はプラットフォームSDK等のドキュメントを参照のこと
        [DllImport("user32.dll")]
        public static extern bool SetCaretPos(int x, int y);
        [DllImport("user32.dll")]
        public static extern bool ShowCaret(IntPtr hWnd);
        [DllImport("user32.dll")]
        public static extern bool HideCaret(IntPtr hWnd);
        [DllImport("user32.dll")]
        public static extern int CreateCaret(IntPtr hwnd, IntPtr hbm, int cx, int cy);
        [DllImport("user32.dll")]
        public static extern int DestroyCaret();
        // 非サポート関数（通常使うべきではない）
        [DllImport("user32.dll")]
        public static extern bool SetCaretBlinkTime(uint uMSeconds);
        [DllImport("user32.dll")]
        public static extern uint GetCaretBlinkTime();
        #endregion

        #region private メンバ変数
        /// <summary>Controlプロパティの実体</summary>
        private Control m_control;
        /// <summary>Sizeプロパティの実体</summary>
        private Size m_size;
        /// <summary>Positionプロパティの実体</summary>
        private Point m_position;
        /// <summary>Visibleプロパティの実体</summary>
        private bool m_visible;
        #endregion

        #region public プロパティ
        /// <summary>
        /// キャレットを表示するコントロール
        /// </summary>
        public Control Control
        {
            get { return m_control; }
        }

        /// <summary>
        /// キャレットのサイズ
        /// </summary>
        public Size Size
        {
            get { return m_size; }
            set { m_size = value; }
        }

        /// <summary>
        /// キャレットの位置
        /// </summary>
        public Point Position
        {
            get { return m_position; }
            set { m_position = value; SetCaretPos(m_position.X, m_position.Y); }
        }

        /// <summary>
        /// キャレットが表示状態であればtrue, でなければfalse
        /// </summary>
        public bool Visible
        {
            get { return m_visible; }
            set
            {
                if (m_visible = value)
                    ShowCaret(Control.Handle);
                else
                    HideCaret(Control.Handle);
            }
        }
        #endregion

        #region コンストラクタ
        /// <summary>
        /// デフォルトコンストラクタ。本メソッドは呼ぶべきではない
        /// </summary>
        private WitchCaret()
        {
            // ほんとは呼んじゃ駄目なやつ
        }

        /// <summary>
        /// キャレットを表示するコントロールを指定してインスタンスを生成する
        /// </summary>
        /// <param name="ctrl">キャレットを表示するコントロール</param>
        /// <exception cref="System.ArgumentNullException">引数<c>ctrl</c>がnull</exception>
        public WitchCaret(Control ctrl)
        {
            if (ctrl == null)
                throw new System.NullReferenceException("ctrl is null");
            this.m_control = ctrl;
            this.m_position = Point.Empty;
            UpdateCaret();

            //Control.Enter += new EventHandler(ControlOnEnter);
            //Control.Leave += new EventHandler(ControlOnLeave);
            Control.GotFocus += new EventHandler(ControlOnGotFocus);
            Control.LostFocus += new EventHandler(ControlOnLostFocus);

            // コントロールにすでにフォーカスが当たっている場合の処理
            if (ctrl.Focused)
                ControlOnGotFocus(ctrl, new EventArgs());
        }
        #endregion

        #region public メソッド
        /// <summary>
        /// キャレットを更新する
        /// </summary>
        /// <remarks>
        /// ここで更新される情報はSizeプロパティであり、
        /// SizeプロパティはコントロールのFontプロパティを基に更新する
        /// </remarks>
        public void UpdateCaret()
        {
            Size = new Size(1, m_control.Font.Height);
            if (Visible)
            {
                ControlOnLostFocus(this, new EventArgs());
                ControlOnGotFocus(this, new EventArgs());
            }
        }

        /// <summary>
        /// キャレットを表示する
        /// </summary>
        public void Show()
        {
            Visible = true;
        }

        /// <summary>
        /// キャレットを非表示にする
        /// </summary>
        public void Hide()
        {
            Visible = false;
        }
        #endregion

        #region Dispose パターン実装
        // コンポーネントじゃないから別になんでもいいんだが、一応警告くらうので実装

        /// <summary>
        /// このオブジェクトの使用している全てのリソースを破棄する
        /// </summary>
        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// オブジェクトの占有メモリを適切に解放する
        /// </summary>
        /// <param name="disposing">true:マネージリソース解放 false:ネイティブリソースのみ解放</param>
        protected virtual void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (m_control.Focused)
                    ControlOnLostFocus(m_control, new EventArgs());

                Control.Enter -= new EventHandler(ControlOnGotFocus);
                Control.Leave -= new EventHandler(ControlOnLostFocus);
            }
            // 残ってる場合にNativeリソース解放
            if (Visible)
            {
                DestroyCaret();
            }
        }

        /// <summary>
        /// デストラクタ
        /// </summary>
        ~WitchCaret()
        {
            Dispose(false);
        }
        #endregion

        #region イベントハンドラ
        /// <summary>
        /// 対象コントロールがフォーカスを得た時に発生するイベントのイベントハンドラ
        /// </summary>
        /// <param name="sender">イベント送出オブジェクト</param>
        /// <param name="e">イベントパラメータ</param>
        public virtual void ControlOnGotFocus(object sender, EventArgs e)
        {
            CreateCaret(Control.Handle, IntPtr.Zero, Size.Width, Size.Height);
            SetCaretPos(Position.X, Position.Y);
            Show();
        }

        /// <summary>
        /// 対象コントロールがフォーカスを失った時に発生するイベントのイベントハンドラ
        /// </summary>
        /// <param name="sender">イベント送出オブジェクト</param>
        /// <param name="e">イベントパラメータ</param>
        public virtual void ControlOnLostFocus(object sender, EventArgs e)
        {
            Hide();
            DestroyCaret();
        }
        #endregion
    }
}
