﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Text;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;

namespace com.ast8.fw.forms
{
    /// <summary>
    /// 魔女の紙
    /// </summary>
    /// <remarks>
    /// 現行バージョンではまだGDI使用箇所がいくつか残っているが、
    /// 工程が切羽詰ってきたのでひとまず保留。
    /// 機能拡張重視である点お詫びしておきます。
    /// </remarks>
    public partial class WitchPaper : System.Windows.Forms.Control, TextOperationIF, StdTextBoxIF
    {
        /// <summary>
        /// 固定ピッチの英字フォントのマルチバイト文字が可変ピッチになってしまう問題に関するテスト用。
        /// </summary>
        private PitchAndFamily testpitch = PitchAndFamily.FixedPitch;

        #region WindowsAPI 関連

        #region 定数
        /// <summary>KeyPress</summary>
        private const int WM_CHAR = 0x0102;
        /// <summary>KeyDown</summary>
        private const int WM_KEYDOWN = 0x0100;
        #endregion

        #endregion

        #region メンバ定数
        /// <summary>一文字の長さを判定するために使用するワード</summary>
        private const string SINGLE_WORD = "a";
        /// <summary>二文字の長さを判定するために使用するワード</summary>
        private const string MULTI_WORD = "ａ";
        /// <summary>常に使用されるDrawStringFormat</summary>
        private const DrawStringFormat STRING_FORMAT = DrawStringFormat.NoClip | DrawStringFormat.SingleLine;
        #endregion

        #region private メンバ変数
        #region 領域系
        /// <summary>テキスト表示領域</summary>
        private System.Drawing.Rectangle m_textArea;
        /// <summary>行番号表示領域</summary>
        private System.Drawing.Rectangle m_numberArea;
        /// <summary>テキスト折り返し点</summary>
        private int m_textTurnPointX = 0;
        #endregion

        #region カーソル系
        /// <summary>カーソルの画面座標(現状m_caret.Positionと等価)</summary>
        private System.Drawing.Point m_cursorPoint;
        /// <summary>キャレット（入力カーソルね）</summary>
        private WitchCaret m_caret;
        /// <summary>
        /// ユーザーの意図する画面座標
        /// </summary>
        /// <remarks>
        /// 本変数はPoint型を使いまわしているため、
        /// Xは画面座標、Yは物理Yインデックスとなっている点に注意
        /// </remarks>
        private System.Drawing.Point m_programPoint = new System.Drawing.Point();
        /// <summary>縦スクロールバーの情報</summary>
        private ScrollMagician.SCROLLINFO m_vScrollInfo;
        /// <summary>横スクロールバーの情報</summary>
        private ScrollMagician.SCROLLINFO m_hScrollInfo;
        /// <summary>仮。ドラッグ中かを示すフラグ（不要かも）</summary>
        private bool m_dragMode;
        #endregion

        #region プロパティ実体
        #region StdTextBoxIF
        /// <summary>TextTurnPointプロパティの実体</summary>
        private ushort m_textTurnPoint;
        /// <summary>AutoTextTurnPointプロパティの実体</summary>
        private bool m_autoTextTurnPoint = true;
        /// <summary>ShowLinesプロパティの実体</summary>
        private ShowLines m_showLines;
        /// <summary>CustomCommandプロパティの実体</summary>
        private TextBoxCustomCommand m_customCommand;
        /// <summary>LineNumberModeプロパティの実体</summary>
        private LineNumberMode m_lineNumberMode = LineNumberMode.Visible;
        /// <summary>TabSpaceプロパティの実体</summary>
        private ushort m_tabSpace = 4;
        /// <summary>Clipboardプロパティの実体</summary>
        private CustomClipboardIF m_clipboard;
        /// <summary>CursorModeプロパティの実体</summary>
        private CursorMode m_cursorMode = CursorMode.Normal;
        /// <summary>SelectModeプロパティの実体</summary>
        private SelectMode m_selectMode = SelectMode.None;
        /// <summary>AcceptsTabプロパティの実体</summary>
        private bool m_acceptsTab = false;
        #endregion

        #region TextBoxIF
        /// <summary>InputModeプロパティの実体</summary>
        private TextInputMode m_inputMode;
        /// <summary>AutoInputModeプロパティの実体</summary>
        private bool m_autoInputMode;
        /// <summary>ReadOnlyプロパティの実体</summary>
        private bool m_readOnly;

        /// <summary>
        /// SelectionAreaプロパティの実体
        /// </summary>
        /// <remarks>
        /// この変数はソース中に大量参照が予想されるため、クラス内ではプロパティではなくフィールドを直接参照すること。
        /// </remarks>
        private TextSelectionArea m_selectionArea;
        #endregion

        #region TextOperateIF
        /// <summary>
        /// Linesプロパティの実体
        /// </summary>
        /// <remarks>
        /// 大量アクセスが予想されるため、本クラス内においては本フィールドの直接操作が望ましい
        /// </remarks>
        private System.Collections.Specialized.StringCollection m_buf
            = new System.Collections.Specialized.StringCollection();
        #endregion
        #endregion

        #region DirectX関連
        /// <summary>デバイス</summary>
        private Device m_device;
        /// <summary>本文描画に使うフォント</summary>
        private Font m_font;
        /// <summary>各種区切り線描画に使うライン</summary>
        private Line m_line;
        /// <summary>本文表示用スプライト</summary>
        private Sprite m_textSprite;
        /// <summary>選択領域表示用...やめよっかな</summary>
        private Line m_selectLine;
        /// <summary>選択領域用nullテクスチャ</summary>
        private Texture m_selectTexture;
        /// <summary>選択領域用スプライト</summary>
        private Sprite m_selectSprite;
        /// <summary>選択領域転送元領域</summary>
        private System.Drawing.Rectangle m_selectSourceRect = new System.Drawing.Rectangle(0, 0, 1, 1);
        /// <summary>選択領域色</summary>
        private int m_selectColor = System.Drawing.Color.FromArgb(64, 96, 128, 160).ToArgb();
        /// <summary>BeginSceneコマンド発行済みかのフラグ</summary>
        private bool m_isBeginScene;
        #endregion

        #region 未整理
        /// <summary>フィジカルアドレスリスト</summary>
        private List<System.Drawing.Point> m_lstPAL = new List<System.Drawing.Point>();
        /// <summary>文字がマルチバイトかどうかを判定するクラス</summary>
        private SingleCharMapper m_singleCharMapper = new SingleCharMapper();
        /// <summary>一行に表示出来る最大文字数</summary>
        private int m_lineCharCount;
        /// <summary>一文字の幅</summary>
        private int m_fontWidth;
        /// <summary>マルチワード一文字の幅</summary>
        private int m_wordFontWidth;
        /// <summary>テキスト表示領域Height（m_textArea.Height - m_lineHeight)</summary>
        private int m_maxHeight;
        /// <summary>一行の高さ</summary>
        private int m_lineHeight;
        #endregion

        #endregion

        #region コンストラクタ
        /// <summary>
        /// 唯一のコンストラクタ
        /// </summary>
        public WitchPaper()
        {
            // メンバ初期化
            m_lstPAL.Add(new System.Drawing.Point(0, 0));
            m_buf.Add("\n");
            m_showLines = new ShowLines();

            m_caret = new WitchCaret(this);
            m_customCommand = new TextBoxCustomCommand();

            // スクロールバー追加
            this.m_vScrollInfo = new ScrollMagician.SCROLLINFO();
            this.m_hScrollInfo = new ScrollMagician.SCROLLINFO();
            //this.m_vScrollInfo.fMask = ScrollMagician.SIF_RANGE | ScrollMagician.SIF_PAGE | ScrollMagician.SIF_POS | ScrollMagician.SIF_DISABLENOSCROLL;
            this.m_vScrollInfo.fMask = ScrollMagician.SIF_RANGE | ScrollMagician.SIF_PAGE | ScrollMagician.SIF_POS | ScrollMagician.SIF_DISABLENOSCROLL;
            this.m_vScrollInfo.nMin = 0;
            this.m_hScrollInfo.fMask = ScrollMagician.SIF_RANGE | ScrollMagician.SIF_PAGE | ScrollMagician.SIF_POS | ScrollMagician.SIF_DISABLENOSCROLL;
            this.m_hScrollInfo.nMin = 0;
            this.m_vScrollInfo.cbSize = (uint)System.Runtime.InteropServices.Marshal.SizeOf(this.m_vScrollInfo);
            this.m_hScrollInfo.cbSize = (uint)System.Runtime.InteropServices.Marshal.SizeOf(this.m_hScrollInfo);

            // 各種初期値設定
            InputMode = TextInputMode.Insert;
            SelectMode = SelectMode.Normal;
            CursorMode = CursorMode.Normal;

            // コントロール初期化
            InitializeComponent();

            // ダブルバッファの設定
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, false);
            this.SetStyle(ControlStyles.ResizeRedraw, false);
            this.SetStyle(ControlStyles.UserPaint, true);
            this.SetStyle(ControlStyles.AllPaintingInWmPaint, true);
            this.SetStyle(ControlStyles.Opaque, true);  // Vista的ではないが...
            this.SetStyle(ControlStyles.Selectable, true);  // マウスクリックではフォーカスをあててくれません。。

            // リソース情報の初期化
            updateAllResource();

            // 選択領域の初期化
            clearSelectArea();
        }
        #endregion

        #region protected メソッド

        #region normal
        /// <summary>
        /// 対象エリアの現在位置がオーバーフローしているかを検査する
        /// </summary>
        /// <param name="area">対象選択領域</param>
        /// <returns>true:オーバーフロー false:正常</returns>
        protected bool isOverflowCurrent(TextSelectionArea area)
        {
            if ((area.CurrentY < 0) || (area.CurrentY >= m_lstPAL.Count))
                return true;
            return false;
        }

        /// <summary>
        /// 対象選択領域がオーバーフローしているかを検査する
        /// </summary>
        /// <param name="area">対象選択領域</param>
        /// <returns>true:オーバーフロー false:正常</returns>
        protected bool isOverflowSelection(TextSelectionArea area)
        {
            if ((area.StartY < 0) || (area.StartY >= m_lstPAL.Count) || (area.EndY < 0) || (area.EndY >= m_lstPAL.Count))
                return true;
            return false;
        }

        /// <summary>
        /// 対象物理Yの次の物理行が異なる論理行かを取得する
        /// </summary>
        /// <param name="index">対象物理Y</param>
        /// <returns>true:異なる論理行、または文章の末尾 false:同一の論理行</returns>
        protected bool isNextLineFromNextPALLine(int index)
        {
            if (((index + 1) == m_lstPAL.Count) ||
                (m_lstPAL[index + 1].Y != m_lstPAL[index].Y))
                return true;
            return false;
        }

        /// <summary>
        /// 指定した物理Yの表示文字数を返す
        /// </summary>
        /// <param name="index">対象物理Y</param>
        /// <returns>表示文字数</returns>
        protected int getPALLineLength(int index)
        {
            if (isNextLineFromNextPALLine(index))
                return m_buf[m_lstPAL[index].Y].Length - m_lstPAL[index].X;
            return m_lstPAL[index + 1].X - m_lstPAL[index].X;
        }
        #endregion

        #region virtual
        /// <summary>
        /// 複数行一括入力に対応したInsertメソッドの実体
        /// </summary>
        /// <remarks>
        /// 本メソッドは通常Insertメソッドから引数valueにgetInputStringsをかけた結果を渡してコールされる。
        /// WitchPaperのデータ形態（末尾に\nが入れる必要があるなど）を変更したい場合、
        /// 将来的には本メソッドをオーバーライドすれば動作するようになるようにしたい。
        /// 現行はDeleteメソッドとの兼ね合いからまだ本メソッドをオーバーライドしても、
        /// 入力時のパフォーマンスを変えられる程度の威力しかない。
        /// </remarks>
        /// <param name="values">入力する文字配列</param>
        /// <param name="isOverwrite">上書き指定</param>
        /// <param name="startIndex">入力開始index</param>
        /// <param name="length">入力配列長</param>
        protected virtual void multiInsert(string[] values, bool isOverwrite, int startIndex, int length)
        {
            // MEMO:いらないんじゃないのかこれ
            if ((values == null) || (values.Length < (startIndex + length)))
                return;

            int ubound = length + startIndex - 1;
            int y = m_lstPAL[m_selectionArea.CurrentY].Y;
            int x = m_lstPAL[m_selectionArea.CurrentY].X + m_selectionArea.CurrentX;
            string rightString, leftString;
            int totalLength = 0;

            // 総文字数抽出
            for (int i = startIndex; i <= ubound; i++)
                totalLength += values[i].Length;
            totalLength += (values.Length - 1); // 改行文字数補正

            if (x < (m_buf[y].Length))
            {
                // 既存テキスト右の抽出
                rightString = m_buf[y].Substring(x);
            }
            else
            {
                // 既存テキスト右が存在しないので\nを付与
                rightString = "\n";  // MEMO:このケースは基本的に存在しないはずだが念のため
            }
            // テキスト左の抽出
            leftString = m_buf[y].Substring(0, x);

            for (int i = startIndex; i < ubound; i++, y++)
            {
                // 改行して追加
                m_buf.Insert(y, values[i] + "\n");
                x = 0;  // if (x != 0) のが早いのか？ x
            }

            if ((ubound - startIndex) > 0)
            {
                // 最後の一行と右側処理
                m_buf[y] = values[ubound] + rightString;
                // 最初の一行左側処理
                m_buf[m_lstPAL[m_selectionArea.CurrentY].Y] = leftString + m_buf[m_lstPAL[m_selectionArea.CurrentY].Y];
                // 改行があるため以後の行を全更新
                MustUpdatePhysicalAddressList(m_selectionArea.CurrentY);
            }
            else
            {
                // 右と左を連結させて最初の一行処理（一文字入力が多いためあえて別分け）
                m_buf[y] = leftString + values[ubound] + rightString;
                // 改行がないため最低限の更新
                UpdatePhysicalAddressList(m_selectionArea.CurrentY);
            }
            updateVScrollMax();
            cursorRight(totalLength);
        }

        /// <summary>
        /// 入力処理に使用する文字列配列を生成して返す
        /// </summary>
        /// <param name="value">元文字列</param>
        /// <returns>入力処理に使用する文字列配列</returns>
        protected virtual string[] getInputStrings(ref string value)
        {
            char[] targetChars = value.ToCharArray();
            int startIndex = 0;
            int j = targetChars.Length;
            int ubound = j - 1;
            int i = 0;
            System.Collections.Generic.List<string> m_buf = new List<string>(8);
            for (; i < j; i++)
            {
                if (targetChars[i] == '\r')
                {
                    if ((i < ubound) && (targetChars[i + 1] == '\n'))
                    {
                        // rは飛ばしてnつき補正
                        m_buf.Add(new string(targetChars, startIndex, (i - startIndex)));
                        i++;
                        startIndex = i + 1;
                    }
                    else
                    {
                        // CRのみの改行
                        m_buf.Add(new string(targetChars, startIndex, (i - startIndex)));
                        startIndex = i + 1;
                    }
                }
                else if (targetChars[i] == '\n')
                {
                    // 改行
                    m_buf.Add(new string(targetChars, startIndex, (i - startIndex)));
                    startIndex = i + 1;
                }
            }
            // 最後の一行
            if (startIndex == i)
            {
                m_buf.Add("");
            }
            else if (startIndex < i)
            {
                m_buf.Add(new string(targetChars, startIndex, i - startIndex));
            }
            return m_buf.ToArray();
        }

        /// <summary>
        /// <seealso cref="BeforeTextChanged"/>イベントを送出する
        /// </summary>
        /// <param name="e">イベントパラメータ</param>
        protected virtual void OnBeforeTextChanged(BeforeTextChangedEventArgs e)
        {
            if (BeforeTextChanged != null)
            {
                BeforeTextChanged(this, e);
            }
        }

        /// <summary>
        /// <seealso cref="SelectModeChanged"/>イベントを送出する
        /// </summary>
        /// <param name="e">イベントパラメータ</param>
        protected virtual void OnSelectModeChanged(OldValueCancelEventArgs<SelectMode> e)
        {
            if (SelectModeChanged != null)
            {
                SelectModeChanged(this, e);
            }
        }

        /// <summary>
        /// <seealso cref="CursorModeChanged"/>イベントを送出する
        /// </summary>
        /// <param name="e">イベントパラメータ</param>
        protected virtual void OnCursorModeChanged(OldValueCancelEventArgs<CursorMode> e)
        {
            if (CursorModeChanged != null)
            {
                CursorModeChanged(this, e);
            }
        }

        /// <summary>
        /// <seealso cref="InputModeChanged"/>イベントを送出する
        /// </summary>
        /// <param name="e">イベントパラメータ</param>
        protected virtual void OnInputModeChanged(OldValueCancelEventArgs<TextInputMode> e)
        {
            if (InputModeChanged != null)
            {
                InputModeChanged(this, e);
            }
        }

        /// <summary>
        /// 指定された範囲からフィジカルアドレスリストを強制的に更新します
        /// </summary>
        /// <remarks>
        /// 本メソッドは全更新において最適化された<seealso cref="UpdatePhysicalAddressList"/>(x)関数です。
        /// 100％全更新がかかる場合（ファイルを読み込んだ場合）等において効力を発揮することが期待されます。
        /// 通常本メソッドは自動的に呼び出されるため呼び出す必要はありません。
        /// </remarks>
        /// <param name="startY">更新を開始する物理Y座標</param>
        protected virtual void MustUpdatePhysicalAddressList(int startY)
        {
            if (m_lstPAL.Count == 0)
                m_lstPAL.Add(new System.Drawing.Point(0, 0));

            // 非表示対策
            if (getShowLength(SINGLE_WORD.ToCharArray(), 0) == 0)
                return;

            int currentY = startY;
            int prevIndex = m_lstPAL[startY].X;

            // 計算して描画!
            for (int i = m_lstPAL[startY].Y; i < this.m_buf.Count; i++)
            {
                char[] lineChars = m_buf[i].ToCharArray();
                int currentLineLength = m_buf[i].Length - 1;   // cut \n

                while (true)
                {
                    if (m_lstPAL.Count <= currentY)
                        m_lstPAL.Add(new System.Drawing.Point(prevIndex, i));
                    else
                        m_lstPAL[currentY] = new System.Drawing.Point(prevIndex, i);

                    currentY++;

                    // 表示可能文字数の算出(unicodeはめんどくさいなぁ)
                    prevIndex += getShowLength(lineChars, prevIndex);

                    // 表示完了判定
                    if (prevIndex >= currentLineLength)
                        break;
                }
                // 表示したX座標をクリア
                prevIndex = 0;
            }

            // 全てチェックを終えた場合に余計な領域を削除
            if (currentY < m_lstPAL.Count)
            {
                m_lstPAL.RemoveRange(currentY, m_lstPAL.Count - currentY);
            }
        }

        /// <summary>
        /// 指定された範囲からフィジカルアドレスリストを更新します。
        /// </summary>
        /// <remarks>
        /// 本メソッドは一行のみの更新に最適化されています。
        /// 複数行の更新を行う場合は<seealso cref="MustUpdatePhysicalAddressList"/>を呼び出す必要があります。
        /// 通常本メソッドは自動的に呼び出されるため呼び出す必要はありません。
        /// </remarks>
        /// <param name="startY">更新を開始する物理Y座標</param>
        protected virtual void UpdatePhysicalAddressList(int startY)
        {
            int currentY = startY;
            // if (startY >= m_lstPAL.Count) throw new NullReferenceException
            int prevIndex = m_lstPAL[startY].X;

            // 計算して描画!
            for (int i = m_lstPAL[startY].Y; i < this.m_buf.Count; i++)
            {
                char[] lineChars = m_buf[i].ToCharArray();
                int currentLineLength = m_buf[i].Length - 1;   // cut \n

                while (true)
                {
                    // 追加判定
                    if (m_lstPAL.Count <= currentY)
                        m_lstPAL.Add(new System.Drawing.Point(prevIndex, i));
                    else
                        // 更新終了判定（はじめの一行はもう少し効率的にすべきか？）
                        if ((m_lstPAL[startY].Y != i)
                            && (m_lstPAL[currentY].Y == i)
                            && (m_lstPAL[currentY].X == prevIndex))
                            return;
                        else
                            m_lstPAL[currentY] = new System.Drawing.Point(prevIndex, i);

                    currentY++;

                    // 表示可能文字数の算出(unicodeはめんどくさいなぁ)
                    prevIndex += getShowLength(lineChars, prevIndex);

                    // 表示完了判定
                    if (prevIndex >= currentLineLength)
                        break;
                }
                // 表示したX座標をクリア
                prevIndex = 0;
            }

            // 全てチェックを終えた場合に余計な領域を削除
            if (currentY < m_lstPAL.Count)
            {
                m_lstPAL.RemoveRange(currentY, m_lstPAL.Count - currentY);
            }
        }
        #endregion

        #endregion

        #region private メソッド
        #region 画面描画系
        /// <summary>
        /// DirectXに関連するリソースを初期化する
        /// </summary>
        /// <returns>true:成功 false:失敗</returns>
        public bool InitializeGraphics()
        {
            try
            {
                // 解放
                if (m_device != null)
                    m_device.Dispose();
                if (m_font != null)
                    m_font.Dispose();
                if (m_textSprite != null)
                    m_textSprite.Dispose();
                if (m_selectLine != null)
                    m_selectLine.Dispose();
                if (m_selectSprite != null)
                    m_selectSprite.Dispose();
                if (m_selectTexture != null)
                    m_selectTexture.Dispose();

                // デバイスパラメータの生成
                PresentParameters presentParams = new PresentParameters();
                presentParams.IsWindowed = true;
                presentParams.SwapEffect = SwapEffect.Discard;
                presentParams.BackBufferCount = 1;
                System.Drawing.Size targetSize = ClientSize;
                // バックバッファ指定
                if (targetSize.Width + targetSize.Height == 0)
                    targetSize = new System.Drawing.Size(1, 1);
                presentParams.BackBufferWidth = targetSize.Width;
                presentParams.BackBufferHeight = targetSize.Height;
                // デバイス生成
                m_device = new Device(
                    0,
                    DeviceType.Hardware,
                    this.Handle,
                    CreateFlags.SoftwareVertexProcessing,
                    presentParams
                );
                m_device.RenderState.Lighting = false;    // ライティング無効化
                // テキスト用スプライト生成
                m_textSprite = new Sprite(m_device);
                // 通常線生成
                m_line = new Line(m_device);
                m_line.Width = 1.0f;            // ラインの太さを指定
                m_line.Pattern = -1;            // 直線パターンを指定
                m_line.PatternScale = 1.0f;     // パターン間隔を指定
                // 選択領域生成
                m_selectLine = new Line(m_device);
                m_selectLine.Width = m_lineHeight;    // ラインの太さを指定
                m_selectLine.Pattern = -1;            // 直線パターンを指定
                m_selectLine.PatternScale = 1.0f;     // パターン間隔を指定

                // 選択領域用スプライト生成
                m_selectSprite = new Sprite(m_device);
                
                // こいつは遅いらしい
                m_selectTexture = Texture.FromBitmap(m_device,
                    WitchResource.NullImage,
                    Usage.None,
                    Pool.Managed
                );
                
                // テキスト描画用フォントの生成
                m_font = new Font(
                    m_device,
                    (int)(this.Font.GetHeight() + 0.5f),
                    0,
                    (this.Font.Bold) ? FontWeight.Bold : FontWeight.Regular,
                    1,
                    this.Font.Italic,
                    CharacterSet.Default,
                    Precision.Default,
                    FontQuality.Default,
                    testpitch,  // Unicode系はちっともFixedじゃないっす・・
                    this.Font.FontFamily.Name);
            }
            catch (DirectXException)
            {
                // DirectXExceptionが発生した場合は異常終了を返却
                return false;
            }

            // 正常終了を返却
            return true;
        }

        /// <summary>
        /// 全描画
        /// </summary>
        private void render()
        {
            if (m_device == null)
                return;

            int lineIndex = this.m_vScrollInfo.nPos;
            // MEMO:m_lstPAL[lindeIndex].Y-1なら必ず論理行番号が表示される
            int prevY = (lineIndex > 0) ? m_lstPAL[lineIndex-1].Y : -1;
            System.Drawing.Color foreColor = this.ForeColor;
            // 表示領域横幅取得

            try
            {
                m_device.BeginScene();
                m_isBeginScene = true;
                m_device.Clear(ClearFlags.Target, this.BackColor, 1.0f, 0);
                // text sprite Let's Go!!
                m_textSprite.Begin(SpriteFlags.AlphaBlend);
                System.Drawing.Rectangle rect = m_textArea;
                System.Drawing.Rectangle lineRect = m_numberArea;
                lineRect.Width -= 4;    // 余白
                // 文字列描画!
                for (int i = lineIndex; i < this.m_lstPAL.Count; i++)
                {
                    if (m_lineNumberMode == LineNumberMode.Visible)
                    {
                        // 論理行表示
                        if (prevY != m_lstPAL[i].Y)
                        {
                            prevY = m_lstPAL[i].Y;
                            lineRect.Y = rect.Y;
                            m_font.DrawString(m_textSprite,
                                (prevY + 1).ToString(),
                                lineRect,
                                STRING_FORMAT | DrawStringFormat.Right,
                                foreColor
                            );
                        }
                    }
                    else if (m_lineNumberMode == LineNumberMode.Physical)
                    {
                        // 物理行表示
                        lineRect.Y = rect.Y;
                        m_font.DrawString(m_textSprite,
                            (i + 1).ToString(),
                            lineRect,
                            STRING_FORMAT | DrawStringFormat.Right,
                            foreColor
                        );
                    }

                    //string showString = getShowString(i);
                    string[] tabString = getShowString(i).Split('\t');
                    int oldX = rect.X;
                    for (int j = 0; j < tabString.Length; j++)
                    {
                        m_font.DrawString(
                            m_textSprite,
                            tabString[j],
                            rect,
                            STRING_FORMAT,
                            foreColor
                        );
                        // ちと遅い。最適化するならgetShowStringとgetStringWidthはセットの方がいい
                        rect.X += getStringWidth(tabString[j].ToCharArray()) + m_tabSpace * m_fontWidth;
                    }
                    rect.X = oldX;
                    /*m_font.DrawString(
                        m_textSprite,
                        showString,
                        rect,
                        STRING_FORMAT,
                        foreColor
                    );
                    */
                    rect.Y += m_lineHeight;
                    if (rect.Y > m_maxHeight)
                    {
                        break;
                    }
                }
                // end of text sprite
                m_textSprite.End();

                // select area
                #region 選択領域についてのコメント
                // 以下の領域があり、y0:x0～y2:x5までが選択領域の場合
                // ========選択前=========
                // あいうえお012
                // かきくけこ345
                // さしすせそ678
                // ========通常時=========
                // ◆◆◆◆◆***
                // ◆◆◆◆◆***
                // ◆◆◆せそ678
                // ========矩形時=========
                // ◆◆◆◆◆012
                // ◆◆◆◆◆345
                // ◆◆◆◆◆678
                // =======================
                // ◆, * の領域に対して矩形をxor転送。
                // 現実的には一行単位で転送するのがいいと思うので、
                // いずれのケースでも3回のXor矩形転送処理としたいところ。
                // ※矩形時は一回、通常時は1回～2回でも済ませられるため、
                // ※最適化を行う場合は行単位ではなく矩形一つごとの処理とすること。
                #endregion

                renderSelectArea();

                renderLines();
            }
            finally
            {
                if (m_isBeginScene)
                {
                    //End the scene
                    m_isBeginScene = false;     // EndSceneでエラーが起きた場合に備えてここでfalseに。
                    m_device.EndScene();
                }
            }
            m_device.Present(); 
        }


        /// <summary>
        /// ラインを描画する
        /// </summary>
        private void renderLines()
        {
            // 線描画
            m_line.Begin();
            if (m_lineNumberMode != LineNumberMode.None)
            {
                // 行番号区切り線
                m_line.Draw(
                    new Vector2[] {
                    new Vector2(m_numberArea.Right, m_numberArea.Top),
                    new Vector2(m_numberArea.Right, m_numberArea.Bottom) 
                },
                    System.Drawing.Color.Gray.ToArgb()
                );
            }
            // 折り返し線描画
            if (m_showLines.TextTurnPoint)
            {
                m_line.Draw(
                    new Vector2[] {
                    new Vector2(m_textTurnPointX, m_textArea.Top),
                    new Vector2(m_textTurnPointX, m_textArea.Bottom) 
                },
                    System.Drawing.Color.Gray.ToArgb()
                );
            }
            // 現在行の強調表示
            if (m_showLines.CurrentLine)
            {
                int y = (m_selectionArea.CurrentY - m_vScrollInfo.nPos + 1) * m_lineHeight;
                m_line.Draw(
                    new Vector2[] {
                    new Vector2(m_textArea.Left,y),
                    new Vector2(m_textArea.Right, y) 
                },
                    System.Drawing.Color.Gray.ToArgb()
                );
            }
            m_line.End();
        }

        /// <summary>
        /// 選択領域を描画する
        /// </summary>
        /// <remarks>
        /// TODO:ほんとはこのメソッドは単体で呼べるようにし、
        /// カーソルを一つ動かすごとにm_textAreaスプライト領域の全更新がかからないようにしたいのだが。。
        /// TODO:矩形選択には未対応
        /// </remarks>
        private void renderSelectArea()
        {
            // 選択領域がない場合は抜ける
            if (m_selectionArea.StartX == -1)
                return;

            // MEMO:現在Spriteを使っているが、Mesh.Box辺りを使うといいのか？

            bool isPresent = false;
            int lineCount = this.m_maxHeight / m_lineHeight;
            // 単体で呼ばれた場合の対処
            if (m_isBeginScene == false)
            {
                isPresent = true;
                m_isBeginScene = true;
                m_device.BeginScene();
                m_device.Clear(ClearFlags.Target, this.BackColor, 1.0f, 0);
            }

            // start select area
            m_selectSprite.Begin(SpriteFlags.AlphaBlend);
            // 選択領域抽出
            System.Drawing.Rectangle[] selectRect = getSelectArea();
            for (int i = 0; i < selectRect.Length; i++)
            {
                // 画面座標への補正
                if (selectRect[i].Width == -1)
                {
                    selectRect[i].Width = m_textArea.Width;
                }
                else
                {
                    selectRect[i].Width = getStringWidth(
                            m_buf[m_lstPAL[selectRect[i].Y].Y].ToCharArray(),
                            selectRect[i].X + m_lstPAL[selectRect[i].Y].X,
                            selectRect[i].Width,
                            false
                    );
                }
                selectRect[i].X = m_textArea.X + getStringWidth(
                        m_buf[m_lstPAL[selectRect[i].Y].Y].ToCharArray(),
                        m_lstPAL[selectRect[i].Y].X,
                        selectRect[i].X,
                        false
                );
                selectRect[i].Y -= m_vScrollInfo.nPos;
                selectRect[i].Y = m_textArea.Y + selectRect[i].Y * m_lineHeight;
                selectRect[i].Height = m_textArea.Y + selectRect[i].Height * m_lineHeight;
            }

            for (int i = 0; i < selectRect.Length; i++)
            {
                // TODO:反転転送がわからん。とりあえずこれで。。
                m_selectSprite.Draw2D(
                    m_selectTexture,
                    m_selectSourceRect,
                    selectRect[i].Size,
                    selectRect[i].Location,
                    m_selectColor
                );
            }
            // end of select area
            m_selectSprite.End();

            // 単体で呼ばれた場合の対処
            if (isPresent)
            {
                renderLines();
                m_device.EndScene();
                m_isBeginScene = false;
                m_device.Present();
            }
        }
        #endregion

        #region 画面情報の取得 - 基本
        /// <summary>
        /// 現在設定されているフォントの半角時の幅を取得する
        /// </summary>
        /// <remarks>
        /// メンバ変数m_fontがnullの場合、本メソッドは常に1を返す
        /// </remarks>
        /// <param name="targetString">測定対象文字列。半角であれば'a'、全角であれば'ａ'を指定することが推奨される</param>
        /// <returns>フォントの幅（ピクセル）</returns>
        private int getFontWidth(string targetString)
        {
            if (m_font == null)
                return 1;
            return m_font.MeasureString(m_textSprite, targetString, STRING_FORMAT, 0).Width / targetString.Length;
        }

        /// <summary>
        /// 現在設定されているフォントの高さを取得する
        /// </summary>
        /// <returns>フォントの高さ（ピクセル）</returns>
        private int getLineHeight()
        {
            // とりあえず四捨五入処理付きで一行の高さを取得
            return (int)(this.Font.GetHeight() + 0.5f);
        }

        /// <summary>
        /// 一行の描画可能領域の幅を取得する
        /// </summary>
        /// <param name="g">計算対象のGraphicsオブジェクト</param>
        /// <returns>一行の描画可能領域の幅</returns>
        private int getLineWidth(int fontWidth)
        {
            int lineWidth = this.m_textArea.Width;
            // 文字数分の丸め処理(TODO:GDI用)
            return (lineWidth / fontWidth) * fontWidth;
        }
        #endregion

        #region 画面情報の取得
        /// <summary>
        /// 表示出来る最大文字数を取得する
        /// </summary>
        /// <param name="lineChars">表示対象行を表すキャラクタ配列</param>
        /// <param name="startIndex">表示開始論理インデックス</param>
        /// <returns>表示可能文字数</returns>
        private int getShowLength(char[] lineChars, int startIndex)
        {
            int showLength;
            int currentMaxIndex = lineChars.Length - startIndex;
            int x = 0;
            for (showLength = 0; showLength < currentMaxIndex; showLength++)
            {
                // 改行コードはlineCharsの最後一文字のみしか想定していない点に注意
                x += getCharCount(lineChars[startIndex + showLength]);
                if (x > m_textTurnPoint)
                    return showLength;
            }
            return showLength;
        }

        /// <summary>
        /// 対象キャラクタの画面横幅を取得する
        /// </summary>
        /// <param name="target">判定対象キャラクタ</param>
        /// <returns>取得した横幅（ピクセル）</returns>
        private int getCharWidth(char target)
        {
            // MEMO:一部1文字でも2文字でもないようなコードには未対応。
            // MEMO:これらに対応する場合は根本的に変更する必要アリ。
            // MEMO:大量に呼び出されるのでgetCharCountは呼び出さずにダイレクト計算
            return (target == '\t') ? (m_tabSpace * m_fontWidth) : (m_singleCharMapper.isSingleByte(target)) ? m_fontWidth : m_wordFontWidth;
        }

        /// <summary>
        /// 対象キャラクタの文字数を取得する
        /// </summary>
        /// <param name="target">判定対象キャラクタ</param>
        /// <returns>取得した文字数</returns>
        private int getCharCount(char target)
        {
            return (target == '\t') ? m_tabSpace : (m_singleCharMapper.isSingleByte(target)) ? 1 : 2;
        }


        /// <summary>
        /// 表示対象文字列を取得する
        /// </summary>
        /// <param name="i">対象の物理Y</param>
        /// <returns>表示対象文字列</returns>
        private string getShowString(int i)
        {
            return m_buf[m_lstPAL[i].Y].Substring(m_lstPAL[i].X, getPALLineLength(i));
        }

        /// <summary>
        /// 渡された文字列の画面に表示した場合のwidthを返す
        /// </summary>
        /// <param name="targetStr">判定対象文字列</param>
        /// <returns>文字列の画面サイズ</returns>
        /// <exception cref="System.NullReferenceException">引数がnull</exception>
        private int getStringWidth(string targetStr)
        {
            return getStringWidth(targetStr.ToCharArray());
        }

        /// <summary>
        /// 渡された文字列の画面に表示した場合のwidthを返す
        /// </summary>
        /// <param name="targetChars">判定対象文字列</param>
        /// <returns>文字列の画面サイズ</returns>
        /// <exception cref="System.NullReferenceException">引数がnull</exception>
        private int getStringWidth(char[] targetChars)
        {
            return getStringWidth(targetChars, 0, targetChars.Length, false);
        }

        /// <summary>
        /// 渡された文字列の指定された範囲の、画面に表示した場合のwidthをを返す
        /// </summary>
        /// <param name="targetChars">判定対象文字列</param>
        /// <param name="startIndex">判定開始index</param>
        /// <param name="length">判定文字列長</param>
        /// <param name="freeCursor">フリーカーソルモード。trueなら存在しない領域もの値も計算する</param>
        /// <returns></returns>
        private int getStringWidth(char[] targetChars, int startIndex, int length, bool freeCursor)
        {
            int j = startIndex + length;
            int width = 0;
            if (freeCursor)
            {
                // フリーカーソルモードならフリーカーソルの加算値をデフォルト値に設定
                if (j > targetChars.Length)
                {
                    width += (j - targetChars.Length) * m_fontWidth;
                }
            }

            // はみ出した分を丸める
            if (j > targetChars.Length)
                j = targetChars.Length;

            // 文字数分のX座標加算処理
            for (int i = startIndex; i < j; i++)
                width += getCharWidth(targetChars[i]);

            // 計算したwidthを返却
            return width;
        }

        /// <summary>
        /// 画面座標に該当する渡された文字列の文字indexを返す
        /// </summary>
        /// <param name="targetStr">判定対象文字列</param>
        /// <param name="targetX">判定対象画面X座標</param>
        /// <param name="freeCursor">trueにした場合は存在しない文字indexも計算する</param>
        /// <returns>画面座標に該当する文字index</returns>
        private int getStringPosition(string targetStr, int targetX, bool freeCursor)
        {
            return getStringPosition(targetStr.ToCharArray(), targetX, freeCursor);
        }

        /// <summary>
        /// 画面座標に該当する渡された文字列の文字indexを返す
        /// </summary>
        /// <param name="targetChars">判定対象文字列</param>
        /// <param name="targetX">判定対象画面X座標</param>
        /// <param name="freeCursor">trueにした場合は存在しない文字indexも計算する</param>
        /// <returns>画面座標に該当する文字index。<c>freeCursor</c>がfalseでターゲットが存在しない場合はtargetChars.Length-1が返る</returns>
        private int getStringPosition(char[] targetChars, int targetX, bool freeCursor)
        {
            int j = targetChars.Length;
            int width = m_textArea.X;
            int targetWidth;
            for (int i = 0; i < j; i++)
            {
                targetWidth = getCharWidth(targetChars[i]);
                if ((width + targetWidth / 2) >= targetX)
                {
                    return i;
                }
                width += targetWidth;
            }
            if (freeCursor)
            {
                return j + (targetX - width) / m_fontWidth;
            }
            return j;
        }

        /// <summary>
        /// 画面座標から物理Pointを取得する
        /// </summary>
        /// <param name="p">画面座標</param>
        /// <returns>物理ポイント</returns>
        public System.Drawing.Point getXYFromDisplayPoint(System.Drawing.Point p)
        {
            System.Drawing.Point ret = new System.Drawing.Point();
            // Y座標の特定
            ret.Y = m_vScrollInfo.nPos + p.Y / m_lineHeight;
            if (ret.Y >= m_lstPAL.Count)
                ret.Y = m_lstPAL.Count - 1;
            else if (ret.Y < 0)
                ret.Y = 0;
            // X座標の特定
            string targetString = getShowString(ret.Y);
            ret.X = getStringPosition(targetString.ToCharArray(), p.X, false);
            return ret;
        }
        #endregion

        #region リソース管理
        /// <summary>
        /// 全てのリソース情報を更新する
        /// </summary>
        private void updateAllResource()
        {
            // 計測用にfontだけ作り直し。ちょっと苦しいがやむなしか？
            bool errorFlag = false;
            try
            {
                if (this.m_font != null)
                {
                    m_font.Dispose();
                    m_font = new Font(
                        m_device,
                        (int)(this.Font.GetHeight() + 0.5f),
                        0,
                        (this.Font.Bold) ? FontWeight.Bold : FontWeight.Regular,
                        1,
                        this.Font.Italic,
                        CharacterSet.Default,
                        Precision.Default,
                        FontQuality.Default,
                        testpitch,
                        this.Font.FontFamily.Name);
                }
            }
            catch (DirectXException)
            {
                errorFlag = true;
            }
            RETRY_CREATE_RESOURCE:
            this.m_fontWidth = getFontWidth(SINGLE_WORD);                   // 1バイト文字の幅
            this.m_wordFontWidth = getFontWidth(MULTI_WORD);                // 2バイト文字の幅
            this.m_lineHeight = getLineHeight();                               // 一行の高さ（updateWindowResourceではない）
            updateWindowResource();
            m_caret.UpdateCaret();
            if (errorFlag)
            {
                errorFlag = false;
                goto RETRY_CREATE_RESOURCE;
            }
        }

        /// <summary>
        /// ウィンドウ情報に関するリソースを更新する
        /// </summary>
        private void updateWindowResource()
        {
            // create area
            // line and text
            if (m_lineNumberMode == LineNumberMode.None)
            {
                m_numberArea = new System.Drawing.Rectangle(0, 0, 0, 0);
            }
            else
            {
                int numberCount = m_lstPAL.Count.ToString().Length;
                m_numberArea = new System.Drawing.Rectangle(0, 0, m_fontWidth * ((numberCount < 5) ? 5 : numberCount + 1), ClientSize.Height);
            }
            m_textArea = new System.Drawing.Rectangle(
                m_numberArea.Width + 5,
                0,
                ClientSize.Width - (m_numberArea.Width + 5),
                ClientSize.Height
            );
            // auto update TextTurnPoint
            if (m_autoTextTurnPoint)
            {
                // オーバーフローは無視
                m_textTurnPoint = (ushort)(m_textArea.Width / m_fontWidth);
            }
            // update TextTurnPoint X
            m_textTurnPointX = m_textArea.Left + m_fontWidth * m_textTurnPoint + 2;

            // count value
            this.m_lineCharCount = getLineWidth(this.m_fontWidth);
            this.m_maxHeight = this.m_textArea.Height - this.m_lineHeight;

            // scroll
            this.m_vScrollInfo.nPage = (uint)(m_maxHeight / m_lineHeight);
            this.m_hScrollInfo.nPage = 0;//(uint)(m_textArea.Width / m_fontWidth);  // かな？

            // カーソル論理位置のバックアップ
            System.Drawing.Point prevPoint = new System.Drawing.Point(
                m_lstPAL[m_selectionArea.CurrentY].X + m_selectionArea.CurrentX,
                m_lstPAL[m_selectionArea.CurrentY].Y);
            char[] targetChars;
            if (m_buf[m_lstPAL[m_selectionArea.CurrentY].Y].Length <= 1)
            {
                targetChars = string.Empty.ToCharArray();
            }
            else
            {
                targetChars = m_buf[m_lstPAL[m_selectionArea.CurrentY].Y].Substring(
                    0, m_buf[m_lstPAL[m_selectionArea.CurrentY].Y].Length - 1).ToCharArray();
            }
            // フィジカルアドレスリスト全更新
            MustUpdatePhysicalAddressList(0);
            // 新しい物理位置に更新
            // TODO:面倒くさいので超遅コードです。機会があれば修正すること。
            for (int i = prevPoint.Y; i < m_lstPAL.Count; i++)  // 論理行の最小indexからというループ
            {
                if (prevPoint.Y == m_lstPAL[i].Y)
                {
                    // 発見
                    if ((m_lstPAL[i].X + getShowLength(targetChars, m_lstPAL[i].X))
                        >= prevPoint.X)
                    {
                        m_selectionArea.CurrentY = i;
                        m_selectionArea.CurrentX = prevPoint.X - m_lstPAL[i].X;
                        break;
                    }
                }
            }
            updateTextArea();

            InitializeGraphics();
        }

        /// <summary>
        /// テキスト表示領域に関する全ての情報を更新する
        /// </summary>
        /// <remarks>
        /// ここで更新される情報はスクロールバーの情報、
        /// カーソルの画面座標についての情報、
        /// ならびに選択領域の解除である。（選択領域は物理XYでやっているので修復作業は結構めんどい）
        /// 内部的にカーソル位置へのスクロールも行っている点に注意。
        /// </remarks>
        private void updateTextArea()
        {
            // スクロールバー最大値更新
            updateVScrollMax();
            updateHScrollMax();
            // カーソル位置更新
            updateProgramPoint(m_selectionArea.CurrentX, m_selectionArea.CurrentY);
            updateCursorPoint();
            // 選択領域初期化
            clearSelectArea();
        }
        #endregion

        #region 選択領域系
        /// <summary>
        /// 選択領域を更新する
        /// </summary>
        /// <remarks>
        /// CurrentX, CurrentYに応じて選択領域の値を更新。
        /// また、SelectModeの値に応じてclearSelectAreaを呼ぶ
        /// </remarks>
        private void updateSelectArea()
        {
            updateSelectArea(ref m_selectionArea);
        }

        /// <summary>
        /// 選択領域を更新する
        /// </summary>
        /// <remarks>
        /// CurrentX, CurrentYに応じて選択領域の値を更新。
        /// また、SelectModeの値に応じてclearSelectAreaを呼ぶ
        /// </remarks>
        /// <param name="targetArea">更新対象の選択領域</param>
        private void updateSelectArea(ref TextSelectionArea targetArea)
        {
            // m_selectModeは指定出来なくていいか
            if (m_selectMode != SelectMode.None)
            {
                if (targetArea.StartX == -1)
                {
                    // まだ選択状態になっていないため選択開始
                    targetArea.StartX = targetArea.CurrentX;
                    targetArea.StartY = targetArea.CurrentY;
                }
                targetArea.EndX = targetArea.CurrentX;
                targetArea.EndY = targetArea.CurrentY;
            }
            else
            {
                clearSelectArea(ref targetArea);
            }
        }

        /// <summary>
        /// 選択領域を未選択の状態にする
        /// </summary>
        private void clearSelectArea()
        {
            clearSelectArea(ref m_selectionArea);
        }

        /// <summary>
        /// 選択領域を未選択の状態にする
        /// </summary>
        /// <param name="targetArea">対象選択領域</param>
        private void clearSelectArea(ref TextSelectionArea targetArea)
        {
            targetArea.StartX = -1;
            targetArea.StartY = 0;
            targetArea.EndX = -1;
            targetArea.EndY = 0;
        }

        /// <summary>
        /// 現在の選択領域の物理座標リストを返す
        /// </summary>
        /// <remarks>
        /// 本メソッドで取得出来る矩形領域の中でWidthが-1になっているものは行全体を現す点に注意。
        /// </remarks>
        /// <returns>取得した物理座標リスト</returns>
        private System.Drawing.Rectangle[] getSelectArea()
        {
            return getSelectArea(m_selectionArea);
        }

        /// <summary>
        /// 指定した選択領域の物理座標リストを返す
        /// </summary>
        /// <remarks>
        /// 本メソッドで取得出来る矩形領域の中でWidthが-1になっているものは行全体を現す点に注意。
        /// </remarks>
        /// <param name="area">対象選択領域</param>
        /// <returns>取得した物理座標リスト</returns>
        private System.Drawing.Rectangle[] getSelectArea(TextSelectionArea area)
        {
            System.Drawing.Rectangle[] selectRect = new System.Drawing.Rectangle[3];
            // 描画矩形計算
            int minX, maxX, minY, maxY;
            int minusX, plusX;
            if (area.StartY > area.EndY)
            {
                minY = area.EndY;
                maxY = area.StartY;
                minX = area.EndX;
                maxX = area.StartX;
            }
            else
            {
                minY = area.StartY;
                maxY = area.EndY;
                minX = area.StartX;
                maxX = area.EndX;
            }
            if (area.StartX > area.EndX)
            {
                minusX = area.EndX;
                plusX = area.StartX;
            }
            else
            {
                minusX = area.StartX;
                plusX = area.EndX;
            }
            if (m_selectMode == SelectMode.Rectangle)
            {
                // 矩形描画!
                // TODO:しばし未対応
            }
            else
            {
                // 通常描画! めんどくさいなオイ！
                // 中間領域計算
                int heightPoint = maxY - minY - 1;

                if (heightPoint == -1)
                {
                    if (minusX != plusX)
                    {
                        // 一行のみの処理
                        selectRect[0].X = minusX;
                        selectRect[0].Width = plusX - minusX;
                    }
                }
                else
                {
                    // selectRect[0]を生成
                    selectRect[0].X = minX;
                    // 二行以上ある場合の処理
                    selectRect[0].Width = -1;//m_textArea.Width / m_fontWidth + 1;
                }
                selectRect[0].Y = minY;
                selectRect[0].Height = 1;

                if ((minY != maxY))
                {
                    // selectRect[2]を生成
                    selectRect[2].X = 0;
                    // 二行以上ある場合の処理
                    selectRect[2].Width = maxX;
                    selectRect[2].Y = maxY;
                    selectRect[2].Height = 1;
                }
                if (heightPoint > 0)
                {
                    // selectRect[1]を生成
                    selectRect[1].X = 0;
                    selectRect[1].Y = selectRect[0].Bottom;
                    selectRect[1].Width = -1; //m_textArea.Width / m_fontWidth + 1;
                    selectRect[1].Height = selectRect[2].Y - selectRect[1].Y;
                }
            }
            // 選択領域正規化(ary.Count = 0..3)
            System.Collections.Generic.List<System.Drawing.Rectangle> ary
                = new System.Collections.Generic.List<System.Drawing.Rectangle>(selectRect.Length);
            for (int i = 0; i < selectRect.Length; i++)
            {
                if (selectRect[i].X != selectRect[i].Right)
                    ary.Add(selectRect[i]);
            }
            return ary.ToArray();
        }
        #endregion

        #region スクロールバー関連
        /// <summary>
        /// 縦スクロールバーの最大値を更新する
        /// </summary>
        /// <remarks>
        /// 本メソッドは併せて<seealso cref="ScrollToCaret"/>関数を呼び出しスクロール処理も行っている
        /// </remarks>
        private void updateVScrollMax()
        {
            // MEMO:???なぜか-2でちょうどよくなる。以前はこんなことなかった気がするが..
            if (this.m_lstPAL.Count < 2)
            {
                this.m_vScrollInfo.nMax = 0;
            }
            else
            {
                this.m_vScrollInfo.nMax = this.m_lstPAL.Count - 2 + (int)this.m_vScrollInfo.nPage;
            }
            if (this.m_vScrollInfo.nPos > this.m_vScrollInfo.nMax)
                this.m_vScrollInfo.nPos = this.m_vScrollInfo.nMax;
            ScrollToCaret();
        }

        /// <summary>
        /// 横スクロールバーの最大値を更新する
        /// </summary>
        /// <remarks>
        /// 本メソッドは併せて<seealso cref="ScrollToCaret"/>関数を呼び出しスクロール処理も行っている
        /// </remarks>
        private void updateHScrollMax()
        {
            this.m_hScrollInfo.nMax = 0 + (int)this.m_hScrollInfo.nPage - 1;
            if (this.m_hScrollInfo.nPos > this.m_hScrollInfo.nMax)
                this.m_hScrollInfo.nPos = this.m_hScrollInfo.nMax;
            ScrollToCaret();
        }

        /// <summary>
        /// 垂直スクロールバーの情報を更新する
        /// </summary>
        /// <param name="vPoint">垂直スクロールバーの移動量</param>
        private void moveVScroll(int vPoint)
        {
            if ((vPoint != 0) && (m_vScrollInfo.nMax >= m_vScrollInfo.nPage))
            {
                m_vScrollInfo.nPos += vPoint;
                if (vPoint > 0)
                {
                    if (m_vScrollInfo.nPos > (m_vScrollInfo.nMax - (int)m_vScrollInfo.nPage + 1))
                        m_vScrollInfo.nPos = m_vScrollInfo.nMax - (int)m_vScrollInfo.nPage + 1;
                }
                else if (vPoint < 0)
                {
                    if (m_vScrollInfo.nPos < m_vScrollInfo.nMin)
                        m_vScrollInfo.nPos = m_vScrollInfo.nMin;
                }
                // MEMO:ScrollToCaretじゃないのはカーソル位置はそのままでスクロールすることがあるため
                ScrollMagician.SetScrollInfo(this.Handle, ScrollMagician.SB_VERT, ref this.m_vScrollInfo, true);
                updateCursorPoint();
                this.Refresh();
            }
        }

        /// <summary>
        /// 水平スクロールバーの情報を更新する
        /// </summary>
        /// <param name="hPoint">水平スクロールバーの移動量</param>
        private void moveHScroll(int hPoint)
        {
            if (hPoint != 0)
            {
                m_hScrollInfo.nPos += hPoint;
                if (m_hScrollInfo.nPos > m_hScrollInfo.nMax - m_hScrollInfo.nPage + 1)
                    m_hScrollInfo.nPos = m_hScrollInfo.nMax;
                if (m_hScrollInfo.nPos < m_hScrollInfo.nMin)
                    m_hScrollInfo.nPos = m_hScrollInfo.nMin;
                // MEMO:ScrollToCaretじゃないのはカーソル位置はそのままでスクロールすることがあるため
                ScrollMagician.SetScrollInfo(this.Handle, ScrollMagician.SB_HORZ, ref this.m_hScrollInfo, true);
                this.Refresh();
            }
        }

        // ScrollToCaretはpublicの方です
        #endregion

        #region カーソル系
        /// <summary>
        /// 現在カーソル位置を更新する
        /// </summary>
        private void updateCursorPoint()
        {
            int currentY = this.m_textArea.Y;
            for (int i = this.m_vScrollInfo.nPos; i < this.m_lstPAL.Count; i++)
            {
                // カーソル表示対象行かの判定
                if (this.m_selectionArea.CurrentY == i)
                {
                    // X位置抽出
                    char[] showChars = getShowString(i).ToCharArray();
                    int cursorX = 0;
                    for (int j = 1; j <= this.m_selectionArea.CurrentX; j++)
                        cursorX += getCharWidth(showChars[j - 1]);
                    m_cursorPoint.X = m_textArea.X + cursorX;
                    m_cursorPoint.Y
                        = (this.m_selectionArea.CurrentY - this.m_vScrollInfo.nPos) * m_lineHeight;
                    m_caret.Position = new System.Drawing.Point(m_cursorPoint.X - 1, m_cursorPoint.Y);
                    return;
                }

                currentY += m_lineHeight;
                if (currentY > m_maxHeight)
                {
                    // カーソルが画面の外なら画面外に飛ばし
                    m_cursorPoint.X = -64;
                    m_caret.Position = new System.Drawing.Point(m_cursorPoint.X - 1, m_cursorPoint.Y);
                    return;
                }
            }
            // 表示領域未作成と判断してカーソルを画面外に飛ばし（別になんでもいい）
            m_cursorPoint.X = -64;
            m_caret.Position = new System.Drawing.Point(m_cursorPoint.X - 1, m_cursorPoint.Y);
        }

        /// <summary>
        /// 現在カーソルの位置に応じた更新処理を行う
        /// </summary>
        /// <param name="clearSelect">trueの場合選択領域をクリアし、falseの場合選択領域を更新する</param>
        /// <param name="refreshProgramPoint">ユーザー指定位置の更新を行うかどうか</param>
        private void setCursor(bool clearSelect, bool refreshProgramPoint)
        {
            ScrollToCaret();
            if (clearSelect)
                clearSelectArea();
            else
                updateSelectArea();
            if (refreshProgramPoint)
                updateProgramPoint(m_selectionArea.CurrentX, m_selectionArea.CurrentY);
        }

        /// <summary>
        /// カーソル位置を移動させ、カーソルに関する更新処理を行う
        /// </summary>
        /// <param name="x">新しいカーソルの物理X座標</param>
        /// <param name="y">新しいカーソルの物理Y座標</param>
        /// <param name="clearSelect">trueの場合選択領域をクリアし、falseの場合選択領域を更新する</param>
        /// <param name="refreshProgramPoint">ユーザー指定位置の更新を行うかどうか</param>
        private void setCursor(int x, int y, bool clearSelect, bool refreshProgramPoint)
        {
            m_selectionArea.CurrentX = x;
            m_selectionArea.CurrentY = y;
            // Y位置オーバーフロー対策
            if (m_selectionArea.CurrentY >= (m_lstPAL.Count))
                m_selectionArea.CurrentY = m_lstPAL.Count - 1;

            // フリーカーソルじゃない場合のX位置オーバーフロー処理
            if (m_cursorMode == CursorMode.Normal)
            {
                // 改行位置までいくので注意!
                if ((m_selectionArea.CurrentX + m_lstPAL[m_selectionArea.CurrentY].X)
                    >= m_buf[m_lstPAL[m_selectionArea.CurrentY].Y].Length)
                    m_selectionArea.CurrentX =
                        m_buf[m_lstPAL[m_selectionArea.CurrentY].Y].Length - m_lstPAL[m_selectionArea.CurrentY].X - 1;
            }
            setCursor(clearSelect, refreshProgramPoint);
        }

        /// <summary>
        /// m_programPointの値を更新する
        /// </summary>
        private void updateProgramPoint(int xIndex, int yIndex)
        {
            m_programPoint.Y = yIndex;
            m_programPoint.X = m_textArea.Left + getStringWidth(m_buf[m_lstPAL[yIndex].Y].ToCharArray(), m_lstPAL[yIndex].X, xIndex, false);
        }

        /// <summary>
        /// ユーザーの意思を汲んだ物理座標を返す
        /// </summary>
        /// <param name="y">現在の物理Y座標</param>
        /// <returns>ユーザーの意志を汲んだ物理X座標</returns>
        private int getProgramX(int y)
        {
            char[] targetChars = getShowString(y).ToCharArray();
            return getStringPosition(targetChars, m_programPoint.X - 1, false); // -1 は + m_fontWidth / 2対策
        }

        /// <summary>
        /// 下移動
        /// </summary>
        /// <param name="movePoint">移動量</param>
        private void cursorDown(int movePoint)
        {
            int y = m_selectionArea.CurrentY + movePoint;
            if (y >= m_lstPAL.Count)
            {
                // 下端対策
                setCursor(getPALLineLength(m_lstPAL.Count - 1), m_lstPAL.Count - 1, false, true);
            }
            else
            {
                if (m_cursorMode == CursorMode.Normal)
                {
                    setCursor(getProgramX(y), y, false, false);
                }
            }
        }

        /// <summary>
        /// 上移動
        /// </summary>
        /// <param name="movePoint">移動量</param>
        private void cursorUp(int movePoint)
        {
            int y = m_selectionArea.CurrentY - movePoint;
            if (y < 0)
            {
                // 上端対策
                setCursor(0, 0, false, true);
            }
            else
            {
                if (m_cursorMode == CursorMode.Normal)
                {
                    setCursor(getProgramX(y), y, false, false);
                }
            }
        }

        /// <summary>
        /// 左移動
        /// </summary>
        /// <param name="movePoint">移動量</param>
        private void cursorLeft(int movePoint)
        {
            int x = m_selectionArea.CurrentX;
            int y = m_selectionArea.CurrentY;
            for (int i = 0; i < movePoint; i++)
            {
                x--;
                if (x < 0)
                {
                    // 画面左上端に対する処理
                    if (y == 0)
                    {
                        x = 0;
                        break;
                    }
                    // 上スクロール
                    y--;
                    x = getPALLineLength(y);
                    // 論理Yが変わらない場合の改行ポジション飛ばし処理
                    if (isNextLineFromNextPALLine(y) == false)
                        x--;
                }
            }
            setCursor(x, y, false, true);
        }

        /// <summary>
        /// 右移動
        /// </summary>
        /// <param name="movePoint">移動量</param>
        private void cursorRight(int movePoint)
        {
            int x = m_selectionArea.CurrentX;
            int y = m_selectionArea.CurrentY;
            int palLineLength = getPALLineLength(y);
            for (int i = 0; i < movePoint; i++)
            {
                x++;
                if (x >= palLineLength)
                {
                    if ((y + 1) >= m_lstPAL.Count)
                    {
                        // ドキュメント末尾に達している場合はオーバーフロー対策をやってbreak
                        x = palLineLength;
                        break;
                    }
                    // カーソルを右端まで行かせる処理
                    // 現行の一般的なテキストボックスでは右端まで行かないのが多いが、
                    // 左端で入力した前行の右端に文字が挿入されるなど不自然さが目につくので
                    // これがいいのではなかろうか。
                    bool noWrap = false;
                    if ((isNextLineFromNextPALLine(y) == false))
                    {
                        if (x == palLineLength)
                            noWrap = true;
                        else
                            x = 1;
                    }
                    else
                    {
                        x = 0;
                    }
                    if (noWrap == false)
                    {
                        y++;
                        palLineLength = getPALLineLength(y);
                    }
                }
            }
            setCursor(x, y, false, true);
        }
        #endregion

        #region IME
        /// <summary>
        /// IMEの入力コンテキストを設定する
        /// </summary>
        private void setImeContext()
        {
            IntPtr hIMC = IMEMagician.ImmGetContext(this.Handle);
            if (hIMC == IntPtr.Zero)
                return;
            try
            {
                // ウィンドウセット
                IMEMagician.COMPOSITIONFORM form = new IMEMagician.COMPOSITIONFORM();
                form.dwStyle = IMEMagician.CFS_POINT;
                form.ptCurrentPos.x = m_cursorPoint.X;
                form.ptCurrentPos.y = m_cursorPoint.Y;
                IMEMagician.ImmSetCompositionWindow(hIMC, ref form);

                /* パターンＦ（いい加減にしろMicrosoft Ver。一応動く）
                 * これもLOGFONT structでないと動かない。また、ToLogFontではフォント名のコピーがバグる
                 * lfFaceNameの下記宣言の事実上動作していない。
                 * [MarshalAs(UnmanagedType.ByValTStr, SizeConst = LF_FACESIZE)]
                 * public string lfFaceName;
                 * 結局下記コードしかないというのが結論
                 */
                object logFontWrap = new GDIMagician.LOGFONT();
                this.Font.ToLogFont(logFontWrap);   // 使えねぇ・・。
                GDIMagician.LOGFONT logFont = (GDIMagician.LOGFONT)logFontWrap;
                byte[] by = Encoding.Default.GetBytes(this.Font.Name);  // 英語環境とかの挙動怪しいかも
                for (int i = 0; i < logFont.lfFaceName.Length; i++)
                {
                    logFont.lfFaceName[i] = (i >= by.Length) ? (byte)0 : by[i];
                }
                IMEMagician.ImmSetCompositionFont(hIMC, ref logFont);

            }
            finally
            {
                IMEMagician.ImmReleaseContext(this.Handle, hIMC);
            }
        }
        #endregion

        #region 文字列操作系
        /// <summary>
        /// 現在の状態に応じた文字列入力処理を行う
        /// </summary>
        /// <param name="str">入力する文字列</param>
        private void inputString(string str)
        {
            switch (InputMode)
            {
                case TextInputMode.Append:
                    Append(str);
                    break;
                case TextInputMode.Unknown:
                    // TODO:例外でも出した方がいいか？
                    break;
                default:
                    Insert(str, InputMode == TextInputMode.Overwrite);
                    break;
            }
        }

        /// <summary>
        /// 文字列の削除と連結を行う
        /// </summary>
        /// <param name="startPX">削除を行う物理X</param>
        /// <param name="targetY">削除を行う物理Y</param>
        private void deleteJoin(int startPX, int targetY)
        {
            if ((targetY + 1) == m_lstPAL.Count)
            {
                // 連結出来ないから削除のみ
                m_buf[m_lstPAL[targetY].Y] = m_buf[m_lstPAL[targetY].Y].Remove(startPX);
            }
            else if (m_lstPAL[targetY + 1].Y != m_lstPAL[targetY].Y)
            {
                // 別行と連結
                m_buf[m_lstPAL[targetY].Y] = m_buf[m_lstPAL[targetY].Y].Remove(startPX)
                    + m_buf[m_lstPAL[targetY].Y + 1];
                // 連結した行を削除
                m_buf.RemoveAt(m_lstPAL[targetY].Y + 1);
            }
            else
            {
                // 自行と連結
                m_buf[m_lstPAL[targetY].Y] = m_buf[m_lstPAL[targetY].Y].Remove(
                    startPX,
                    m_lstPAL[targetY + 1].X - startPX);
            }
        }
        #endregion
        #endregion

        #region override メソッド
        /// <summary>
        /// ウィンドウのスタイルを取得する
        /// </summary>
        /// <remarks>
        /// VScrollBarクラスやHScrollBarクラスは残念ながらウィンドウが別々に作成されてしまうので、
        /// API操作でスクロール処理を実装するものとした。
        /// 本メソッドはControlクラスの基本スタイルにWS_VSCROLLとWS_HSCROLLを追加して返している
        /// </remarks>
        protected override CreateParams CreateParams
        {
            get
            {
                CreateParams ret = base.CreateParams;
                ret.Style = ret.Style | ScrollMagician.WS_VSCROLL | ScrollMagician.WS_HSCROLL;
                return ret;
            }
        }

        /// <summary>
        /// 渡された文字が処理対象文字かを判定する
        /// </summary>
        /// <param name="charCode">判定する文字</param>
        /// <returns>true:処理対象 false:処理非対象</returns>
        protected override bool IsInputChar(char charCode)
        {
            return (!char.IsControl(charCode)) || (charCode == '\t');
        }

        /// <summary>
        /// 渡されたキーが処理対象キーかを判定する
        /// </summary>
        /// <param name="keyData">判定するキー</param>
        /// <returns>true:処理対象キー false:処理非対象</returns>
        protected override bool IsInputKey(Keys keyData)
        {
            // ShortCutの可能性のあるキー
            if (((keyData & Keys.Control) | (keyData & Keys.Alt)) != 0)
                return base.IsInputKey(keyData);

            // 移動キーかの判定
            if (KeysMagician.isMoveKey(keyData))
                return true;
            
            // その他処理キー
            if ((keyData == Keys.Return) || (keyData == Keys.Tab))
                return true;
            return base.IsInputKey(keyData);
        }
        #endregion

        #region インターフェース以外のpublicメンバ
        /// <summary>
        /// 物理アドレスと論理アドレスのマッパを取得します
        /// </summary>
        /// <remarks>
        /// 本プロパティは<seealso cref="SelectionArea"/>プロパティを操作する場合に必要になります。
        /// <seealso cref="SelectionArea"/>プロパティのY座標は本プロパティのインデックスを表します。
        /// また、<seealso cref="SelectionArea"/>プロパティのX座標は本プロパティのX座標との相対座標です。
        /// この結果と<seealso cref="Lines"/>プロパティを組み合わせてドキュメントの詳細な調査を行う事が出来ます。
        /// 以下に例を示します。
        /// <example>
        /// <code>
        /// // 現在カーソル位置から左側の文字列を取得する
        /// // PhysicalAddressListのXではなく0を指定すれば論理行全体を取得出来る
        /// Lines[PhysicalAddressList[SelectionArea.CurrentY].Y].Substring(
        ///     PhysicalAddressList[SelectionArea.CurrentY].X, SelectionArea.CurrentX + 1
        /// );
        /// 
        /// // 文章のX,Y座標を取得
        /// // ※本例で掲載するコードと同一の処理はGetLastXYメソッドを使用する事が推奨される
        /// // 文章の最終行のY座標を取得する
        /// PhysicalAddressList.Count - 1;
        /// // 文章の最終行の最終文字列X座標を取得する
        /// // 最後の-1はLinesプロパティの末尾に常に'\n'が含まれている点の対処
        /// Lines[PhysicalAddressList[PhysicalAddressList.Count - 1].Y].Length
        ///     - PhysicalAddressList[PhysicalAddressList.Count - 1].X - 1;
        /// </code>
        /// </example>
        /// </remarks>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public List<System.Drawing.Point> PhysicalAddressList
        {
            get { return this.m_lstPAL; }
        }

        /// <summary>
        /// 文章先頭の物理X,Y座標を取得する
        /// </summary>
        /// <returns>文章先頭の物理X,Y座標を格納する<seealso cref="System.Drawing.Point"/>構造体</returns>
        public virtual System.Drawing.Point GetFirstXY()
        {
            return new System.Drawing.Point(0, 0);
        }

        /// <summary>
        /// 文章末尾の物理X,Y座標を取得する
        /// </summary>
        /// <returns>文章末尾の物理X,Y座標を格納する<seealso cref="System.Drawing.Point"/>構造体</returns>
        public virtual System.Drawing.Point GetLastXY()
        {
            return new System.Drawing.Point(
                getPALLineLength(m_lstPAL.Count - 1) - 1,
                m_lstPAL.Count - 1
                );
        }
        #endregion

        #region イベントハンドラ
        #region 変更系イベントハンドラ
        /// <summary>
        /// フォント変更時に全てのリソース情報を更新する
        /// </summary>
        /// <param name="e">空</param>
        protected override void OnFontChanged(EventArgs e)
        {
            updateAllResource();
            base.OnFontChanged(e);
        }

        /// <summary>
        /// サイズ変更時にウィンドウに関するリソース情報を更新する
        /// </summary>
        /// <param name="e">空</param>
        protected override void OnResize(EventArgs e)
        {
            updateWindowResource();
            base.OnResize(e);
            // MEMO:これをやらないと一部だけしか描画されないので念のため
            this.Refresh();
        }
        #endregion

        #region 描画系イベントハンドラ
        /// <summary>
        /// 描画イベントハンドラ
        /// </summary>
        /// <param name="e">イベントパラメータ</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            if (e.ClipRectangle.Equals(this.ClientRectangle))
            {
                try
                {
                    render();
                }
                catch (DirectXException)
                {
#if DEBUG
                    Console.WriteLine("Render Error");
#endif
                    // ホントは何か例外を出した方がいいか？
                    if (InitializeGraphics() == false)
                    {
#if DEBUG
                        Console.WriteLine("InitializeGraphics Error");
#endif
                        return;
                    }
                    render();
                }
            }
            else
            {
                // 合ってる？
                try
                {
                    m_device.Present();
                }
                catch (DirectXException)
                {
#if DEBUG
                    Console.WriteLine("Present Error");
#endif
                    // ホントは何か例外を出した方がいいか？
                    if (InitializeGraphics() == false)
                    {
#if DEBUG
                        Console.WriteLine("InitializeGraphics Error");
#endif
                        return;
                    }
                    render();
                }
            }

            this.RaisePaintEvent(this, e);
        }
        #endregion

        #region キー操作系イベントハンドラ
        /// <summary>
        /// 文字入力のキーイベント（IMEは別処理）
        /// </summary>
        /// <param name="e">入力されたキー情報を含むイベントパラメータ</param>
        protected override void OnKeyPress(KeyPressEventArgs e)
        {
            // base.WndProcはやってくれないようなので独自判定
            if (IsInputChar(e.KeyChar) == false)
                return;
            inputString(e.KeyChar.ToString());
            e.Handled = true;
            this.Refresh();
        }

        /// <summary>
        /// キーが押された時に発生するイベント
        /// </summary>
        /// <param name="e">入力されたキー情報を含むイベントパラメータ</param>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            // 一応Control.OnKeyDownを実行
            // TODO:処理が不要でかつ重い場合はカットすること
            base.OnKeyDown(e);
            Keys key = e.KeyCode;

            bool defaultFlag = false;
            // bool refreshMode = false;
            int tmpVPos = m_vScrollInfo.nPos;
            int tmpHPos = m_hScrollInfo.nPos;

            // TODO:矩形選択は未実装だよーん
            if (e.Shift) {
                m_selectMode = (((key >= Keys.Prior) && (key <= Keys.Down)) || (key == Keys.None) || (key == Keys.ControlKey) || (key == Keys.ShiftKey)) ? SelectMode.Normal : SelectMode.None;
            } else {
                m_selectMode = SelectMode.None;
            }

            // 選択開始処理
            if ((m_selectMode != SelectMode.None) && (m_selectionArea.StartX == -1))
            {
                updateSelectArea();
            }

            // SHIFTが押されている場合はstartXやstartYの処理も必要
            // SHIFTが押されていない場合は選択範囲は全解除で可
            switch (key)
            {
                case Keys.Enter:
                    // 改行処理
                    inputString("\n");
                    break;
                case Keys.Back:
                    // バックスペース
                    Delete(DeleteMode.Backspace);
                    // refreshMode = true;
                    break;
                case Keys.Delete:
                    // delete
                    Delete(DeleteMode.Delete);
                    // refreshMode = true;
                    break;
                case Keys.Right:
                    // →
                    cursorRight(1);
                    break;
                case Keys.Left:
                    // ←
                    cursorLeft(1);
                    break;
                case Keys.Down:
                    // ↓
                    cursorDown(1);
                    break;
                case Keys.Up:
                    // ↑
                    cursorUp(1);
                    break;
                case Keys.PageUp:
                    // PageUp
                    cursorUp(m_maxHeight / m_lineHeight);
                    break;
                case Keys.PageDown:
                    // PageDown
                    cursorDown(m_maxHeight / m_lineHeight);
                    break;
                case Keys.End:
                    // End
                    // Ctrl+Endで終了までの移動などは呼び出し元にやらせりゃいいかなと
                    int endIndex;
                    for (endIndex = m_selectionArea.CurrentY + 1; endIndex < m_buf.Count; endIndex++)
                    {
                        if (m_lstPAL[endIndex].Y != m_lstPAL[m_selectionArea.CurrentY].Y)
                            break;
                    }
                    endIndex--;
                    setCursor(m_buf[m_lstPAL[endIndex].Y].Length - m_lstPAL[endIndex].X - 1, endIndex, false, true);
                    break;
                case Keys.Home:
                    // Home
                    int startIndex;
                    for (startIndex = m_selectionArea.CurrentY - 1; startIndex >= 0; startIndex--)
                    {
                        if (m_lstPAL[startIndex].Y != m_lstPAL[m_selectionArea.CurrentY].Y)
                            break;
                    }
                    setCursor(0, startIndex + 1, false, true);
                    break;
                default:
                    // その他
                    defaultFlag = true;
                    break;
            }

            // デフォルト処理切り替え
            if (defaultFlag)
            {
                e.SuppressKeyPress = false;
            }
            else
            {
                e.SuppressKeyPress = true;
                this.Refresh();
                /*
                // MEMO:選択範囲更新の関係で毎回全更新をしなければならなくなった。テキストバッファのみの転送さえ出来れば選択範囲のみ毎回全更新すれば済むのだが。。
                if ((refreshMode) || (tmpVPos != m_vScrollInfo.nPos) || (tmpHPos != m_hScrollInfo.nPos))
                    this.Refresh();
                */
            }
        }
        #endregion

        #region マウス操作系イベントハンドラ
        /// <summary>
        /// マウスボタン押下時のイベントハンドラ
        /// </summary>
        /// <param name="e">イベントパラメータ</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            // クリックされてもフォーカスが当たらない仕様の対処
            if (this.Focused == false)
                this.Focus();

            // 選択位置修正処理
            m_selectMode = SelectMode.None;
            System.Drawing.Point getPosition = getXYFromDisplayPoint(e.Location);
            // 選択範囲修正処理
            clearSelectArea();
            m_selectMode = SelectMode.Normal;
            setCursor(getPosition.X, getPosition.Y, false, true);

            // 画面更新
            this.Refresh();
            m_dragMode = true;
        }

        /// <summary>
        /// マウス動作時のイベントハンドラ。ドラッグ中ならカーソル移動を行う
        /// </summary>
        /// <param name="e"></param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);
            if (m_dragMode)
            {
                System.Drawing.Point legalPoint = e.Location;
                TextSelectionArea oldArea = m_selectionArea;
                if (legalPoint.X < 0)
                    legalPoint.X = 0;
                System.Drawing.Point getPosition = getXYFromDisplayPoint(legalPoint);
                // フォントがでかすぎた場合の上移動補正
                if ((e.Location.Y < 0) && (getPosition.Y == m_selectionArea.CurrentY) && (getPosition.Y > 0))
                    getPosition.Y--;
                if ((m_selectionArea.CurrentX != getPosition.X) || (m_selectionArea.CurrentY != getPosition.Y))
                {
                    setCursor(getPosition.X, getPosition.Y, false, true);
                    this.Refresh();
                }
            }
        }

        /// <summary>
        /// マウスボタンを離した時のイベントハンドラ
        /// </summary>
        /// <param name="e"></param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);
            if (m_selectionArea.StartX == -1)
            {
                m_selectMode = SelectMode.None;
            }
            m_dragMode = false;
        }

        #endregion

        #region WndProc
        /// <summary>
        /// ウィンドウメッセージプロシージャ
        /// </summary>
        /// <remarks>
        /// 今のところスクロール処理とIME制御でのみ使用。
        /// スクロール処理にはマウスホイールに対する処理も含むので注意
        /// </remarks>
        /// <param name="m">ウィンドウメッセージ</param>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case IMEMagician.WM_IME_STARTCOMPOSITION:
                    m_caret.Hide();
                    setImeContext();
                    break;
                case IMEMagician.WM_IME_ENDCOMPOSITION:
                    m_caret.Show();
                    this.Refresh(); // MEMO:これをしないと更新かからないので注意
                    break;
                case IMEMagician.WM_IME_CHAR:
                    // 切ないほど強引だがWM_IME_CHARはKeyPressではなく独自処理
                    // RefreshをWM_IME_ENDCOMPOSITIONメッセージで行うことで
                    // 処理を単一化することが狙い。
                    // ちなみにここでbreakするとWM_CHARが発生する。
                    inputString(((char)m.WParam.ToInt32()).ToString());
                    return;
                case ScrollMagician.WM_MOUSEWHEEL:
                    moveVScroll(-((short)(((uint)m.WParam >> 16) & 0xFFFF) / ScrollMagician.WHEEL_DELTA));
                    break;
                case ScrollMagician.WM_HSCROLL:
                    // 横スクロール
                    switch ((short)(m.WParam))    // LOWORD(wParam)
                    {
                        case ScrollMagician.SB_LINEUP:
                            break;
                        case ScrollMagician.SB_LINEDOWN:
                            break;
                        case ScrollMagician.SB_PAGEUP:
                            break;
                        case ScrollMagician.SB_PAGEDOWN:
                            break;
                        case ScrollMagician.SB_THUMBTRACK:
                            break;
                        default:
                            break;
                    }
                    //highWParam = (short)(((uint)wParam >> 16) & 0xFFFF);
                    break;
                case ScrollMagician.WM_VSCROLL:
                    // 縦スクロール
                    int vPoint;
                    switch ((short)(m.WParam))    // LOWORD(wParam)
                    {
                        case ScrollMagician.SB_LINEUP:
                            vPoint = -1;
                            break;
                        case ScrollMagician.SB_LINEDOWN:
                            vPoint = 1;
                            break;
                        case ScrollMagician.SB_PAGEUP:
                            vPoint = -(int)this.m_vScrollInfo.nPage;
                            break;
                        case ScrollMagician.SB_PAGEDOWN:
                            vPoint = (int)this.m_vScrollInfo.nPage;
                            break;
                        case ScrollMagician.SB_THUMBTRACK:
                            vPoint = ((short)(((uint)m.WParam >> 16) & 0xFFFF)) - this.m_vScrollInfo.nPos;
                            if (vPoint < 0)
                            {
                            }
                            break;
                        default:
                            vPoint = 0;
                            break;
                    }
                    moveVScroll(vPoint);
                    break;
            }
            // TODO:一応全て転送。特に問題なさそうならdefault句に移すこと。
            base.WndProc(ref m);
        }

        #endregion
        #endregion

        #region TextOperationIF メンバ
        /// <summary>
        /// 現在の選択領域をクリップボードにコピーする
        /// </summary>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Copy()
        {
            Copy(m_selectionArea);
        }

        /// <summary>
        /// 指定された選択領域をクリップボードにコピーする
        /// </summary>
        /// <param name="targetArea">コピー対象選択領域</param>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Copy(TextSelectionArea targetArea)
        {
            if (this.ReadOnly)
                return;

            // イベント送出とキャンセル処理
            BeforeTextChangedEventArgs e = new BeforeTextChangedEventArgs(TextBoxOperation.Copy);
            OnBeforeTextChanged(e);
            if (e.Cancel)
                return;

            if (this.m_clipboard != null)
            {
                StringBuilder buf = new StringBuilder();
                System.Drawing.Rectangle[] selectRect = getSelectArea();
                for (int i = 0; i < selectRect.Length; i++)
                {
                    for (int j = selectRect[i].Y; j <= (selectRect[i].Bottom - 1); j++)
                    {
                        // 一行ずつ抽出（矩形選択にも対応するため）
                        if (selectRect[i].Width == -1)  // 矩形はこの中に来ない
                        {
                            if (isNextLineFromNextPALLine(j))
                            {
                                buf.AppendLine(
                                    m_buf[m_lstPAL[j].Y].Substring(m_lstPAL[j].X + selectRect[i].X, getPALLineLength(j) - 1)
                                );
                            }
                            else
                            {
                                buf.Append(
                                    m_buf[m_lstPAL[j].Y].Substring(m_lstPAL[j].X + selectRect[i].X, getPALLineLength(j))
                                );
                            }
                        }
                        else
                        {
                            if (m_selectMode == SelectMode.Rectangle)
                            {
                                buf.AppendLine(
                                    m_buf[m_lstPAL[j].Y].Substring(m_lstPAL[j].X + selectRect[i].X, selectRect[i].Width)
                                );
                            }
                            else
                            {
                                buf.Append(
                                    m_buf[m_lstPAL[j].Y].Substring(m_lstPAL[j].X + selectRect[i].X, selectRect[i].Width)
                                );
                            }
                        }
                    }
                }
                string copyStr = buf.ToString();
                // TODO:矩形選択時の補正
                if ((m_selectMode == SelectMode.Rectangle) && (copyStr.Length > 2))
                    copyStr = (copyStr[copyStr.Length - 3] == '\r') ? copyStr.Substring(0, copyStr.Length - 2) : copyStr.Substring(0, copyStr.Length - 1);
                this.m_clipboard.SetText(copyStr);
            }
        }

        /// <summary>
        /// 現在の選択領域の文字列を削除する
        /// </summary>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Delete()
        {
            Delete(DeleteMode.Delete);
        }

        /// <summary>
        /// 指定された選択領域の文字列を削除する
        /// </summary>
        /// <param name="targetArea">対象選択領域</param>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Delete(TextSelectionArea targetArea)
        {
            Delete(targetArea, DeleteMode.Delete);
        }

        /// <summary>
        /// 現在の選択領域の文字列を削除する
        /// </summary>
        /// <param name="backCursor">選択領域がない場合の削除モードを指定する</param>
        public void Delete(DeleteMode deleteMode)
        {
            // 正確にはTextBoxOperationIFのメンバではないがここに配属
            // 読み取り専用判定
            if (this.ReadOnly)
                return;

            // イベント送出とキャンセル処理
            BeforeTextChangedEventArgs e = new BeforeTextChangedEventArgs(
                (deleteMode == DeleteMode.Backspace) ? TextBoxOperation.Backspace : TextBoxOperation.Delete
            );
            OnBeforeTextChanged(e);
            if (e.Cancel)
                return;

            char[] lineChars = this.m_buf[m_lstPAL[m_selectionArea.CurrentY].Y].ToCharArray();
            bool deleteSelect = false;
            int minY = 0, minX = 0, prevDisplayX = 0;

            // 共通の選択範囲削除処理
            if (m_selectionArea.StartX != -1)
            {
                System.Drawing.Rectangle[] selectRect = getSelectArea(m_selectionArea);
                if (selectRect.Length == 0)
                    goto NORMAL_DELETE;
                minY = int.MaxValue;
                minX = int.MaxValue;
                for (int i = 0; i < selectRect.Length; i++)
                {
                    if (minY > selectRect[i].Y)
                    {
                        minY = selectRect[i].Y;
                        minX = selectRect[i].X;
                    }
                }
                prevDisplayX = m_textArea.X +
                    getStringWidth(m_buf[m_lstPAL[minY].Y].ToCharArray(),
                        m_lstPAL[minY].X,
                        minX,
                        false);
                // 後ろから削除
                // MEMO:前からだとPALが狂うのでちとめんどくなると思われる
                for (int i = selectRect.Length - 1; i >= 0; i--)
                {
                    // 一行ずつ処理
                    for (int j = selectRect[i].Bottom - 1; j >= selectRect[i].Y; j--)
                    {
                        if (selectRect[i].Width == -1)  // 矩形時はこの中に入らない予定（つまり行削除はない）
                        {
                            if (selectRect[i].X == 0)
                            {
                                // 一行丸ごと削除
                                // 物理行削除、論理行連結または削除
                                if (
                                    (m_lstPAL[j].X == 0) &&                     // Xが論理行最初を指し
                                    (
                                        ((j + 1) == m_lstPAL.Count) ||          // Jが物理行最終を指すか
                                        (m_lstPAL[j + 1].Y != m_lstPAL[j].Y)    // 論理行が一つの物理行しかない
                                    )
                                   )
                                {
                                    // 行ごと削除
                                    m_buf.RemoveAt(m_lstPAL[j].Y);
                                }
                                else
                                {
                                    // 連結
                                    deleteJoin(m_lstPAL[j].X, j);
                                }
                            }
                            else
                            {
                                // 行の途中から終わりまで削除
                                // 物理行生存、論理行連結
                                deleteJoin(m_lstPAL[j].X + selectRect[i].X, j);
                            }
                        }
                        else
                        {
                            // ある文字からある文字まで削除
                            // 物理行、論理行共に生存
                            m_buf[m_lstPAL[j].Y] = m_buf[m_lstPAL[j].Y].Remove(m_lstPAL[j].X + selectRect[i].X, selectRect[i].Width);
                        }
                    }
                }
                deleteSelect = (selectRect.Length != 0);
            }

            if (deleteSelect)
            {
                MustUpdatePhysicalAddressList(minY);
                // MEMO:このカーソル補正が一番自然な模様
                setCursor(minX, minY, true, true);
                updateVScrollMax();
                return;
            }

            // 通常削除処理（リファクタリング余地あり）
            NORMAL_DELETE:
            bool bDelete = (deleteMode != DeleteMode.None);
            if (bDelete)
            {
                if (deleteMode == DeleteMode.Backspace)
                {
                    // バックスペース時はcursorLeft(1) で カーソルが動いた場合 DeleteMode.Delete を 実行
                    TextSelectionArea oldArea = m_selectionArea;
                    cursorLeft(1);
                    if ((oldArea.CurrentX == m_selectionArea.CurrentX)
                        && (oldArea.CurrentY == m_selectionArea.CurrentY))
                        return;
                }
                if ((m_lstPAL[m_selectionArea.CurrentY].X + m_selectionArea.CurrentX) < (lineChars.Length - 1))
                {
                    m_buf[m_lstPAL[m_selectionArea.CurrentY].Y] = m_buf[m_lstPAL[m_selectionArea.CurrentY].Y].Remove(m_lstPAL[m_selectionArea.CurrentY].X + m_selectionArea.CurrentX, 1);
                    UpdatePhysicalAddressList(m_selectionArea.CurrentY);
                }
                else
                {
                    if (m_selectionArea.CurrentY >= (m_lstPAL.Count - 1))
                        return;
                    // Collectionをずらす処理
                    string joinText = m_buf[m_lstPAL[m_selectionArea.CurrentY].Y].Substring(0, m_buf[m_lstPAL[m_selectionArea.CurrentY].Y].Length - 1);
                    m_buf[m_lstPAL[m_selectionArea.CurrentY].Y] = joinText + m_buf[m_lstPAL[m_selectionArea.CurrentY].Y + 1];
                    m_buf.RemoveAt(m_lstPAL[m_selectionArea.CurrentY].Y + 1);
                    MustUpdatePhysicalAddressList(m_selectionArea.CurrentY);
                    updateVScrollMax();
                }
            }
        }

        /// <summary>
        /// 指定された選択領域の文字列を削除する
        /// </summary>
        /// <param name="targetArea">対象選択領域</param>
        /// <param name="backCursor">選択領域がない場合の削除モードを指定する</param>
        public void Delete(TextSelectionArea targetArea, DeleteMode deleteMode)
        {
            // 正確にはTextBoxOperationIFのメンバではないがここに配属
            m_selectionArea = targetArea;
            Delete(deleteMode);
        }

        /// <summary>
        /// 現在の選択領域の文字列をクリップボードにコピーしてから削除する
        /// </summary>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Cut()
        {
            Copy();
            Delete();
        }

        /// <summary>
        /// 指定された選択領域の文字列をクリップボードにコピーしてから削除する
        /// </summary>
        /// <param name="targetArea">対象選択領域</param>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Cut(TextSelectionArea targetArea)
        {
            m_selectionArea = targetArea;
            Cut();
        }

        /// <summary>
        /// 現在のカーソル位置にクリップボードの文字列を挿入する。
        /// 選択領域が存在すれば、選択領域の文字列をクリップボードの文字列で上書きする。
        /// </summary>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Paste()
        {
            if (this.ReadOnly == false)
            {
                BeforeTextChangedEventArgs e = new BeforeTextChangedEventArgs(TextBoxOperation.Paste);
                OnBeforeTextChanged(e);
                if (e.Cancel)
                    return;
                Insert(m_clipboard.GetText());
            }
        }

        /// <summary>
        /// 指定された領域のカーソル位置にクリップボードの文字列を挿入する。
        /// 選択領域が存在すれば、選択領域の文字列をクリップボードの文字列で上書きする。
        /// </summary>
        /// <param name="targetArea">対象選択領域</param>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Paste(TextSelectionArea targetArea)
        {
            m_selectionArea = targetArea;
            Paste();
        }

        /// <summary>
        /// 現在管理している文字列の最後尾に文字列を追加する
        /// </summary>
        /// <param name="value">追加する文字列</param>
        public void Append(string value)
        {
            clearSelectArea();
            m_selectionArea.CurrentY = m_lstPAL.Count - 1;
            m_selectionArea.CurrentX = getPALLineLength(m_selectionArea.CurrentY) - 1;
            Insert(value, false);
        }

        /// <summary>
        /// 現在のカーソル位置に文字列を挿入する。
        /// 選択領域が存在すれば、選択領域の文字列を削除してから挿入する。
        /// </summary>
        /// <param name="value">挿入する文字列</param>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Insert(string value)
        {
            Insert(value, false);
        }

        /// <summary>
        /// 現在のカーソル位置から文字列を挿入または上書きする。
        /// 選択領域が存在すれば、選択領域の文字列を削除してから挿入または上書きする。
        /// </summary>
        /// <param name="value">対象文字列</param>
        /// <param name="isOverwrite">true:上書き false:挿入</param>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Insert(string value, bool isOverwrite)
        {
            // 読み取り専用判定
            if (this.ReadOnly)
                return;

            // 挿入処理
            // Changedイベント送出
            BeforeTextChangedEventArgs e = new BeforeTextChangedEventArgs(
                value,
                TextBoxOperation.InputString
            );
            OnBeforeTextChanged(e);
            if (e.Cancel)
            {
                return;
            }

            // 選択範囲の削除
            if (m_selectionArea.StartX != -1)
            {
                Delete(DeleteMode.None);
            }

            // 入力処理用utilメソッドコール
            string[] targetStrings = getInputStrings(ref value);
            multiInsert(targetStrings, isOverwrite, 0, targetStrings.Length);
        }

        /// <summary>
        /// 指定された領域のカーソル位置に文字列を挿入する。
        /// 選択領域が存在すれば、選択領域の文字列を削除してから挿入する。
        /// </summary>
        /// <param name="value">挿入する文字列</param>
        /// <param name="targetArea">対象領域</param>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Insert(string value, TextSelectionArea targetArea)
        {
            Insert(value, targetArea, false);
        }

        /// <summary>
        /// 指定された領域のカーソル位置に文字列を挿入または上書きする。
        /// 選択領域が存在すれば、選択領域の文字列を削除してから挿入または上書きする。
        /// </summary>
        /// <param name="value">対象文字列</param>
        /// <param name="targetArea">対象領域</param>
        /// <param name="isOverwrite">true:上書き false:挿入</param>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        public void Insert(string value, TextSelectionArea targetArea, bool isOverwrite)
        {
            m_selectionArea = targetArea;
            Insert(value, isOverwrite);
        }

        /// <summary>
        /// このテキストボックスが管理している文字列を一行単位で取得する
        /// </summary>
        /// <exception cref="System.IndexOutOfRangeException">選択領域に不正な値が指定された</exception>
        [Browsable(false)]  // MEMO:trueでもいいが
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public System.Collections.Specialized.StringCollection Lines
        {
            get { return m_buf; }
        }

        /// <summary>
        /// このテキストボックスが管理している文字列を取得または設定する
        /// </summary>
        [Editor("System.ComponentModel.Design.MultilineStringEditor, System.Design, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a", typeof(System.Drawing.Design.UITypeEditor))]
        [Localizable(true)]
        [Browsable(true)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        public override string Text
        {
            get
            {
                StringBuilder buf = new StringBuilder(Lines.Count * 64);
                for (int i = 0; i < Lines.Count-1; i++)
                {
                    //buf.AppendLine(Lines[i].Substring(0, Lines[i].Length - 1));
                    buf.Append(Lines[i]);
                }
                buf.Append(Lines[Lines.Count - 1].Substring(0, Lines[Lines.Count - 1].Length - 1));
                return buf.ToString();
            }
            set
            {
                // 重っ..
                m_selectMode = SelectMode.Normal;
                m_selectionArea = new TextSelectionArea(0, 0, 0, 0, getPALLineLength(m_lstPAL.Count - 1) - 1, m_lstPAL.Count - 1);
                if (m_selectionArea.CurrentX == -1)
                    m_selectionArea.CurrentX = 0;
                Insert(value);
                m_selectMode = SelectMode.None;
                m_selectionArea.CurrentX = 0;
                m_selectionArea.CurrentY = 0;
                updateTextArea();
                if (DesignMode)
                    this.Refresh();
            }
        }


        #endregion

        #region StdTextBoxIF メンバ

        #region イベント
        /// <summary>
        /// テキストボックスの内容が更新される直前に発生するイベントを定義する
        /// </summary>
        [Description("テキストボックスの内容が更新されようとした時に発生するイベントです")]
        public event BeforeTextChangedEventHandler BeforeTextChanged;

        /// <summary>
        /// <c>CursorMode</c>プロパティの値が変更されると発生するイベントを定義する
        /// </summary>
        [Description("CursorModeプロパティの値が更新された時に発生するイベントです")]
        public event CursorModeChangedEventHandler CursorModeChanged;

        /// <summary>
        /// <c>InputMode</c>プロパティの値が変更されると発生するイベントを定義する
        /// </summary>
        [Description("InputModeプロパティの値が更新された時に発生するイベントです")]
        public event InputModeChangedEventHandler InputModeChanged;

        /// <summary>
        /// <c>SelectMode</c>プロパティの値が変更されると発生するイベントを定義する
        /// </summary>
        [Description("SelectModeプロパティの値が更新された時に発生するイベントです")]
        public event SelectModeChangedEventHandler SelectModeChanged;
        #endregion

        #region プロパティ
        /// <summary>
        /// テキストボックスの特殊コマンドとそれに割り当てるキーを取得する
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("テキストボックスの特殊コマンドとそれに割り当てるキーを指定します")]
        public TextBoxCustomCommand CustomCommand
        {
            get { return this.m_customCommand; }
        }

        /// <summary>
        /// 表示する線を取得または設定する
        /// </summary>
        /// <remarks>
        /// 本プロパティの値を変えた結果を画面に即表示する場合、
        /// <seealso cref="Refresh"/>メソッドを呼び出す必要がある。
        /// </remarks>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("表示する線を指定します")]
        [Browsable(false)]  // TODO:ほんとはtrue
        public ShowLines ShowLines
        {
            get { return this.m_showLines; }
            set { this.m_showLines = value; }
        }

        /// <summary>
        /// テキストボックスの折り返し行数を取得または設定する。
        /// 本プロパティに設定出来る有効な値は2 ～ 4096までです。
        /// </summary>
        /// <remarks>
        /// 本プロパティ<c>AutoTextTurnPoint</c>プロパティがfalseの場合のみ有効となる
        /// </remarks>
        [Description("文字列の折り返し桁数を指定します。AutoTextTurnプロパティがtrueの場合、本プロパティの値を変更することは出来ません。")]
        [Browsable(true)]
        public ushort TextTurnPoint
        {
            get { return this.m_textTurnPoint; }
            set
            {
                if (AutoTextTurnPoint)
                    return;
                if ((value < 2) || (value > 4096))
                    return;
                this.m_textTurnPoint = value;
                updateWindowResource();
                this.Refresh();
            }
        }

        /// <summary>
        /// テキストボックスの折り返しを自動的に設定するかを取得または設定する
        /// </summary>
        /// <remarks>
        /// 本プロパティをtrueにした場合、<c>TextTurnPoint</c>プロパティの設定内容は無視され、また自動的に同プロパティは更新される。
        /// 本プロパティをtrueにした場合の<c>TextTurnPoint</c>の値は、原則としてウィンドウの描画領域横幅 / フォントの横幅である。
        /// </remarks>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        [Description("ウィンドウのサイズに合わせて自動的に折り返し点を更新するかを指定します")]
        [Browsable(true)]
        public bool AutoTextTurnPoint
        {
            get { return this.m_autoTextTurnPoint; }
            set
            {
                this.m_autoTextTurnPoint = value;
                updateWindowResource();
                this.Refresh();
            }
        }

        /// <summary>
        /// テキストボックスの選択状態を取得または設定する
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [Description("テキストボックスの選択状態を指定します")]
        public SelectMode SelectMode
        {
            get { return this.m_selectMode; }
            set
            {
                if (value == this.m_selectMode)
                    return;
                SelectMode oldValue = this.m_selectMode;
                this.m_selectMode = value;
                OldValueCancelEventArgs<SelectMode> e = new OldValueCancelEventArgs<SelectMode>(oldValue);
                OnSelectModeChanged(e);
                if (e.Cancel)
                {
                    this.m_selectMode = e.OldValue;
                }
            }
        }

        /// <summary>
        /// テキストボックスのカーソルモードを取得または設定する
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("テキストボックスのカーソルモードを指定します")]
        public CursorMode CursorMode
        {
            get { return this.m_cursorMode; }
            set
            {
                if (value == this.m_cursorMode)
                    return;
                CursorMode oldValue = this.m_cursorMode;
                this.m_cursorMode = value;
                OldValueCancelEventArgs<CursorMode> e = new OldValueCancelEventArgs<CursorMode>(oldValue);
                OnCursorChanged(e);
                if (e.Cancel)
                {
                    this.m_cursorMode = e.OldValue;
                }
            }
        }

        /// <summary>
        /// 行番号の表示モードを取得または設定する
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("テキストボックスの行番号表示モードを指定します")]
        public LineNumberMode LineNumberMode
        {
            get
            {
                return this.m_lineNumberMode;
            }
            set
            {
                this.m_lineNumberMode = value;
                updateWindowResource();
                this.Refresh();
            }
        }

        /// <summary>
        /// テキストボックスに割り当てられているクリップボードを取得または設定する
        /// </summary>
        /// <remarks>
        /// 本プロパティの操作により、履歴を持つクリップボードと本クラスの連動などが可能になる
        /// </remarks>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public CustomClipboardIF Clipboard
        {
            get
            {
                return this.m_clipboard;
            }
            set
            {
                this.m_clipboard = value;
            }
        }

        /// <summary>
        /// タブ文字の文字数を取得または設定する。
        /// </summary>
        [Description("タブ文字の文字数です")]
        public ushort TabSpace
        {
            get
            {
                return this.m_tabSpace;
            }
            set
            {
                this.m_tabSpace = value;
                // 全更新
                MustUpdatePhysicalAddressList(0);
                updateTextArea();
            }
        }

        /// <summary>
        /// TabキーでTab文字を入力させるかどうかの値を取得または設定します
        /// </summary>
        /// <remarks>
        /// 本プロパティの値がtrueの場合、挙動は次のようになります。
        /// SHIFT + TAB 機能せず // TODO:本当はTAB文字の削除
        /// CTRL  + TAB 一つ先のコントロールに移動
        /// TAB         TAB文字の入力
        /// 
        /// 本プロパティがfalseの場合、挙動は次のようになります。
        /// TAB         一つ先のコントロールに移動
        /// SHIFT + TAB 一つ前のコントロールに移動
        /// CTRL  + TAB TAB文字の入力
        /// </remarks>
        [Description("TabキーでTab文字を入力させるかどうかを指定します")]
        public bool AcceptsTab
        {
            get
            {
                return m_acceptsTab;
            }
            set
            {
                m_acceptsTab = value;
            }
        }

        #region stub
        /// <summary>
        /// EOFのマークを取得または設定する。
        /// 表示しない場合、本プロパティはnullに設定されるべきである。
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("EOF記号のイメージを指定します")]
        public System.Drawing.Image EofImage
        {
            get
            {
                throw new Exception("The method or operation is not implemented.");
            }
            set
            {
                throw new Exception("The method or operation is not implemented.");
            }
        }

        /// <summary>
        /// 表示できない種類の文字に対して置き換える画像を取得または設定する。
        /// 表示しない場合、本プロパティはnullに設定されるべきである。
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("表示出来ない文字のイメージを指定します")]
        public System.Drawing.Image UnknownCodeImage
        {
            get
            {
                throw new Exception("The method or operation is not implemented.");
            }
            set
            {
                throw new Exception("The method or operation is not implemented.");
            }
        }

        /// <summary>
        /// ラインフィードに対して置き換える画像を取得または設定する。
        /// 表示しない場合、本プロパティはnullに設定されるべきである。
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("改行記号のイメージを指定します")]
        public System.Drawing.Image LineFiedImage
        {
            get
            {
                throw new Exception("The method or operation is not implemented.");
            }
            set
            {
                throw new Exception("The method or operation is not implemented.");
            }
        }

        /// <summary>
        /// TAB文字に対して置き換える画像を取得または設定する。
        /// 表示しない場合、本プロパティはnullに設定されるべきである。
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("タブ記号のイメージを指定します")]
        public System.Drawing.Image TabImage
        {
            get
            {
                throw new Exception("The method or operation is not implemented.");
            }
            set
            {
                throw new Exception("The method or operation is not implemented.");
            }
        }

        /// <summary>
        /// 全角スペース文字に対して置き換える画像を取得または設定する。
        /// 表示しない場合、本プロパティはnullに設定されるべきである。
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("全角スペースのイメージを指定します")]
        public System.Drawing.Image WideSpaceImage
        {
            get
            {
                throw new Exception("The method or operation is not implemented.");
            }
            set
            {
                throw new Exception("The method or operation is not implemented.");
            }
        }

        /// <summary>
        /// 半角スペース文字に対して置き換える画像を取得または設定する。
        /// 表示しない場合、本プロパティはnullに設定されるべきである。
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("半角スペースのイメージを指定します")]
        public System.Drawing.Image SpaceImage
        {
            get
            {
                throw new Exception("The method or operation is not implemented.");
            }
            set
            {
                throw new Exception("The method or operation is not implemented.");
            }
        }
        #endregion

        #endregion
        #endregion

        #region TextBoxIF メンバ
        /// <summary>
        /// テキストボックスの入力モードを取得または設定する
        /// </summary>
        [Browsable(false)]  // TODO:ほんとはtrue
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("テキストボックスの入力モードを指定します")]
        public TextInputMode InputMode
        {
            get { return this.m_inputMode; }
            set
            {
                if (value == this.m_inputMode)
                    return;
                TextInputMode oldValue = this.m_inputMode;
                OldValueCancelEventArgs<TextInputMode> e = new OldValueCancelEventArgs<TextInputMode>(oldValue);
                this.m_inputMode = value;
                OnInputModeChanged(e);
                if (e.Cancel)
                {
                    this.m_inputMode = e.OldValue;
                }
            }
        }

        /// <summary>
        /// テキストボックスの入力モードを自動的に本クラスが切り替えるかを取得または設定する
        /// </summary>
        [Browsable(true)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        [Description("テキストボックスの入力モードを自動的に切り替えるかを指定します")]
        public bool AutoInputMode
        {
            get { return this.m_autoInputMode; }
            set { this.m_autoInputMode = value; }
        }

        /// <summary>
        /// テキストボックスの現在選択されている領域を取得または設定する
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public TextSelectionArea SelectionArea
        {
            get { return this.m_selectionArea; }
            set
            { 
                this.m_selectionArea = value;
                updateProgramPoint(m_selectionArea.CurrentX, m_selectionArea.CurrentY);
                updateSelectArea();
                updateCursorPoint();
            }
        }

        /// <summary>
        /// テキストボックスの読み取り専用属性を取得または設定する
        /// </summary>
        [Browsable(true)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        [Description("テキストボックスが読み取り専用かを指定します")]
        public bool ReadOnly
        {
            get { return this.m_readOnly; }
            set { this.m_readOnly = value; }
        }

        /// <summary>
        /// 現在カーソル位置の論理行番号を取得または設定する
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public int CurrentLineIndex
        {
            get { return m_lstPAL[this.m_selectionArea.CurrentY].Y; }
            set
            {
                if (m_lstPAL[this.m_selectionArea.CurrentY].Y == value)
                    return;

                // 対象の物理indexを検索して設定。対象がない場合は何もせずに処理を抜ける
                // MEMO:対象が見つからない場合に例外を出す場合はインターフェースのコメントも修正すること
                int targetY = value;
                TextSelectionArea area = new TextSelectionArea(
                    0,m_selectionArea.CurrentY,0,0,0,0);
                for (int i = targetY; i < m_lstPAL.Count; i++)
                {
                    if (m_lstPAL[i].Y == targetY)
                    {
                        area.CurrentY = i;
                        area.CurrentX = getProgramX(area.CurrentY);
                        break;
                    }
                }
                // カーソル更新もかけるためプロパティ経由で更新
                SelectionArea = area;
            }
        }

        /// <summary>
        /// コントロールの内容を現在のキャレットの位置までスクロールします。 
        /// </summary>
        public void ScrollToCaret()
        {
            if (m_selectionArea.CurrentY < this.m_vScrollInfo.nPos)
            {
                // 上スクロール
                this.m_vScrollInfo.nPos = m_selectionArea.CurrentY;
            }
            else if ((m_selectionArea.CurrentY - this.m_vScrollInfo.nPos) * m_lineHeight >= m_maxHeight)
            {
                // 下スクロール
                this.m_vScrollInfo.nPos += ((m_selectionArea.CurrentY - this.m_vScrollInfo.nPos + 1) * m_lineHeight - m_maxHeight) / m_lineHeight;
            }
            ScrollMagician.SetScrollInfo(this.Handle, ScrollMagician.SB_VERT, ref this.m_vScrollInfo, true);
            updateCursorPoint();
        }
        #endregion
    }
}