/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.text.FieldPosition;
import java.text.Format;
import java.text.ParsePosition;
import java.util.HashMap;
import java.util.Map;

import woolpack.utils.UtilsConstants;

/**
 * 有限個の値をフォーマットする変換器。
 * @author nakamura
 *
 */
public class LimitedValueFormat extends Format {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private final Map<String,? extends Object> parseMap;
	private final Object defaultParsedValue;
	private final Map<? extends Object,String> formatMap;
	private final String defaultFormattedValue;
	
	/**
	 * コピーコンストラクタ。
	 * {@link #clone()}から呼び出される(called)。
	 * @param format コピー元。
	 */
	protected LimitedValueFormat(final LimitedValueFormat format){
		this.parseMap = format.parseMap;
		this.defaultParsedValue = format.defaultParsedValue;
		this.formatMap = format.formatMap;
		this.defaultFormattedValue = format.defaultFormattedValue;
	}
	
	/**
	 * コンストラクタ。
	 * @param parseMap {@link #parseObject(String, ParsePosition)}で使用する{@link Map}。null を指定した場合は formatMap を逆にしたものを使用する。
	 * @param defaultParsedValue parseMap に変更元が定義されていない場合の変更先。null を指定した場合は変更しない。
	 * @param formatMap {@link #format(Object, StringBuffer, FieldPosition)}で使用する{@link Map}。null を指定した場合は parseMap を逆にしたものを使用する。
	 * @param defaultFormattedValue formatMap に変更元が定義されていない場合の変更先。null を指定した場合はを変更しない。
	 * @throws NullPointerException parseMap と formatMap がともに null の場合。
	 * @throws IllegalArgumentException parseMap または formatMap の値が重複している場合。
	 */
	public LimitedValueFormat(
			final Map<? extends Object,String> formatMap, 
			final String defaultFormattedValue, 
			final Map<String,? extends Object> parseMap, 
			final Object defaultParsedValue){
		this.parseMap = (parseMap == null)?inverse(formatMap):new HashMap<String,Object>(parseMap);
		this.defaultParsedValue = defaultParsedValue;
		this.formatMap = (formatMap == null)?inverse(parseMap):new HashMap<Object,String>(formatMap);
		this.defaultFormattedValue = defaultFormattedValue;
	}
	
	/**
	 * コンストラクタ。
	 * @param parseMap {@link #parseObject(String, ParsePosition)}で使用する{@link Map}。null を指定した場合は formatMap を逆にしたものを使用する。
	 * @param formatMap {@link #format(Object, StringBuffer, FieldPosition)}で使用する{@link Map}。null を指定した場合は parseMap を逆にしたものを使用する。
	 * @throws NullPointerException parseMap と formatMap がともに null の場合。
	 * @throws IllegalArgumentException parseMap が null の場合は formatMap の値が重複している場合。formatMap が null の場合は parseMap の値が重複している場合。
	 */
	public LimitedValueFormat( 
			final Map<? extends Object,String> formatMap,
			final Map<String,? extends Object> parseMap){
		this(formatMap, null, parseMap, null);
	}
	
	/**
	 * コンストラクタ。
	 * @param formatMap {@link #format(Object, StringBuffer, FieldPosition)}で使用する{@link Map}。
	 * @param defaultFormattedValue formatMap に変更元が定義されていない場合の変更先。null を指定した場合はを変更しない。
	 * @throws NullPointerException formatMap が null の場合。
	 * @throws IllegalArgumentException formatMap の値が重複している場合。
	 */
	public LimitedValueFormat(final Map<? extends Object,String> formatMap, final String defaultFormattedValue){
		this(formatMap, defaultFormattedValue, null, null);
	}
	
	/**
	 * コンストラクタ。
	 * @param formatMap {@link #format(Object, StringBuffer, FieldPosition)}で使用する{@link Map}。
	 * @throws NullPointerException formatMap が null の場合。
	 * @throws IllegalArgumentException formatMap の値が重複している場合。
	 */
	public LimitedValueFormat(final Map<? extends Object,String> formatMap){
		this(formatMap, null, null, null);
	}
	
	private static <K,V> Map<V,K> inverse(final Map<K,V> before){
		final Map<V,K> after = new HashMap<V,K>();
		{
			final Map<V,K> m = UtilsConstants.unoverwritableMap(after);
			for(K key:before.keySet()){
				m.put((V)before.get(key), (K)key);
			}
		}
		return after;
	}
	
	@Override
	public StringBuffer format(final Object obj, final StringBuffer toAppendTo,
			final FieldPosition pos) {
		final int start = toAppendTo.length();
		String o = formatMap.get(obj);
		if(o == null){
			o = defaultFormattedValue;
		}
		if(o != null){
			toAppendTo.append(o);
		}else{
			toAppendTo.append(obj);
		}
		pos.setBeginIndex(start);
		pos.setEndIndex(toAppendTo.length());
		return toAppendTo;
	}

	@Override
	public Object parseObject(final String source, final ParsePosition pos) {
		for(final String key:parseMap.keySet()){
			if(source.startsWith(key, pos.getIndex())){
				pos.setIndex(pos.getIndex()+key.length());
				return parseMap.get(key);
			}
		}
		if(defaultParsedValue != null){
			pos.setIndex(source.length());
			return defaultParsedValue;
		}
		pos.setErrorIndex(pos.getIndex());
		return null;
	}

	@Override public Object clone(){
		return new LimitedValueFormat(this);
	}
	
	/**
	 * JavaScriptのコンストラクタ表現を返す。
	 */
	@Override public String toString(){
		final StringBuilder sb = new StringBuilder();
		sb.append("new LimitedValueFormat({");
		boolean flag = false;
		for(final Object key:formatMap.keySet()){
			if(flag){
				sb.append(",");
			}
			flag = true;
			sb.append("\"");
			sb.append(key);
			sb.append("\":\"");
			sb.append(formatMap.get(key));
			sb.append("\"");
		}
		sb.append("},\"");
		sb.append(defaultFormattedValue);
		sb.append("\",{");
		flag = false;
		for(final String key:parseMap.keySet()){
			if(flag){
				sb.append(",");
			}
			flag = true;
			sb.append("\"");
			sb.append(key);
			sb.append("\":\"");
			sb.append(parseMap.get(key));
			sb.append("\"");
		}
		sb.append("},\"");
		sb.append(defaultParsedValue);
		sb.append("\")");
		return sb.toString();
	}
}
