/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import javax.xml.namespace.QName;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * {@link XPath}のラッパークラス。
 * JPSE6b84 にスレッドセーフでない/リエントラントでないと明記されているため、
 * JPSE5 もスレッドセーフでないと判断してスレッドセーフなラッパーを作成した。
 * @author nakamura
 *
 */
public class XPathWrapper {
	private final String s;
	
	/**
	 * コンストラクタ。
	 * @param s XPath の文字列表現。
	 * @throws RuntimeException ({@link XPathExpressionException})XPath のコンパイルに失敗した場合。
	 */
	public XPathWrapper(final String s){
		this.s = s;
		compile(s);
	}
	
	private static XPathExpression compile(final String s){
		final XPath base = XPathFactory.newInstance().newXPath();
		try {
			return base.compile(s);
		} catch (final XPathExpressionException e) {
			throw new RuntimeException(e);
		}
	}
	
	private static Object evaluate(final String s, QName kind, final Object node){
		final XPathExpression expression = compile(s);
		try {
			return expression.evaluate(node, kind);
		} catch (final XPathExpressionException e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * 検索する。
	 * @param node 検索の基点。
	 * @return 検索結果。
	 * @throws NullPointerException 引数が null の場合。
	 * @throws RuntimeException ({@link XPathExpressionException})引数の評価に失敗した場合。
	 */
	public NodeList evaluateList(final Object node){
		return (NodeList)evaluate(s, XPathConstants.NODESET, node);
	}
	
	/**
	 * 検索する。
	 * @param node 検索の基点。
	 * @return 検索結果。
	 * @throws NullPointerException 引数が null の場合。
	 * @throws RuntimeException ({@link XPathExpressionException})引数の評価に失敗した場合。
	 */
	public Node evaluateOne(final Object node){
		return (Node)evaluate(s, XPathConstants.NODE, node);
	}
}
