/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.adapter;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.TestCase;
import ognl.OgnlException;
import woolpack.TestBean;
import woolpack.TestUtils;
import woolpack.adapter.OGE;
import woolpack.el.ELTargetRuntimeException;

public class OGETest extends TestCase {

	public void testConstructor(){
		try{
			new OGE(null);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new OGE("!");
			fail();
		}catch(final RuntimeException expected){
			assertTrue(expected.getCause() instanceof OgnlException);
		}
		try{
			new OGE("");
			fail();
		}catch(final RuntimeException expected){
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}
	
	public void testSetValueGetValue(){
		final Map<String, String> map = new HashMap<String, String>();
		map.put("param0", "value0");
		assertEquals("value0", new OGE("param0").getValue(map));
		assertEquals("value0", new OGE("param0").getValue(map, String.class));
		
		new OGE("param1").setValue(map, "value1");
		assertEquals("value1", new OGE("param1").getValue(map));

		assertNull(new OGE("param2").getValue(map));

		new OGE("param3").setValue(map, null);
		assertNull(new OGE("param3").getValue(map));
	}
	
	public void testString(){
		final Map<String, Object> map = new HashMap<String, Object>();
		map.put("param0", new Integer(7));
		assertEquals(new Integer(7), new OGE("param0").getValue(map));
		assertTrue(new OGE("param0").getValue(map, String.class) instanceof String);
		assertEquals("7", new OGE("param0").getValue(map, String.class));
		
		new OGE("param1").setValue(map, new Integer(11));
		assertEquals(new Integer(11), new OGE("param1").getValue(map));
		assertEquals("11", new OGE("param1").getValue(map, String.class));
	}
	
	public void testCollection(){
		assertTrue(TestUtils.equals(Arrays.asList(13, 17), new OGE("#this").getValue(new int[]{13,17}, List.class)));
	}
	
	public void testGetValueException(){
		final Map map = new HashMap();
		try{
			new OGE("param0.param1").getValue(map);
			fail();
		}catch(final RuntimeException expected){
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}
	
	public void testGetStringException(){
		final Map map = new HashMap();
		try{
			new OGE("param0.param1").getValue(map, String.class);
			fail();
		}catch(final RuntimeException expected){
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}
	
	public void testSetValueException(){
		final Map map = new HashMap();
		try{
			new OGE("param0.param1").setValue(map, "a");
			fail();
		}catch(final RuntimeException expected){
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}
	
	public void testInvoke(){
		final Map<String, Object> root = new HashMap<String, Object>();
		root.put("component0", new TestBean());
		{
			final Object result = new OGE("component0.executeVoid()").getValue(root);
			assertNull(result);
		}
		{
			Object result = new OGE("component0.executeInt()").getValue(root);
			assertEquals(new Integer(1), result);

			result = new OGE("component0.executeInt()").getValue(root);
			assertEquals(new Integer(2), result);
		}
		try{
			new OGE("component0.executeException()").getValue(root);
			fail();
		}catch(final ELTargetRuntimeException expected){
			assertTrue(expected.getCause() instanceof IndexOutOfBoundsException);
		}
	}
	
	public void testInvokeMethodNotFound(){
		final Map<String, Object> root = new HashMap<String, Object>();
		root.put("component0", new TestBean());
		try{
			new OGE("component0.executeNotFound()").getValue(root);
			fail();
		}catch(final RuntimeException expected){
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}
	
	public void testInvokeException(){
		final Map<String, Object> root = new HashMap<String, Object>();
		root.put("component0", new TestBean());
		try{
			new OGE("component0.notFoundProperty").getValue(root);
			fail();
		}catch(final RuntimeException expected){
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}
	
	public void testEquals(){
		assertTrue(new OGE("a").equals(new OGE("a")));
		assertFalse(new OGE("a").equals(new OGE("b")));
		assertFalse(new OGE("a").equals(new Object()));
	}
	
	public void testHashCode(){
		assertEquals("a".hashCode(), new OGE("a").hashCode());
		assertEquals("b".hashCode(), new OGE("b").hashCode());
	}
	
	public void testToString(){
		assertEquals("a", new OGE("a").toString());
	}
}
