/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.Map;

import junit.framework.TestCase;

import org.w3c.dom.Element;

import woolpack.TestUtils;
import woolpack.el.EL;
import woolpack.el.FixEL;
import woolpack.el.ToELTargetExceptionEL;

public class ExecTest extends TestCase {

	public void testConstructor(){
		try{
			new Exec("a", null, DomConstants.NULL, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new Exec("a", EL.NULL, null, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new Exec("a", EL.NULL, DomConstants.NULL, null);
			fail();
		}catch(final NullPointerException expected){
		}
		new Exec(null, EL.NULL, DomConstants.NULL, DomConstants.NULL);
	}
	
	public void testTrue(){
		final Count counter0 = new Count(DomConstants.NULL);
		final Count counter1 = new Count(DomConstants.NULL);
		final DomExpression expression = new Exec(new FixEL(Boolean.TRUE), counter0, counter1);
		final DomContext context = new DomContext();
		expression.interpret(context);
		assertEquals(1, counter0.getCount());
		assertEquals(0, counter1.getCount());
		assertTrue(context == counter0.getContextList().getLast());
	}
	
	public void testFalse(){
		final Count counter0 = new Count(DomConstants.NULL);
		final Count counter1 = new Count(DomConstants.NULL);
		final DomExpression expression = new Exec(new FixEL(Boolean.FALSE), counter0, counter1);
		final DomContext context = new DomContext();
		expression.interpret(context);
		assertEquals(0, counter0.getCount());
		assertEquals(1, counter1.getCount());
		assertTrue(context == counter1.getContextList().getLast());
	}
	
	public void testOther(){
		final Count counter0 = new Count(DomConstants.NULL);
		final Count counter1 = new Count(DomConstants.NULL);
		final DomExpression expression = new Exec(new FixEL(new Integer(0)), counter0, counter1);
		final DomContext context = new DomContext();
		expression.interpret(context);
		assertEquals(0, counter0.getCount());
		assertEquals(0, counter1.getCount());
	}
	
	public void testResource(){
		final Count counter0 = new Count(DomConstants.NULL);
		final Count counter1 = new Count(DomConstants.NULL);
		final DomExpression expression = new Exec("id0", new ToELTargetExceptionEL(){
			@Override public Object execute(Object root, Object value) throws Exception {
				// new OGE("context.id == resource")
				return ((DomContext)((Map)root).get("context")).getId().equals(((Map)root).get("resource"));
			}
		}, counter0, counter1);
		final DomContext context = new DomContext();
		
		context.setId("id0");
		expression.interpret(context);
		assertEquals(1, counter0.getCount());
		assertEquals(0, counter1.getCount());
		
		context.setId("id1");
		expression.interpret(context);
		assertEquals(1, counter0.getCount());
		assertEquals(1, counter1.getCount());
	}
	
	public void testTipsToContext(){
		final DomExpression expression = new Exec(new ToELTargetExceptionEL(){
			@Override public Object execute(Object root, Object value) throws Exception {
				// new OGE("context.local.key0 = \"value0\"")
				return ((DomContext)((Map)root).get("context")).getLocal().put("key0", "value0");
			}
		});
		final DomContext context = new DomContext();
		expression.interpret(context);
		assertEquals("value0", context.getLocal().get("key0"));
	}
	
	public void testTipsIdEquation(){
		final Count counter0 = new Count(DomConstants.NULL);
		final Count counter1 = new Count(DomConstants.NULL);
		final DomExpression expression = new Exec(new ToELTargetExceptionEL(){
			@Override public Object execute(Object root, Object value) throws Exception {
				// new OGE("context.id == \"id0\"")
				return "id0".equals(((DomContext)((Map)root).get("context")).getId());
			}
		}, counter0, counter1);
		{
			final DomContext context = new DomContext();
			context.setId("id0");
			expression.interpret(context);
			assertEquals(1, counter0.getCount());
			assertEquals(0, counter1.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			context.setId("id1");
			expression.interpret(context);
			assertEquals(1, counter0.getCount());
			assertEquals(1, counter1.getCount());
			assertTrue(context == counter1.getContextList().getLast());
		}
	}
	
	public void testTipsAttrValueEquation0(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<INPUT type=\"text\" name=\"name0\" value=\"value0\" />"),
			new XPathAttrValue("name", "name0", 
					new Exec(new ToELTargetExceptionEL(){
						@Override public Object execute(Object root, Object value) throws Exception {
							// new OGE("context.node.getAttribute(\"value\") == \"value0\"")
							return "value0".equals(((Element)((DomContext)((Map)root).get("context")).getNode()).getAttribute("value"));
						}
					}, 
					new UpdateAttrValue("value", new FixEL("trueValue")), 
					new UpdateAttrValue("value", new FixEL("falseValue")))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<INPUT type=\"text\" name=\"name0\" value=\"trueValue\" />"));
	}
	
	public void testTipsAttrValueEquation1(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<INPUT type=\"text\" name=\"name0\" value=\"value0\" />"),
			new XPathAttrValue("name", "name0", 
					new Exec(new ToELTargetExceptionEL(){
						@Override public Object execute(Object root, Object value) throws Exception {
							// new OGE("context.node.getAttribute(\"value\") == \"value1\"")
							return "value1".equals(((Element)((DomContext)((Map)root).get("context")).getNode()).getAttribute("value"));
						}
					}, 
					new UpdateAttrValue("value", new FixEL("trueValue")), 
					new UpdateAttrValue("value", new FixEL("falseValue")))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<INPUT type=\"text\" name=\"name0\" value=\"falseValue\" />"));
	}
	
	public void testIdRegExp(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<INPUT name=\"name0\" value=\"value0\" />"),
				new XPathAttrValue("name", "name0", new Exec(
						new ToELTargetExceptionEL(){
							@Override public Object execute(Object root, Object value) throws Exception {
								// new OGE("context.id.matches(\"id[0-9]\")")
								return ((DomContext)((Map)root).get("context")).getId().matches("id[0-9]");
							}
						},  
						new UpdateAttrValue("value", new FixEL("trueValue")), 
						new UpdateAttrValue("value", new FixEL("falseValue"))
				))
		);
		{
			final DomContext context = new DomContext();
			context.setId("id0");
			expression.interpret(context);

			assertTrue(TestUtils.equalsForm(context, "<INPUT name=\"name0\" value=\"trueValue\" />"));
		}
		{
			final DomContext context = new DomContext();
			context.setId("idA");
			expression.interpret(context);

			assertTrue(TestUtils.equalsForm(context, "<INPUT name=\"name0\" value=\"falseValue\" />"));
		}
	}
}
