/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.HashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import junit.framework.TestCase;
import woolpack.AllTests;
import woolpack.TestConcurrent;
import woolpack.utils.UtilsConstants;

public class TrySemaphoreSessionTest extends TestCase {
	private final ExecutorService threadPool = Executors.newFixedThreadPool(AllTests.MAX_THREAD);

	public void testConstructor(){
		try{
			new TrySemaphoreSession(null, 1, DomConstants.NULL, DomConstants.NULL);
		}catch(final NullPointerException expected){
		}
		try{
			new TrySemaphoreSession("", 1, DomConstants.NULL, DomConstants.NULL);
		}catch(final StringIndexOutOfBoundsException expected){
		}
		try{
			new TrySemaphoreSession("key0", 1, null, DomConstants.NULL);
		}catch(final NullPointerException expected){
		}
		try{
			new TrySemaphoreSession("key0", 1, DomConstants.NULL, null);
		}catch(final NullPointerException expected){
		}
	}
	
	private void scenario(
			final int testCount, 
			final int threshold) throws InterruptedException{
		final Object lock = new Object();
		final ConcurrentMap<String,Object> session0 = UtilsConstants.concurrentMap(new HashMap<String,Object>(), new Object());
		final ConcurrentMap<String,Object> session1 = UtilsConstants.concurrentMap(new HashMap<String,Object>(), new Object());
		final TestConcurrent concurrentTrue0 = new TestConcurrent("true0", lock, 1, AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentFalse0 = new TestConcurrent("false0", lock, 1, AllTests.MAX_THREAD - AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentTrue1 = new TestConcurrent("true1", lock, 1, AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentFalse1 = new TestConcurrent("false1", lock, 1, AllTests.MAX_THREAD - AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		
		final DomExpression expression = new TrySemaphoreSession(AllTests.MAX_CONCURRENT,
				new DomExpression(){
					public void interpret(final DomContext context) {
						if(context.getSession() == session0){
							concurrentTrue0.execute();
						}else{
							concurrentTrue1.execute();
						}
					}
				}, new DomExpression(){
					public void interpret(final DomContext context) {
						if(context.getSession() == session0){
							concurrentFalse0.execute();
						}else{
							concurrentFalse1.execute();
						}
					}
				});

		final CountDownLatch latch = new CountDownLatch(AllTests.MAX_THREAD);
		for(int i=0; i<AllTests.MAX_THREAD; i++){
			final int ii = i;
			threadPool.execute(new Runnable(){
				public void run() {
					try{
						concurrentTrue0.setThreadId(ii);
						concurrentFalse0.setThreadId(ii);
						concurrentTrue1.setThreadId(ii);
						concurrentFalse1.setThreadId(ii);
						for(int i=0; i<testCount; i++){
							Thread.yield();
							final DomContext context = new DomContext();
							Thread.yield();
							context.setSession((Math.random() < 0.5)?session0:session1);
							Thread.yield();
							try{
								expression.interpret(context);
							}catch(final RuntimeException e){
							}
							Thread.yield();
						}
					}finally{
						latch.countDown();
					}
				}
			});
		}

//		System.out.println("-- DoSemaphoreSessionTest start --");
//		concurrentTrue0.print();
//		concurrentFalse0.print();
//		concurrentTrue1.print();
//		concurrentFalse1.print();
//		System.out.println("-- DoSemaphoreSessionTest end --");
		assertTrue(concurrentTrue0.assertValid());
		assertTrue(concurrentFalse0.assertValid());
		assertTrue(concurrentTrue1.assertValid());
		assertTrue(concurrentFalse1.assertValid());
	}
	
	public void testNormal() throws InterruptedException{
		scenario(AllTests.TEST_COUNT, AllTests.TEST_THRESHOLD);
		for(int i=0 ;i<AllTests.TEST_COUNT; i++){
			scenario(10, 10);
		}
	}
}
