/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;

import javax.servlet.http.HttpSessionBindingListener;

import junit.framework.TestCase;
import woolpack.AllTests;
import woolpack.TestConcurrent;
import woolpack.dom.BranchById;
import woolpack.dom.Count;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Exec;
import woolpack.el.PathEL;
import woolpack.utils.DelegationMap;
import woolpack.utils.FixSwitch;
import woolpack.utils.SwitchBuilder;
import woolpack.utils.UtilsConstants;

public class TrySemaphoreTransactionTest extends TestCase {
	private final ExecutorService threadPool = Executors.newFixedThreadPool(AllTests.MAX_THREAD);

	public void testConstructor(){
		try{
			new TrySemaphoreTransaction(
					null, 
					new FixSwitch<String,Semaphore>(null), 
					new ArrayList<String>(), 
					new ArrayList<String>(), 
					DomConstants.NULL, 
					DomConstants.NULL, 
					DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new TrySemaphoreTransaction(
					"", 
					new FixSwitch<String,Semaphore>(null), 
					new ArrayList<String>(), 
					new ArrayList<String>(), 
					DomConstants.NULL, 
					DomConstants.NULL, 
					DomConstants.NULL);
			fail();
		}catch(final StringIndexOutOfBoundsException expected){
		}
		try{
			new TrySemaphoreTransaction(
					"key0", 
					null, 
					new ArrayList<String>(), 
					new ArrayList<String>(), 
					DomConstants.NULL, 
					DomConstants.NULL, 
					DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new TrySemaphoreTransaction(
					"key0", 
					new FixSwitch<String,Semaphore>(null), 
					null, 
					new ArrayList<String>(), 
					DomConstants.NULL, 
					DomConstants.NULL, 
					DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new TrySemaphoreTransaction(
					"key0", 
					new FixSwitch<String,Semaphore>(null), 
					new ArrayList<String>(), 
					null, 
					DomConstants.NULL, 
					DomConstants.NULL, 
					DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new TrySemaphoreTransaction(
					"key0", 
					new FixSwitch<String,Semaphore>(null), 
					new ArrayList<String>(), 
					new ArrayList<String>(), 
					null, 
					DomConstants.NULL, 
					DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new TrySemaphoreTransaction(
					"key0", 
					new FixSwitch<String,Semaphore>(null), 
					new ArrayList<String>(), 
					new ArrayList<String>(), 
					DomConstants.NULL, 
					null, 
					DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new TrySemaphoreTransaction(
					"key0", 
					new FixSwitch<String,Semaphore>(null), 
					new ArrayList<String>(), 
					new ArrayList<String>(), 
					DomConstants.NULL, 
					DomConstants.NULL, 
					null);
			fail();
		}catch(final NullPointerException expected){
		}
	}
	
	public void testDuplicate(){
		try{
			new TrySemaphoreTransaction(
					new SwitchBuilder<String,Semaphore>().put("ids0", new Semaphore(1)).put("ids1", new Semaphore(1)).get(), 
					Arrays.asList("idm0", "ids1"),
					Arrays.asList("ide0", "ide1"), 
					DomConstants.NULL, 
					DomConstants.NULL, 
					DomConstants.NULL);
			fail();
		}catch(final IllegalArgumentException expected){
			assertEquals("duplicate id: [ids1]", expected.getMessage());
		}
		try{
			new TrySemaphoreTransaction(
					new SwitchBuilder<String,Semaphore>().put("ids0", new Semaphore(1)).put("ids1", new Semaphore(1)).get(), 
					Arrays.asList("idm0", "idm1"),
					Arrays.asList("ide0", "ids1"), 
					DomConstants.NULL, 
					DomConstants.NULL, 
					DomConstants.NULL);
			fail();
		}catch(final IllegalArgumentException expected){
			assertEquals("duplicate id: [ids1]", expected.getMessage());
		}
		try{
			new TrySemaphoreTransaction(
					new SwitchBuilder<String,Semaphore>().put("ids0", new Semaphore(1)).put("ids1", new Semaphore(1)).get(), 
					Arrays.asList("idm0", "idm1"),
					Arrays.asList("ide0", "idm1"), 
					DomConstants.NULL, 
					DomConstants.NULL, 
					DomConstants.NULL);
			fail();
		}catch(final IllegalArgumentException expected){
			assertEquals("duplicate id: [idm1]", expected.getMessage());
		}
	}
	
	private ConcurrentMap<String,Object> generateSession(){
		final Map<String,Object> map = new HashMap<String,Object>();
		
//		final Map<string,Object map2> = 
//			new KeyIteratorMap<String,Object>(){
//			@Override
//			protected Iterator<String> getKeyIterator() {
//				final Iterator<String> e = map.keySet().iterator();
//				return new Iterator<String>(){
//					String key = null;
//					public void remove() {
//						final Object oldValue = map.remove(key);
//						if(oldValue instanceof HttpSessionBindingListener){
//							((HttpSessionBindingListener)oldValue).valueUnbound(null);
//						}
//					}
//					public boolean hasNext() {
//						return e.hasNext();
//					}
//					public String next() {
//						key = e.next();
//						return key;
//					}
//				};
//			}
//			@Override
//			protected Object getValue(final Object key) {
//				return map.get(key);
//			}
//			@Override public Object put(final String key, final Object value){
//				final Object oldValue = map.put(key, value);
//				if(oldValue instanceof HttpSessionBindingListener){
//					((HttpSessionBindingListener)oldValue).valueUnbound(null);
//				}
//				return oldValue;
//			}
//		};

		final Map<String,Object> map3 = new DelegationMap<String,Object>(map){
			@Override public Object remove(final Object key){
				final Object value = super.remove(key);
				if(value instanceof HttpSessionBindingListener){
					// 本来は remove 以外の更新系に対して実装する必要があるがテスト対象は remove 以外の更新系を扱わないため実装しない。
					((HttpSessionBindingListener)value).valueUnbound(null);
				}
				return value;
			}
		};
		
		return UtilsConstants.concurrentMap(map3, new Object());
	}
	
	private void check(final Count[] counts, final int[] expected){
		final List<Integer> l0 = new ArrayList<Integer>();
		final List<Integer> l1 = new ArrayList<Integer>();
		for(int i=0; i<counts.length; i++){
			l0.add(expected[i]);
			l1.add(counts[i].getCount());
		}
		assertEquals(l0, l1);
	}
	
	public void testSingleThread() throws InterruptedException{
		final Count[] counts = new Count[12];
		for(int i=0; i<counts.length; i++){
			counts[i] = new Count(new Exec(new PathEL("context.request.empty")));
		}
		final Semaphore semaphore = new Semaphore(1);
		final DomExpression expression = new TrySemaphoreTransaction(
				new SwitchBuilder<String,Semaphore>().put("id0", semaphore).get(), 
				Arrays.asList("id1"), 
				Arrays.asList("id2"), 
				new BranchById(new SwitchBuilder<String,DomExpression>()
						.put("id0", counts[0])
						.put("id1", counts[1])
						.put("id2", counts[2])
						.get(counts[3])),
				new BranchById(new SwitchBuilder<String,DomExpression>()
						.put("id0", counts[4])
						.put("id1", counts[5])
						.put("id2", counts[6])
						.get(counts[7])),
				new BranchById(new SwitchBuilder<String,DomExpression>()
						.put("id0", counts[8])
						.put("id1", counts[9])
						.put("id2", counts[10])
						.get(counts[11])));
		final DomContext context = new DomContext();
		context.setRequest(new HashMap<String,Object>());
		context.setSession(generateSession());
		
		context.setId("id1");
		expression.interpret(context);
		check(counts, new int[]{0,0,0,0, 0,1,0,0, 0,0,0,0});
		assertEquals(1, semaphore.availablePermits());
		
		context.setId("id2");
		expression.interpret(context);
		check(counts, new int[]{0,0,0,0, 0,1,1,0, 0,0,0,0});
		assertEquals(1, semaphore.availablePermits());
		
		context.setId("id0");
		expression.interpret(context);
		check(counts, new int[]{1,0,0,0, 0,1,1,0, 0,0,0,0});
		assertEquals(0, semaphore.availablePermits());
		
		context.setId("id0");
		expression.interpret(context);
		check(counts, new int[]{2,0,0,0, 0,1,1,0, 1,0,0,0});
		assertEquals(0, semaphore.availablePermits());
		
		context.setId("id1");
		expression.interpret(context);
		check(counts, new int[]{2,1,0,0, 0,1,1,0, 1,0,0,0});
		assertEquals(0, semaphore.availablePermits());
		
		context.setId("id0");
		expression.interpret(context);
		check(counts, new int[]{3,1,0,0, 0,1,1,0, 2,0,0,0});
		assertEquals(0, semaphore.availablePermits());
		
		context.setId("id1");
		expression.interpret(context);
		check(counts, new int[]{3,2,0,0, 0,1,1,0, 2,0,0,0});
		assertEquals(0, semaphore.availablePermits());
		
		context.setId("id2");
		expression.interpret(context);
		check(counts, new int[]{3,2,1,0, 0,1,1,0, 2,0,1,0});
		assertEquals(1, semaphore.availablePermits());
		
		context.setId("id0");
		expression.interpret(context);
		check(counts, new int[]{4,2,1,0, 0,1,1,0, 2,0,1,0});
		assertEquals(0, semaphore.availablePermits());
		
		context.setId("id2");
		expression.interpret(context);
		check(counts, new int[]{4,2,2,0, 0,1,1,0, 2,0,2,0});
		assertEquals(1, semaphore.availablePermits());
		
		context.setId("id2");
		expression.interpret(context);
		check(counts, new int[]{4,2,2,0, 0,1,2,0, 2,0,2,0});
		assertEquals(1, semaphore.availablePermits());
		
		context.setId("id1");
		expression.interpret(context);
		check(counts, new int[]{4,2,2,0, 0,2,2,0, 2,0,2,0});
		assertEquals(1, semaphore.availablePermits());
		
		semaphore.acquire();
		
		context.setId("id0");
		expression.interpret(context);
		check(counts, new int[]{4,2,2,0, 1,2,2,0, 2,0,2,0});
		assertEquals(0, semaphore.availablePermits());
		
		context.setId("id1");
		expression.interpret(context);
		check(counts, new int[]{4,2,2,0, 1,3,2,0, 2,0,2,0});
		assertEquals(0, semaphore.availablePermits());
		
		context.setId("id2");
		expression.interpret(context);
		check(counts, new int[]{4,2,2,0, 1,3,3,0, 2,0,2,0});
		assertEquals(0, semaphore.availablePermits());
		
		semaphore.release();
		
		context.setId("id0");
		expression.interpret(context);
		check(counts, new int[]{5,2,2,0, 1,3,3,0, 2,0,2,0});
		assertEquals(0, semaphore.availablePermits());

		context.setRequest(null);
		
		context.setId("id2");
		try{
			expression.interpret(context);
			fail();
		}catch(final RuntimeException expected){
		}
		check(counts, new int[]{5,2,3,0, 1,3,3,0, 2,0,3,0});
		assertEquals(1, semaphore.availablePermits());
		
		context.setRequest(new HashMap<String,Object>());
		
		context.setId("id0");
		expression.interpret(context);
		check(counts, new int[]{6,2,3,0, 1,3,3,0, 2,0,3,0});
		assertEquals(0, semaphore.availablePermits());

		context.setRequest(null);
		
		context.setId("id1");
		try{
			expression.interpret(context);
			fail();
		}catch(final RuntimeException expected){
		}
		check(counts, new int[]{6,3,3,0, 1,3,3,0, 2,1,3,0});
		assertEquals(1, semaphore.availablePermits());
		
		context.setId("id0");
		try{
			expression.interpret(context);
			fail();
		}catch(final RuntimeException expected){
		}
		check(counts, new int[]{7,3,3,0, 1,3,3,0, 3,1,3,0});
		assertEquals(1, semaphore.availablePermits());
		
		context.setRequest(new HashMap<String,Object>());
		
		context.setId("id0");
		expression.interpret(context);
		check(counts, new int[]{8,3,3,0, 1,3,3,0, 3,1,3,0});
		assertEquals(0, semaphore.availablePermits());
		
		context.setId("id3");
		expression.interpret(context);
		check(counts, new int[]{8,3,3,1, 1,3,3,0, 3,1,3,0});
		assertEquals(0, semaphore.availablePermits());
	}
	
	private void scenario(
			final int testCount, 
			final int threshold) throws InterruptedException{
		final Object lock = new Object();
		final TestConcurrent concurrentTrue0 = new TestConcurrent("true0", lock, 1, AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentFalse0 = new TestConcurrent("false0", lock, 1, AllTests.MAX_THREAD - AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentRelease0 = new TestConcurrent("release0", lock, 1, AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentTrue1 = new TestConcurrent("true1", lock, 1, AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentFalse1 = new TestConcurrent("false1", lock, 1, AllTests.MAX_THREAD, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentRelease1 = new TestConcurrent("release1", lock, 1, AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentTrue2 = new TestConcurrent("true2", lock, 1, AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentFalse2 = new TestConcurrent("false2", lock, 1, AllTests.MAX_THREAD, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentRelease2 = new TestConcurrent("release2", lock, 1, AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		
		final DomExpression expression = new TrySemaphoreTransaction(
				new SwitchBuilder<String,Semaphore>().put("id0", new Semaphore(AllTests.MAX_THREAD, true)).get(), 
				Arrays.asList("id1"), 
				Arrays.asList("id2"), 
				new DomExpression(){
					public void interpret(final DomContext context) {
						if("id0".equals(context.getId())){
							concurrentTrue0.execute();
						}else if("id1".equals(context.getId())){
							concurrentTrue1.execute();
						}else if("id2".equals(context.getId())){
							concurrentTrue2.execute();
						}
					}
				}, new DomExpression(){
					public void interpret(final DomContext context) {
						if("id0".equals(context.getId())){
							concurrentFalse0.execute();
						}else if("id1".equals(context.getId())){
							concurrentFalse1.execute();
						}else if("id2".equals(context.getId())){
							concurrentFalse2.execute();
						}
					}
				}, new DomExpression(){
					public void interpret(final DomContext context) {
						if("id0".equals(context.getId())){
							concurrentRelease0.execute();
						}else if("id1".equals(context.getId())){
							concurrentRelease1.execute();
						}else if("id2".equals(context.getId())){
							concurrentRelease2.execute();
						}
					}
				});

		final CountDownLatch latch = new CountDownLatch(AllTests.MAX_THREAD);
		for(int i=0; i<AllTests.MAX_THREAD; i++){
			final int ii = i;
			threadPool.execute(new Runnable(){
				public void run() {
					try{
						concurrentTrue0.setThreadId(ii);
						concurrentFalse0.setThreadId(ii);
						concurrentRelease0.setThreadId(ii);
						concurrentTrue1.setThreadId(ii);
						concurrentFalse1.setThreadId(ii);
						concurrentRelease1.setThreadId(ii);
						concurrentTrue2.setThreadId(ii);
						concurrentFalse2.setThreadId(ii);
						concurrentRelease2.setThreadId(ii);
						final ConcurrentMap<String,Object> session0 = generateSession();
						for(int i=0; i<testCount; i++){
							Thread.yield();
							final DomContext context = new DomContext();
							context.setSession(session0);
							context.setId("id0");
							try{
								expression.interpret(context);
							}catch(final RuntimeException e){
							}
							
							Thread.yield();
							context.setId("id1");
							try{
								expression.interpret(context);
							}catch(final RuntimeException e){
							}
							
							Thread.yield();
							context.setId("id2");
							try{
								expression.interpret(context);
							}catch(final RuntimeException e){
							}
							Thread.yield();
						}
					}finally{
						latch.countDown();
					}
				}
			});
		}

//		System.out.println("-- TrySemaphoreTransactionTest start --");
//		concurrentTrue0.print();
//		concurrentFalse0.print();
//		concurrentRelease0.print();
//		concurrentTrue1.print();
//		concurrentFalse1.print();
//		concurrentRelease1.print();
//		concurrentTrue2.print();
//		concurrentFalse2.print();
//		concurrentRelease2.print();
//		System.out.println("-- TrySemaphoreTransactionTest end --");
		assertTrue(concurrentTrue0.assertValid());
		assertTrue(concurrentFalse0.assertValid());
		assertTrue(concurrentRelease0.assertValid());
		assertTrue(concurrentTrue1.assertValid());
		assertTrue(concurrentFalse1.assertValid());
		assertTrue(concurrentRelease1.assertValid());
		assertTrue(concurrentTrue2.assertValid());
		assertTrue(concurrentFalse2.assertValid());
		assertTrue(concurrentRelease2.assertValid());
	}
	
	public void testNormal() throws InterruptedException{
		scenario(AllTests.TEST_COUNT, AllTests.TEST_THRESHOLD);
	}
}
